# -*- project-name: VASM -*-
package VASM::Catalog::Menu;

use strict;
use warnings;
use VASM::Tree;
use base qw/VASM::Catalog::Menu::Translation/;
use Carp;
use XML::Parser;
use XML::Writer;

our $VERSION = '1.09';

sub new {
  my ($self, $menuFH) = @_; 
  my $instance = { menu => VASM::Tree->new };
  bless $instance, $self;
  
  # Intern the contents of the IO::File $menuFH if given
  $instance->parse($menuFH) if defined $menuFH;

  return $instance;
}

sub store {
  my ($self, $attrs, @path) = (shift, pop, @_);

  # At the very least, there must be a @path with one element and associated
  # properties
  croak 'Path not given' unless @path; 
  croak 'Attributes hashref invalid' unless ref $attrs eq 'HASH';

  # Check contents thereof?

  # Notice that $attrs is copied when interned
  $self->{menu}->store(@path, { %$attrs });
  
  return;
}

sub retrieve {
  my ($self, @path) = @_;
  
  croak 'Path argument not given' unless @path;
  my $attrs = $self->{menu}->retrieve(@path);
  # Here again, we copy the hash so that it may not be modified
  return { %$attrs } if defined $attrs;

  return;
}

sub children {
  my ($self, @path) = @_;
  
  return $self->{menu}->children(@path);
}

sub parse {
  my ($self, $handle) = @_;

  croak 'Handle not given' unless defined $handle;
  # See parse in VASM::Message for lament
  my $parser = XML::Parser->new(Style => 'Tree', ErrorContext => 3);
  my $tree = $parser->parse($handle);

  # _parse is recursive, and it would make no sense to have multiple
  # XML::Parser instances running around
  $self->_parse($tree->[1]);
  
  return;
}

sub _parseFolder {
  my ($self, $tags, @path) = (shift, pop, @_);
  my $attrs = $tags->[0];

  # Label and icon attributes must be defined
  for my $attr (qw/label icon/) {
    croak "$attr attribute undefined in folder tag"
      unless length $attrs->{$attr};
  }

  # Store the folder
  $self->store(@path, $attrs->{label}, { icon => $attrs->{icon} });
  # Recurse
  $self->_parse($tags, @path, $attrs->{label});

  return;
}

sub _parseItem {
  my ($self, $tags, @path) = (shift, pop, @_);
  my $attrs = $tags->[0];
  
  # Certain attributes must be defined
  for my $attr (qw/label path icon/) {
    croak "$attr attribute undefined in item tag" 
      unless length $attrs->{$attr};
  }

  # Store the item
  $self->store(
    @path, $attrs->{label}, { path => $attrs->{path}, icon => $attrs->{icon} }
  );

  return;
}

sub _parse {
  # $tags is the chunk of parsed XML under examination, and @path is a
  # stack representing the current folder hierarchy
  my ($self, $tags, @path) = @_;

  for my $index (grep { $_ % 2 } (0..$#{ $tags })) {
    next if $tags->[$index] eq '0'; # Skip text elements

    # Check for an existing element at this path
    croak 'Two or more elements of the same name at this level'
      if defined $self->retrieve(@path, $tags->[$index + 1]->[0]->{label});

    if ($tags->[$index] eq 'folder') {
      $self->_parseFolder(@path, $tags->[$index + 1]);
    } elsif ($tags->[$index] eq 'item') {
      $self->_parseItem(@path, $tags->[$index + 1]);
    } else {
      croak 'Invalid tag in menu catalog';
    }
  }

  return;
}

sub write {
  my ($self, $menuFH) = @_;

  croak 'Handle argument not given' unless defined $menuFH;

  # Now create an XML::Writer instance and turn that puppy out
  my $xmlFH = XML::Writer->new(
    OUTPUT => $menuFH, ENCODING => 'utf-8', DATA_MODE => 1, DATA_INDENT => 2
  );

  # Open up the document
  $xmlFH->xmlDecl;
  $xmlFH->startTag('catalog');
  $xmlFH->comment('Generated by VASM');

  # Now let _write do its stuff
  $self->_write($xmlFH); 

  # End the document...
  $xmlFH->endTag('catalog'); $xmlFH->end;

  return;
}

sub _write {
  my ($self, $xmlFH, @path) = @_;

  # Do something here
  for my $child ($self->children(@path)) {
    my $attrs = $self->retrieve(@path, $child);
    # Produce an item tag if appropriate
    if (defined $attrs->{path}) {
      $xmlFH->emptyTag('item', label => $child, %{ $attrs });
    }
    # Otherwise, recurse if the child in question has children of its own, in
    # between producing the appropriate opening and closing tags
    elsif ($self->children(@path, $child)) {
      $xmlFH->startTag('folder', label => $child, icon => $attrs->{icon});
      $self->_write($xmlFH, @path, $child);
      $xmlFH->endTag('folder');
    }
    # Must be an empty folder! We will preserve it nevertheless
    else {
      $xmlFH->emptyTag('folder', label => $child, icon => $attrs->{icon});
    }
  }

  return;
}

1;

__END__

=head1 NAME

VASM::Catalog::Menu - general window manager menu definitions

=head1 SYNOPSIS

    use VASM::Catalog::Menu;
    
    my $instance = VASM::Catalog::Menu->new($ioHandle);
    
    # Store some elements in the menu
    $instance->store(qw/Editors/, { icon => "$ENV{PWD}/.pixmaps/kate.png" });
    $instance->store(qw/Editors Emacs/, 
                     { icon => "$ENV{PWD}/.pixmaps/emacs.png",
                       path => '/opt/emacs/bin/emacs' });
    $instance->store(qw/Editors LyX/, 
                     { icon => "$ENV{PWD}/.pixmaps/lyx.png", 
                       path => '/usr/bin/lyx' });
    # No one said this has to look pretty
    $instance->store(qw/Utilities/,
                     { icon => "$ENV{PWD}/.pixmaps/package_utilities.png" });
    $instance->store(qw/Utilities WPrefs/,
                     { icon => "$ENV{PWD}/.pixmaps/WPrefs.xpm", 
                       path => '/usr/X11/bin/WPrefs' });
    $instance->store(qw/Utilities Terminal/,
                     { icon => "$ENV{PWD}/.pixmaps/GNUterm.xpm", 
                       path => '/usr/X11/bin/terminal' });
    $instance->store(qw/Internet/,
                     { icon => "$ENV{PWD}/.pixmaps/browser.png" });
    $instance->store(qw/Internet Seamonkey/,
                     { icon => "$ENV{PWD}/.pixmaps/Mozilla.png", 
                       path => '/usr/bin/seamonkey' });
    $instance->store(qw/Internet Dillo/,
                     { icon => "$ENV{PWD}/.pixmaps/dillo.png", 
                       path => '/usr/X11/bin/dillo' });
    
    # Retrieve one such stored element
    my $attrs = $catalog->retrieve(qw/Editors Emacs/);
    # -> icon => "$ENV{PWD}/.pixmaps/emacs.png"
    #    path => "/opt/emacs/bin/emacs"
    
    # Now retrieve the top-level children
    my @children = $catalog->children;
    # -> qw/Editors Utilities Internet/
    
    # Introduce catalog definition from an IO::Handle
    $instance->parse($ioHandle);
    # This works, too:
    my $otherInstance = VASM::Catalog::Menu->new($ioHandle);
    
    # Write out the XML catalog
    $instance->write($ioHandle);

=head1 DESCRIPTION

VASM::Catalog::Menu furnishes an abstracted menu definition format, which
describes the majority of features present in the menu formats of most common
Unix window managers. The user may render these menu catalogs into practically
useful window manager menu files by means of the
VASM::Catalog::Menu::Translation superclass and the
VASM::Resource::Catalog::Menu infrastructure in concert. To automate
manipulation of menu catalogs through its object methods, VASM::Catalog::Menu
supports the parse and write methods, which, respectively, digest and produce
XML files representing the internal state of a menu catalog object.

=head1 METHODS

=over

=item new([ $handle ])

new constructs a new instance of the VASM::Catalog::Message class, and accepts
an optional argument, indicating an IO::Handle representing an XML file, whose
catalog definition you wish to introduce to the instance upon construction.
See L</FORMAT> below.

=item store(@path, $attrs)

The method store accepts a @path consisting of at least one element and a
hashref containing the attributes of a menu element, be it a program or
folder, and stores a copy of the hashref at the path so specified. Program
and folder elements hold in common the 'icon' attribute, an absolute path to a
representative icon file, and program elements exclusively feature the 'path'
attribute, the path to an executable and any options that follow it. store
does not check the contents of the $attrs argument, but any challenges to
convention will only end in disappointment. :) store returns nothing.

=item retrieve(@path)

This method accepts at least one index in the @path argument and returns a
copy of the hashref held at that @path.

=item children([ @path ])

children accepts an optional list of indices and returns the list of the
children nodes beneath it, retaining the order of internment. If no indices
are given, children returns the list of children at the root of the menu: that
is, the top-level menu items.

=item parse($ioHandle)

This method accepts a single argument, an IO::Handle whose referent XML file
to intern in the instance's menu definition. (See </FORMAT> below.) parse
returns nothing, but will croak if unsuccessful.

=item write($ioHandle)

The write method serves as the counterpart to the parse method, accepting an
IO::Handle with which to serialize the invocant instance. The output produced
takes human beings into account, and will follow reasonable indenting
guidelines.

=back

=head1 FORMAT

(Attribute values have been replaced with ellipses in the interest of
clarity.)

    <?xml version="1.0" encoding="utf-8"?>
    
    <catalog>
      <!-- Generated by VASM -->
      <folder label="Editors" icon="...">
        <item label="Emacs" icon="..." path="..." />
        <item label="LyX" icon="..." path="..." />
      </folder>
      <folder label="Utilities" icon="...">
        <item label="Terminal" icon="..." path="..." />
        <item label="WPrefs" icon="..." path="..." />
      </folder>
      <folder label="Internet" icon="...">
        <item label="Seamonkey" icon="..." path="..." />
        <item label="Dillo" icon="..." path="..." />
      </folder>
    </catalog>

The root node is customarily named <catalog>, although there is no penalty for
calling it by another name. Therein, the definition of the menu catalog itself
consists of free arrangements of 'folder' and 'item' elements with only one
stipulation: that no two elements of any kind at the same level bear the same
label, as defined by the label attribute. Furthermore, the 'label' and 'icon'
attributes are mandatory for both folder and item elements, and the 'path'
attribute is additionally required for all item elements.

=head1 AUTHORS

hanumizzle L<hanumizzle@gmail.com> wrote VASM::Catalog::Menu. Further
considerations by cintyram.

=cut
