# -*- project-name: VASM -*-
package VASM::Catalog::Action;

use strict;
use warnings;
use Carp;
use XML::Parser;
use XML::Writer;

our $VERSION = '1.12';

sub new {
  # Commands pushed and popped using arrayref
  my ($self, $file) = @_; my $instance = bless [], $self;
  # Intern the contents of the IO::File $file if given
  $instance->parse($file) if defined $file;
  
  return $instance;
}

# Add another command string to the end of the actions list
sub push {
  my ($self, $command) = @_;

  croak 'Command argument must be given!'
    unless defined $command;
  push @{ $self }, $command;

  return;
}

# You get the idea...
sub unshift {
  my ($self, $command) = @_;

  return unless defined $command;  
  unshift @{ $self }, $command;

  return;
}

# Just return the contents of the instance
sub dump {
  my ($self) = @_;
  
  return @{ $self };
}

sub clear {
  my ($self) = @_;
  
  $#{ $self } = -1;
  return;
}

sub exec {
  my ($self, @arguments) = @_; my $status;

  croak 'No programs interned in action catalog!' unless @$self;

  for my $program (@$self) {
    $status = system sprintf $program, @arguments;
    last if $status != -1;
  }

  return $status; # Will be -1 if none found
}

sub parse {
  my ($self, $file) = @_;
  # If this fails, XML::Parser will emit an error message and end the program.
  # Unfortunately, this message is always English. :( There is at least an
  # error context, which can be universally understood by the programmer.
  my $parser = XML::Parser->new(Style => 'Tree', ErrorContext => 3);
  my $tree = $parser->parse($file); my $tags = $tree->[1];
  
  # Iterate through tags under <catalog>
  for my $index (grep { $_ % 2 } (0..$#{ $tags })) {
    next if $tags->[$index] eq '0'; # Skip text elements

    # Otherwise, it /has to be/ a command string. Croak if it is misnamed,
    # does not contain a text string, or holds no content.
    croak 'Invalid tag in action catalog!'
      unless $tags->[$index] eq 'command';
    croak 'Non-textual or null command string!'
      unless $tags->[$index + 1]->[1] eq '0'
        and length $tags->[$index + 1]->[2];

    # Push the new member then
    $self->push($tags->[$index + 1]->[2]);
  }

  return;
}

# Write out the contents of the instance to a file, given an IO::File
sub write {
  my ($self, $file) = @_;

  croak 'File argument must be given!' unless defined $file;

  # Now create an XML::Writer instance and turn that puppy out
  my $xmlFile = XML::Writer->new(
    OUTPUT => $file, ENCODING => 'utf-8',
    DATA_MODE => 1, DATA_INDENT => 2
  );

  # Open up the document
  $xmlFile->xmlDecl;
  $xmlFile->startTag('catalog');
  $xmlFile->comment('Generated by VASM');

  # Emit the 'command' tags
  for my $command (@{ $self }) {
    $xmlFile->dataElement('command', $command);
  }

  $xmlFile->endTag('catalog'); $xmlFile->end; # End the document...

  return;
}
  
1;

__END__

=head1 NAME

VASM::Action - preference registration for diverse media types

=head1 SYNOPSIS

    use VASM::Action;
    
    # Instantiate VASM::Action object using an IO::File. This will call the
    # parse method of the new instance, passing that handle in.
    my $instance = VASM::Action->new($ioFile);
    
    # Add some new programs
    $instance->push('emacsclient %s');
    $instance->push('gvim %s');
    $instance->push('nedit %s');
    $instance->push('notepad %s');
    
    # Print out contents of catalog
    print for $instance->dump;

    # Execute the first command found in the catalog with an argument. This
    # will run 'emacsclient /home/hanumizzle/sithon-and-manola'
    $instance->exec('/home/hanumizzle/sithon-and-manola');
    
    # Write out the contents of the catalog via another IO::File
    $instance->write($otherIoFile);

=head1 DESCRIPTION

VASM::Action defines lists of programs associated with a given media type as
viewers and/or editors. In principle, VASM::Action can be used with any media
type represented in the arguments to a Unix command, as it is completely
agnostic towards them, being responsible only for executing a command in a
list. In practice, this means action catalogs are used in conjunction with
VASM::Resource::Action to pair vanilla file pathnames and URIs (Uniform
Resource Indicators, such as http://distro.ibiblio.org) to preferred programs.
Once prepared, the exec method of an instance will systematically run through
a list of programs and execute the first valid command with arguments, if so
desired. exec may fail to find a single usable command line; in this case, it
will return -1, propagating the return value of Perl's built-in system
function (as of Perl 5.x; this is liable to change in Perl 6).

=head1 METHODS

=over

=item new([ $handle ])

new constructs a new instance of the VASM::Message class, and accepts an
optional argument, indicating an IO::File representing an XML file, whose
catalog definition you wish to introduce to the instance upon construction.
See L</FORMAT> below.

=item push($command)

This corresponds to the built-in Perl operator of the same name, adding
definitions as one would expect. The method does not return anything, though.

=item unshift($command)

=item dump

This returns a list representing the catalog as it stands at the time the
method was executed. Exposing state like this isn't very OO, but necessary in
a number of cases.

...besides, you can: #always use: #Smalltalk.

=item parse($handle)

This method accepts a single argument, an IO::Handle whose referent XML file
to intern in the instance's message collection. (See </FORMAT> below.) parse
returns nothing, but will croak if unsuccessful.

=item write($handle)

The write method serves as the counterpart to the parse method, accepting an
IO::Handle with which to serialize the invocant instance. The output produced
takes human beings into account, and will follow reasonable indenting
guidelines. write returns nothing, but will croak if unsuccessful.

=item exec([ @arguments ])

The method exec attempts to execute each command string in the catalog, to
which it will apply the list of arguments passed in (if any), using sprintf
formatting directives. The method will return when an attempt is successful,
or when all possibilities in the catalog have been executed. The return value
is the same as the value returned by Perl's system function, which will
certainly be -1 if no command could be found.

=back

=head1 FORMAT

The XML format for VASM::Action is even simpler than that used in
VASM::Message catalogs, essentially representing a single flat list. The only
reason it /is/ XML is because the author likes consistency. Anyway:

    <?xml version="1.0" encoding="utf-8"?>

    <catalog>
      <command>emacsclient %s</command>
      <command>gvim %s</command>
    </catalog>

Each command tag pairing represents a command in an VASM::Action instance. The
entries will be interned in the order they are written.

=head1 AUTHORS

hanumizzle L<hanumizzle@gmail.com> wrote VASM::Action.

=cut
