#!/usr/bin/perl
# -*- project-name: VASM -*-

use strict;
use warnings;
use Test::More tests => 14;
use IO::Scalar;

# XML::Writer seems to think it is dealing with a tied handle?
BEGIN { package IO::Scalar; sub BINMODE {} } # UGLY HACK

### use VASM::Catalog::Message
BEGIN { use_ok('VASM::Catalog::Message') }

### Ability test
can_ok('VASM::Catalog::Message', 
       qw/new Store SetQuantity Render List Parse Write/);

### new
my $catalog = VASM::Catalog::Message->new;
# The catalog is a member of the appropriate class?
isa_ok($catalog, 'VASM::Catalog::Message');

### Store and Render
# Set up some quantities
$catalog->SetQuantity(sunya => '^-1$');
$catalog->SetQuantity(nullar => '^0$');
$catalog->SetQuantity(singular => '^1$');
$catalog->SetQuantity(plural => '');

# Store a few messages
for my $pair ( [ sunya => 'Foo!' ], [ nullar => 'Foo quxha!' ],
               [ singular => 'Foo quxhe!' ], [ plural => 'Foo quxho!' ] ) {
  $catalog->Store(id => 'Foo', quantity => $pair->[0], message => $pair->[1]);
}

for my $pair ( [ sunya => 'Bar!' ], [ nullar => 'Bar %sha!' ],
               [ singular => 'Bar %she!' ], [ plural => 'Bar %sho!' ] ) {
  $catalog->Store(id => 'Bar', quantity => $pair->[0], message => $pair->[1]);
}

# Will they render properly?
is($catalog->Render('Foo'), 'Foo!',
   'Foo catalog sunya message OK');
is($catalog->Render(id => 'Foo', quantity => 0), 
   'Foo quxha!',
   'Foo catalog nullar message OK');
is($catalog->Render(id => 'Foo', quantity => 1),
   'Foo quxhe!',
   'Foo catalog singular message OK');
is($catalog->Render(id => 'Foo', quantity => 2),
   'Foo quxho!',
   'Foo catalog plural message OK');

is($catalog->Render('Bar'), 
   'Bar!',
   'Bar catalog sunya message OK');
is($catalog->Render(id => 'Bar', quantity => 0, arguments => [ qw/qux/ ]), 
   'Bar quxha!',
   'Bar catalog nullar message OK');
is($catalog->Render(id => 'Bar', quantity => 1, arguments => [ qw/qux/ ]), 
   'Bar quxhe!',
   'Bar catalog singular message OK');
is($catalog->Render(id => 'Bar', quantity => 2, arguments => [ qw/qux/ ]), 
   'Bar quxho!',
   'Bar catalog plural message OK');

### List
# Are the catalog contents valid?
is_deeply([ $catalog->List ], [ qw/Foo Bar/ ], 
          'Catalog contents valid');

### Parse
# Instantiate an IO::Scalar containing an XML description of a catalog and
# parse it
my $xmlParseString = q{<?xml version="1.0" encoding="utf-8"?>

<catalog>
  <!-- Generated by VASM -->
  <quantity degree="sunya" match="(?-xism:^-1$)" />
  <quantity degree="nullar" match="(?-xism:^0$)" />
  <quantity degree="singular" match="(?-xism:^1$)" />
  <quantity degree="plural" match="(?-xism:)" />
  <message id="Foo">
    <sunya>Foo!</sunya>
    <nullar>Foo quxha!</nullar>
    <singular>Foo quxhe!</singular>
    <plural>Foo quxho!</plural>
  </message>
  <message id="Bar">
    <sunya>Bar!</sunya>
    <nullar>Bar %sha!</nullar>
    <singular>Bar %she!</singular>
    <plural>Bar %sho!</plural>
  </message>
</catalog>
};
my $xmlParseHandle = IO::Scalar->new(\$xmlParseString);
my $catalogParse = VASM::Catalog::Message->new($xmlParseHandle);

# Do $catalogParse and $catalog hold the same values?
is_deeply($catalogParse, $catalog, 'Parsed catalog valid');

### Write
my $xmlWriteString; my $xmlWriteHandle = IO::Scalar->new(\$xmlWriteString);
$catalog->Write($xmlWriteHandle);

# Is the written XML the same as the XML parse string? 
is($xmlWriteString, $xmlParseString, 'Written XML valid');
