# -*- project-name: VASM -*-
package VASM::Utility::Expect::Password;

use strict;
use warnings;
use constant +{
  OLD_PASSWORD => 0,
  NEW_PASSWORD => 1,
  CONFIRM_PASSWORD => 2,
  SUCCESS => 3
};
use base qw/Exporter/;
use Carp;
use Expect;

our $VERSION = '1.01';
our @EXPORT = qw/OLD_PASSWORD NEW_PASSWORD CONFIRM_PASSWORD SUCCESS/;

# Don't show the stdout of programs; it looks sloppy.
$Expect::Log_Stdout = 0;

sub new {
  my ($self, $username) = @_;

  # Assume $< for username -- if username is given, validate it; if that
  # fails, return
  if (defined $username) {
    my $invalid = validateUsername($username);
    print defined $invalid, "\n";
    return $invalid if defined $invalid;
  } else {
    $username = getpwuid $<;
  }

  # The key 'status' is the current stage in the transaction (e.g., waiting
  # for a new password), and 'username' is of course the user for whom we are
  # configuring
  my $instance = { status => OLD_PASSWORD, username => $username };
  bless $instance, $self;
  $instance->instantiateExpect;
  
  return $instance;
}

my @submitMethods = (
  # From OLD_PASSWORD to NEW_PASSWORD
  sub {
    my ($self, $password) = @_;

    # A minute seems to be a reasonable timeout to receive an answer
    $self->{expect}->expect(60, [ qr/^Old password: / ]);
    $self->{expect}->send("$password\n");

    # Old password matched?
    my @patterns = ([ qr/^New password: / ], [ qr/^Incorrect password/ ]);
    
    if ($self->{expect}->expect(60, @patterns) == 1) {
      # Advance to the next stage
      $self->{status}++;
    } else {
      # Unsuccessful; create a new Expect instance
      $self->instantiateExpect;
    }
    
    return;
  },
  # From NEW_PASSWORD to CONFIRM_PASSWORD
  sub {
    my ($self, $password) = @_;

    # Send the new password
    $self->{expect}->send("$password\n");

    # Expect either 'Bad password' or 'Re-enter new password'
    my @patterns = ([ qr/^Re-enter new password: / ],
                    [ qr/^The password for $self->{username} is unchanged/ ],
                    [ qr/^Bad password: / ]);

    # Advance to the next stage if successful
    my $patidx = $self->{expect}->expect(60, @patterns);

    if ($patidx == 1) {
      # Re-enter the new password -- successful
      $self->{status}++;
    } elsif ($patidx == 2) {
      # 'The password for foo is unchanged' -- unsuccessful
      # Generate a new Expect instance
      $self->instantiateExpect;
    }

    return;
  },
  # From CONFIRM_PASSWORD to SUCCESS
  sub {
    my ($self, $password) = @_;

    # Send the confirmation of the new password
    $self->{expect}->send("$password\n");

    # Expect either:
    # 'Password changed' (success)
    # 'The password for $self->{username} unchanged' (too many tries)
    # 'q{They don't match}' (an unsuccessful try)
    my @patterns = ([ qr/^Password changed/ ],
                    [ qr/^The password for $self->{username} unchanged/ ],
                    [ qr/^They don't match; try again/ ]);
    my $patidx = $self->{expect}->expect(60, @patterns);
    
    if ($patidx == 1) {
      # 'Password changed' -- successful
      $self->{status}++;
      # Wait for EOF and close the Expect instance
      $self->{expect}->expect;
      $self->{expect}->soft_close;
    } elsif ($patidx == 2) {
      # 'The password for foo unchanged' -- unsuccessful
      # Generate a new Expect instance
      $self->instantiateExpect;
    } elsif ($patidx == 3) {
      # q{They don't match; try again}
      # The process will now be awaiting the new password again
      $self->{status} = NEW_PASSWORD;
    }

    return;
  }
);

sub Submit {
  my ($self, $password) = @_;

  # There can be no advancement after completion!
  return SUCCESS if $self->Status == SUCCESS;
  
  # Call the submit method associated with the current status
  my $method = $submitMethods[$self->Status];
  my $success = $self->$method($password);

  # Return the current status
  return $self->Status;
}

sub Status {
  my ($self) = @_;
  
  return $self->{status};
}

sub Username {
  my ($self) = @_;
  
  return $self->{username};
}

sub instantiateExpect {
  my ($self) = @_;
  
  # Create a new Expect instance for verifying/establishing the password
  $self->{expect} = Expect->new; 
  # Disable echoing and translation
  $self->{expect}->raw_pty(1);

  # Initiate prompt for old password
  {
    # Localized strings from passwd will screw up my Expect patterns
    local $ENV{PWD} = 'C';
    $self->{expect}->spawn("passwd $self->{username}")
      or croak 'Egad! Cannot spawn passwd!';
  }

  # The instance will be waiting for the old password then
  $self->{status} = OLD_PASSWORD;

  return;
}

sub validateUsername {
  my ($username) = @_;
  
  # One needs to be root to change the password of a user other than oneself
  return 'not-root' unless getpwuid $< eq 'root';
  # The user needs to exist as well...
  return 'non-existent' unless getpwnam $username;

  return;
}

1;
