#!/usr/bin/perl
# -*- project-name: VASM -*-

use strict;
use warnings;
use Test::More tests => 9;
use IO::Scalar;
use File::Spec::Functions;
use VASM::Resource;

# Set up $ENV{XDG_CONFIG_HOME} and $ENV{XDG_DATA_HOME} to the working
# directory (the same directory as the test script) in '.config' and '.data'
$ENV{XDG_CONFIG_HOME} = catdir($ENV{PWD}, '.config');
$ENV{XDG_DATA_HOME} = catdir($ENV{PWD}, '.data');

# XML::Writer seems to think it is dealing with a tied handle?
BEGIN { package IO::Scalar; sub BINMODE {} } # UGLY HACK

### use VASM::Catalog::Menu (hence, the translation package) 
BEGIN { use_ok('VASM::Catalog::Menu') }

### Ability test
can_ok('VASM::Catalog::Menu::Translation',
       qw/TranslateWindowMaker/);

# Instantiate an IO::Scalar containing an XML description of a catalog and
# parse it
my $xmlParseString = qq{<?xml version="1.0" encoding="utf-8"?>

<catalog>
  <!-- Generated by VASM -->
  <folder label="Editors" icon="$ENV{PWD}/.pixmaps/kate.png">
    <item label="Emacs" icon="$ENV{PWD}/.pixmaps/emacs.png" path="/opt/emacs/bin/emacs" />
    <item label="LyX" icon="$ENV{PWD}/.pixmaps/lyx.png" path="/usr/bin/lyx" />
  </folder>
  <folder label="Utilities" icon="$ENV{PWD}/.pixmaps/package_utilities.png">
    <item label="WPrefs" icon="$ENV{PWD}/.pixmaps/WPrefs.xpm" path="/usr/X11/bin/WPrefs" />
    <item label="Terminal" icon="$ENV{PWD}/.pixmaps/GNUterm.xpm" path="/usr/X11/bin/terminal" />
  </folder>
  <folder label="Internet" icon="$ENV{PWD}/.pixmaps/browser.png">
    <item label="Seamonkey" icon="$ENV{PWD}/.pixmaps/Mozilla.png" path="/usr/bin/seamonkey" />
    <item label="Dillo" icon="$ENV{PWD}/.pixmaps/dillo.png" path="/usr/X11/bin/dillo" />
  </folder>
</catalog>
};
my $xmlParseHandle = IO::Scalar->new(\$xmlParseString);
my $catalog = VASM::Catalog::Menu->new($xmlParseHandle);

### TranslateFluxbox
my $fluxboxTranslateString;
my $fluxboxTranslateHandle = IO::Scalar->new(\$fluxboxTranslateString);
$catalog->TranslateFluxbox($fluxboxTranslateHandle);

# Is the written menu consistent with expectations?
is($fluxboxTranslateString, qq{# Generated by VASM

[submenu] (Vector Linux Menu)
  [submenu] (Editors)
    [exec] (Emacs) {/opt/emacs/bin/emacs} <$ENV{XDG_CONFIG_HOME}/vasm/Menu/Translation/Icon/Fluxbox/emacs.png.xpm>
    [exec] (LyX) {/usr/bin/lyx} <$ENV{XDG_CONFIG_HOME}/vasm/Menu/Translation/Icon/Fluxbox/lyx.png.xpm>
  [end]
  [submenu] (Utilities)
    [exec] (WPrefs) {/usr/X11/bin/WPrefs} <$ENV{XDG_CONFIG_HOME}/vasm/Menu/Translation/Icon/Fluxbox/WPrefs.xpm.xpm>
    [exec] (Terminal) {/usr/X11/bin/terminal} <$ENV{XDG_CONFIG_HOME}/vasm/Menu/Translation/Icon/Fluxbox/GNUterm.xpm.xpm>
  [end]
  [submenu] (Internet)
    [exec] (Seamonkey) {/usr/bin/seamonkey} <$ENV{XDG_CONFIG_HOME}/vasm/Menu/Translation/Icon/Fluxbox/Mozilla.png.xpm>
    [exec] (Dillo) {/usr/X11/bin/dillo} <$ENV{XDG_CONFIG_HOME}/vasm/Menu/Translation/Icon/Fluxbox/dillo.png.xpm>
  [end]
[end]
}, 'Written Fluxbox menu valid');

# Check for icon
ok(
  defined scalar findConfigResource(
    qw/Menu Translation Icon Fluxbox emacs.png.xpm/
  ),
  'Fluxbox translator produced icons'
);

### TranslateIceWM
my $iceWMTranslateString;
my $iceWMTranslateHandle = IO::Scalar->new(\$iceWMTranslateString);
$catalog->TranslateIceWM($iceWMTranslateHandle);

# Is the written menu consistent with expectations?
is($iceWMTranslateString, qq{# Generated by VASM

menu "Editors" "kate.png" {
  prog "Emacs" "$ENV{PWD}/.config/vasm/Menu/Translation/Icon/IceWM/emacs.png" /opt/emacs/bin/emacs
  prog "LyX" "$ENV{PWD}/.config/vasm/Menu/Translation/Icon/IceWM/lyx.png" /usr/bin/lyx
}
menu "Utilities" "package_utilities.png" {
  prog "WPrefs" "$ENV{PWD}/.config/vasm/Menu/Translation/Icon/IceWM/WPrefs.xpm" /usr/X11/bin/WPrefs
  prog "Terminal" "$ENV{PWD}/.config/vasm/Menu/Translation/Icon/IceWM/GNUterm.xpm" /usr/X11/bin/terminal
}
menu "Internet" "browser.png" {
  prog "Seamonkey" "$ENV{PWD}/.config/vasm/Menu/Translation/Icon/IceWM/Mozilla.png" /usr/bin/seamonkey
  prog "Dillo" "$ENV{PWD}/.config/vasm/Menu/Translation/Icon/IceWM/dillo.png" /usr/X11/bin/dillo
}
}, 'Written IceWM menu valid');

# Check for icons
ok(
  defined scalar findConfigResource(
    qw/Menu Translation Icon IceWM emacs.png/
  ), 
  'IceWM translator produced icons'
);

### TranslateWindowMaker
my $wmakerTranslateString;
my $wmakerTranslateHandle = IO::Scalar->new(\$wmakerTranslateString);
$catalog->TranslateWindowMaker($wmakerTranslateHandle);

# Is the written menu consistent with expectations?
is($wmakerTranslateString, q{// Generated by VASM

#include "wmmacros"

"Vector Linux Menu" MENU
  "Editors" MENU
    "Emacs" EXEC /opt/emacs/bin/emacs
    "LyX" EXEC /usr/bin/lyx
  "Editors" END
  "Utilities" MENU
    "WPrefs" EXEC /usr/X11/bin/WPrefs
    "Terminal" EXEC /usr/X11/bin/terminal
  "Utilities" END
  "Internet" MENU
    "Seamonkey" EXEC /usr/bin/seamonkey
    "Dillo" EXEC /usr/X11/bin/dillo
  "Internet" END
"Vector Linux Menu" END
}, 'Written WindowMaker menu valid');

### TranslateXFce4
my $xfce4TranslateString;
my $xfce4TranslateHandle = IO::Scalar->new(\$xfce4TranslateString);
$catalog->TranslateXFce4($xfce4TranslateHandle);

# Is the written menu consistent with expectations?
is($xfce4TranslateString, qq{<?xml version="1.0" encoding="utf-8"?>
<!DOCTYPE xfdesktop-menu>
<!-- Generated by VASM -->

<xfdesktop-menu>
  <menu name="Vector Linux Menu" icon="xfce4-backdrop">
    <title name="Vector Linux Menu" icon="xfce4-backdrop" />
    <separator />
    <menu name="Editors" icon="$ENV{XDG_CONFIG_HOME}/vasm/Menu/Translation/Icon/XFce4/kate.png">
      <app name="Emacs" cmd="/opt/emacs/bin/emacs" icon="$ENV{XDG_CONFIG_HOME}/vasm/Menu/Translation/Icon/XFce4/emacs.png" />
      <app name="LyX" cmd="/usr/bin/lyx" icon="$ENV{XDG_CONFIG_HOME}/vasm/Menu/Translation/Icon/XFce4/lyx.png" />
    </menu>
    <menu name="Utilities" icon="$ENV{XDG_CONFIG_HOME}/vasm/Menu/Translation/Icon/XFce4/package_utilities.png">
      <app name="WPrefs" cmd="/usr/X11/bin/WPrefs" icon="$ENV{XDG_CONFIG_HOME}/vasm/Menu/Translation/Icon/XFce4/WPrefs.xpm" />
      <app name="Terminal" cmd="/usr/X11/bin/terminal" icon="$ENV{XDG_CONFIG_HOME}/vasm/Menu/Translation/Icon/XFce4/GNUterm.xpm" />
    </menu>
    <menu name="Internet" icon="$ENV{XDG_CONFIG_HOME}/vasm/Menu/Translation/Icon/XFce4/browser.png">
      <app name="Seamonkey" cmd="/usr/bin/seamonkey" icon="$ENV{XDG_CONFIG_HOME}/vasm/Menu/Translation/Icon/XFce4/Mozilla.png" />
      <app name="Dillo" cmd="/usr/X11/bin/dillo" icon="$ENV{XDG_CONFIG_HOME}/vasm/Menu/Translation/Icon/XFce4/dillo.png" />
    </menu>
  </menu>
</xfdesktop-menu>
}, 'Written XFce4 menu valid');

# Check for icons
ok(
  defined scalar findConfigResource(
    qw/Menu Translation Icon XFce4 emacs.png/
  ),
 'XFce4 translator produced icons'
);

