#!/usr/bin/perl
# -*- project-name: VASM -*-

use strict;
use warnings;
use constant { FALSE => 0, TRUE => 1 };
use constant { DATA => 0, LABEL => 1 };
use constant { VALUE => 0, KEY => 1, KEY_REAL => 2 };
use constant { FOLDER => 0, ITEM => 1 };
use constant { PAD => 2 };
use VASM::Resource::Catalog::Message;
use VASM::Resource::Catalog::Menu;
use File::Spec::Functions qw/curdir rel2abs splitpath/;
use Gtk2 '-init';

# Message files
my $CommonMsg = findMessageCatalog(qw/Common Message/);
my $UIMsg = findMessageCatalog(qw/Menu UI Gtk2 Message/);
my $ErrorMsg = findErrorCatalog(qw/Menu UI Error/);

sub makeMenuTree {
  my ($menu) = @_;

  # Set up a tree store corresponding to the menu; the single Glib::String
  # column indicates a particular folder or item name in the menu object
  my $menuTree = Gtk2::TreeStore->new('Glib::Scalar', 'Glib::String');

  # Add a top level imaginary root
  my $root = $menuTree->append(undef);
  $menuTree->set($root, LABEL, $UIMsg->Render('Window Manager Menu'));
  $menuTree->set($root, DATA, {});
  
  # makeMenuTreeHeavy needs to be factored out, as it is a /recursive/
  # function that will fill in the tree store itself
  makeMenuTreeHeavy($menu, $menuTree, $root);

  return $menuTree;
}

sub makeMenuTreeHeavy {
  my ($menu, $menuTree, @iters) = @_;
  
  # Get the current 'stack' of folders
  my @path = map { $menuTree->get($_, LABEL) } @iters[1..$#iters];

  # Create more folders under it
  for my $child ($menu->Children(@path)) {
    # If an item (only items have the path attribute)
    if ($menu->Retrieve(@path, $child)->{path}) {
      # Advance the pointer at the current level
      my $iter = $menuTree->append($iters[-1] or undef);
      # Set the item name and properties at the current path
      $menuTree->set($iter, DATA, $menu->Retrieve(@path, $child));
      $menuTree->set($iter, LABEL, $child);
    }
    # If a folder (by exclusion)
    else {
      push @iters, $menuTree->append($iters[-1] or undef);
      # Set the folder name at the current path
      $menuTree->set($iters[-1], LABEL, $child);
      $menuTree->set($iters[-1], DATA, $menu->Retrieve(@path, $child));
      makeMenuTreeHeavy($menu, $menuTree, @iters); # Recurse
      pop @iters; # Move back up a level
    }
  }
}

sub makeMenuTreeView {
  my ($menu, $modified) = @_;

  my $menuTree = makeMenuTree($menu); # The actual data
  # If the tree is changed, set the modified flag to TRUE
  $menuTree->signal_connect('row-changed' => sub { $$modified = TRUE });
  # The rest of this is a 'cookbook' recipe to make a tree view correspond
  # one-for-one with the data in the tree store
  my $menuTreeView = Gtk2::TreeView->new_with_model($menuTree);
  $menuTreeView->set_headers_visible(FALSE);
  my $col = Gtk2::TreeViewColumn->new;
  my $renderer = Gtk2::CellRendererText->new;
  $col->pack_start($renderer, FALSE);
  $col->add_attribute($renderer, text => LABEL);
  $menuTreeView->append_column($col);

  # Attach a signal handler for double clicks on row
  $menuTreeView->signal_connect(
    'row-activated' => sub {
      my ($menuTree, $iter) = $_[0]->get_selection->get_selected;
      editMenuTreeElement($menuTree, $iter) if $menuTree->iter_parent($iter);
    });

  return $menuTreeView;
}

sub makeAttrList {
  my %attrs = @_; my $iter;
  my $attrList = Gtk2::ListStore->new(qw/Glib::String/ x 3);

  for my $attr (qw/label path icon/) {
    # If the folder or item has this attr...
    if (defined $attrs{$attr}) {
      $iter = $attrList->append;
      # Name of attribute, translated
      $attrList->set($iter, KEY, $CommonMsg->Render(ucfirst $attr));
      # Used for ID
      $attrList->set($iter, KEY_REAL, $attr);
      # Value
      $attrList->set($iter, VALUE, $attrs{$attr});
    }
  }

  return $attrList;
}

sub makeAttrListView {
  my ($menuTree, $menuTreeElement) = @_;

  my $attrList = makeAttrList(getMenuTreeElementAsHash($menuTree,
                                                       $menuTreeElement));
  my $attrListView = Gtk2::TreeView->new_with_model($attrList);
  $attrListView->grab_focus; # Using keyboard shortcuts becomes easier

  my @fields = qw/Attribute Value/;
  for my $field (KEY, VALUE) {
    my $col = Gtk2::TreeViewColumn->new;
    # Set 'Attribute' and 'Value' headers
    $col->set_title($CommonMsg->Render($fields[$field]));
    my $renderer = Gtk2::CellRendererText->new;
    # The 'value' column is editable
    if ($field eq VALUE) {
      $renderer->set(editable => TRUE);
      $renderer->signal_connect(edited => \&attrListCellEdited, $attrList);
    }
    $col->pack_start($renderer, FALSE);
    $col->add_attribute($renderer, text => $field);
    $attrListView->append_column($col);
  }

  return $attrListView;
}

sub getMenuTreeElementAsHash {
  my ($menuTree, $menuTreeElement) = @_;
  my %attrs;
  
  $attrs{label} = $menuTree->get($menuTreeElement, LABEL);
  my $menuAttrs = $menuTree->get($menuTreeElement, DATA);
  # Add attributes from the DATA column of the row if defined
  for my $attr (qw/path icon/) {
    $attrs{$attr} = $menuAttrs->{$attr} if defined $menuAttrs->{$attr};
  }
  
  return %attrs;
}

sub getAttrListAsHash {
  my ($attrList) = @_; my %attrs;
  my $iter = $attrList->get_iter_first;
  
  # Iterate over the list store and grab pairs of KEY_REAL => VALUE, the
  # non-translated field name and its value
  do {
    $attrs{$attrList->get($iter, KEY_REAL)} = $attrList->get($iter, VALUE);
  } while ($iter = $attrList->iter_next($iter));

  return %attrs;
}

sub attrListFull {
  my ($attrList) = @_;
  my %attrs = getAttrListAsHash($attrList);
  
  # Return TRUE if the number of elements in $attrList with a length of 1 or
  # more is equal to the number of keys
  return TRUE if (grep { length } values %attrs) == keys %attrs;
  return FALSE;
}

sub attrListUnique {
  my ($menuTree, $menuTreeElement, $attrList) = @_;
  my %attrs = getAttrListAsHash($attrList);
  # The first row at the same level as $menuTreeElement
  my $iter = $menuTree->iter_children($menuTree->iter_parent($menuTreeElement));

  do {{
    # Ignore comparisons between the element being edited and itself
    next if $menuTree->get_string_from_iter($menuTreeElement) eq
            $menuTree->get_string_from_iter($iter);
    my %otherAttrHash = getMenuTreeElementAsHash($menuTree, $iter);
    # Return false if the label for the edited menu element is not unique
    return FALSE if $attrs{label} eq $otherAttrHash{label};
  }} while ($iter = $menuTree->iter_next($iter));
  
  # Otherwise, it falls through and returns true
  return TRUE;
}

sub attrListPathValid {
  my ($attrList) = @_;
  my %attrList = getAttrListAsHash($attrList);
  
  # Return TRUE if the path attribute of %attrList passes -x...if even
  # necessary
  return TRUE if not defined $attrList{path} or -x $attrList{path};
  return FALSE;
}

sub attrListIconValid {
  my ($attrList) = @_;
  my %attrList = getAttrListAsHash($attrList);
  
  # Return TRUE if the icon attribute of %attrList passes -r
  return TRUE if -r $attrList{icon};
  return FALSE;
}

sub setMenuTreeElement {
  my ($menuTree, $menuTreeElement, $attrList) = @_;
  my %attrList = getAttrListAsHash($attrList);

  # Set the label first
  $menuTree->set($menuTreeElement, LABEL, $attrList{label});
  # Now set all that other stuff
  delete $attrList{label};
  $menuTree->set($menuTreeElement, DATA, \%attrList);
}

sub menuTreeAddElementHeavy {
  my ($menuTree, $parent) = @_;

  # Dialog
  my $dialog = Gtk2::Dialog->new_with_buttons(
    $UIMsg->Render('Select Type'),
    undef, [],
    'gtk-ok' => 'ok',
    'gtk-cancel' => 'cancel');
  $dialog->set_position('center');

  # Radio buttons
  # Speaking of radio: http://wdiy.org -- do give it a look!
  my $selected = 'item';

  # Folder
  my $radio = Gtk2::RadioButton->new(undef, $CommonMsg->Render('Folder'));
  $radio->signal_connect(toggled => sub { $selected = 'folder' });
  $dialog->vbox->pack_start($radio, TRUE, TRUE, PAD);
  # Item (default)
  $radio = Gtk2::RadioButton->new($radio, $CommonMsg->Render('Item'));
  $radio->signal_connect(toggled => sub { $selected = 'item' });
  $radio->set_active(TRUE);
  $dialog->vbox->pack_start($radio, TRUE, TRUE, PAD);

  # Expander with detailed description
  my $shortDescription = Gtk2::Expander->new(
    $UIMsg->Render('Select Type Short Description'));
  my $longDescription = Gtk2::Label->new(
    $UIMsg->Render('Select Type Long Description'));
  $longDescription->set_line_wrap(TRUE);
  $shortDescription->add($longDescription);
  $dialog->vbox->pack_start($shortDescription, FALSE, FALSE, PAD);

  $dialog->show_all;
  my $response = $dialog->run;
  $dialog->destroy;
  
  return unless $response eq 'ok';

  # Otherwise, call a function according to the selected radio button
  my $newMenuElement = $menuTree->append($parent);
  # Fill in some blank 'default' values
  $menuTree->set($newMenuElement, LABEL, '');
  if ($selected eq 'folder') { 
    $menuTree->set($newMenuElement, DATA, { icon => '' }); 
  } elsif ($selected eq 'item') {
    $menuTree->set($newMenuElement, DATA, { path => '', icon => '' });
  }
  
  # Remove the new iter unless the user didn't decide to cancel
  $menuTree->remove($newMenuElement)
    unless editMenuTreeElement($menuTree, $newMenuElement);
  
  return;
}

sub menuTreeElementAdd {
  my ($menuTree, $parent) = $_[1]->get_selection->get_selected;

  # Check for item
  if ($menuTree->get($parent, DATA)->{path}) {
    my $error = Gtk2::MessageDialog->new(
      undef, [], qw/error ok/,
      $ErrorMsg->Render('Attempted Addition to Item'));
    $error->set_position('center'); $error->run; $error->destroy;
  } else { # If it is a folder...
    menuTreeElementAddHeavy($menuTree, $parent);
  }

  return;
}

sub menuTreeElementRemove {
  my $view = $_[1];
  my $menuTree = $view->get_model;
  my $iter = $view->get_selection->get_selected;

  # Don't delete r00t!!!
  unless ($menuTree->iter_parent($iter)) {
    my $error = Gtk2::MessageDialog->new(
      undef, [], qw/error ok/,
      $ErrorMsg->Render('Attempted Removal of Root'));
    $error->set_position('center'); $error->run; $error->destroy;
    return;
  }

  # Confirm the deletion, if the iter has children
  if ($menuTree->iter_children($iter)) {
    my $confirmation = Gtk2::MessageDialog->new(
      undef, [], qw/question yes-no/,
      $UIMsg->Render('Confirm Deletion'));
    $confirmation->set_position('center');

    # Get a reply
    $confirmation->show_all;
    my $response = $confirmation->run;
    $confirmation->destroy;
    
    return unless $response eq 'yes';
  }
  
  $menuTree->remove($iter);
  return;
}

sub menuTreeElementBack {
  my $view = $_[1];
  my $menuTree = $view->get_model;
  my $iter = $view->get_selection->get_selected;
  my $prev = eval {
    my $path = $menuTree->get_path($iter);
    return $menuTree->get_iter($path) if $path->prev;
  };
  $menuTree->swap($prev, $iter) if $prev;
  
  return;
}

sub menuTreeElementForward {
  my $view = $_[1];
  my $menuTree = $view->get_model;
  my $iter = $view->get_selection->get_selected;
  my $next = eval {
    my $path = $menuTree->get_path($iter); $path->next;
    return $menuTree->get_iter($path);
  };
  $menuTree->swap($iter, $next) if $next;
  
  return;
}

sub menuTreeElementUp {
  my $view = $_[1];
  my $menuTree = $view->get_model;
  my $iter = $view->get_selection->get_selected;
  # Get parent and make sure it exists
  my $parent = $menuTree->iter_parent($iter);
  unless (defined $parent) {
    my $error = Gtk2::MessageDialog->new(
      undef, [], qw/error ok/,
      $ErrorMsg->Render('Attempted Movement of Root Upward'));
    $error->set_position('center'); $error->run; $error->destroy;
    return;
  }
  # Get **grand**parent and make sure it exists, too -- parent and grandparent
  # are minimum needed to move element up
  my $grandparent = $menuTree->iter_parent($parent);
  unless (defined $grandparent) {
    my $error = Gtk2::MessageDialog->new(
      undef, [], qw/error ok/,
      $ErrorMsg->Render('Attempted Movement of Top-Level Element Upward'));
    $error->set_position('center'); $error->run; $error->destroy;
    return;
  }
  
  # Copy the values to the new iter and delete the existing one
  my $newIter = $menuTree->append($grandparent);
  for my $index (LABEL, DATA) {
    $menuTree->set($newIter, $index, $menuTree->get($iter, $index));
  }
  $menuTree->remove($iter);
  
  return;
}

sub menuTreeElementDownDialog {
  my (@folders) = @_; my (@radioButtons, $selected);

  # Create a group of radio buttons with corresponding labels
  for my $folder (@folders) {
    my $radioButton = Gtk2::RadioButton->new_with_label(
      @radioButtons ? $radioButtons[0]->get_group : undef,
      $folder->[0]);
    unless (@radioButtons) {
      $selected = $folder->[1];
      $radioButton->set_active(TRUE);
    }
    $radioButton->signal_connect(toggled => sub { $selected = $folder->[1] });
    push @radioButtons, $radioButton;
  }

  # Run a dialog with these radio buttons
  my $dialog = Gtk2::Dialog->new_with_buttons(
    $UIMsg->Render('Choose Folder'),
    undef, [],
    'gtk-ok' => 'ok',
    'gtk-cancel' => 'cancel');
  $dialog->set_position('center');

  # Add the radio buttons
  for my $radioButton (@radioButtons) {
    $dialog->vbox->pack_start($radioButton, TRUE, TRUE, PAD);
  }
  # Self-documentation
  $dialog->vbox->pack_start(
    Gtk2::Label->new($UIMsg->Render('Choose Folder Description')),
    FALSE, FALSE, PAD
  );

  $dialog->show_all;
  my $response = $dialog->run;
  $dialog->destroy;
  
  return $selected if $response eq 'ok';
  return;
}

sub menuTreeElementDown {
  my $view = $_[1];
  my $menuTree = $view->get_model;
  my $iter = $view->get_selection->get_selected;
  my $parent = $menuTree->iter_parent($iter);
  
  unless ($parent) { # Can't move r00t down
    my $error = Gtk2::MessageDialog->new(
      undef, [], qw/error ok/,
      $ErrorMsg->Render('Attempted Movement of Root Downward'));
    $error->set_position('center'); $error->run; $error->destroy;
    return;
  }
  my @folders; # alist of labels and iters for folders at the current level
  for my $index (0..$menuTree->iter_n_children($parent) - 1) {
    my $child = $menuTree->iter_nth_child($parent, $index);
    my $attrs = $menuTree->get($child, DATA);

    # If there is no path attribute, then it must be a folder
    unless (defined $attrs->{path} or
            $menuTree->get_string_from_iter($child) eq 
            $menuTree->get_string_from_iter($iter)) {
      push @folders, [ $menuTree->get($child, LABEL), $child ];
    }
  }

  return unless @folders; # Can't work if there be no candiate folders!

  # Many are called, few are chosen -- select a new parent for the item
  my $newParent = menuTreeElementDownDialog(@folders);
  return unless $newParent;
  
  # If needed, copy the values to the new iter and delete the existing one
  my $newIter = $menuTree->append($newParent);
  for my $index (LABEL, DATA) {
    $menuTree->set($newIter, $index, $menuTree->get($iter, $index));
  }
  $menuTree->remove($iter);

  return;
}

sub attrListCellEdited {
  my ($path, $text, $attrList) = @_[1..3];

  # Just pass the text through
  $attrList->set($attrList->get_iter_from_string($path), VALUE, $text);

  return;
}

sub updateIconPreview {
  my ($fileChooser, $preview) = @_;
  
  my $file = $fileChooser->get_preview_filename;
  # For some reason, this callback gets spurious bullshit from time to time
  if (defined $file and -f $file) {
    my $pixbuf = Gtk2::Gdk::Pixbuf->new_from_file_at_size($file, 64, 64);
    $preview->set_from_pixbuf($pixbuf);
    $fileChooser->set_preview_widget_active(defined $pixbuf ? TRUE : FALSE);
  } else {
    $fileChooser->set_preview_widget_active(FALSE);
  }

  return;
}

sub identifyMenuTreeRow {
  my ($menuTree, $iter) = @_; my @path;
  
  # Walk from branch to trunk, unshifting the 'real' column at each point. The
  # result will define the path in the tree store
  do {
    unshift @path, $menuTree->get($iter, LABEL)
      if $menuTree->iter_parent($iter);
    # The top-level 'Window Manager Menu' entry does not count
  } while ($iter = $menuTree->iter_parent($iter));

  return @path;
}

sub generateMenu {
  my ($menuTree) = @_;
  my $menu = VASM::Catalog::Menu->new;

  # Intern each element of the menu description from $menuTree
  $menuTree->foreach(
    sub {
      my ($menuTree, $iter) = @_[0,2];

      # Textual description of the path
      my @path = identifyMenuTreeRow($menuTree, $iter);
      # Handle top-level special case
      return FALSE unless @path;
      # Element attributes
      my %attrs = getMenuTreeElementAsHash($menuTree, $iter); 
      delete $attrs{label};
      # Store the resulting data
      $menu->Store(@path, \%attrs);

      return FALSE;
    });

  return $menu;
}

BEGIN {
  my $wd = rel2abs(curdir);
  
  sub findPathOrIcon {
    my $view = $_[1];
    my ($attrList, $iter) = $view->get_selection->get_selected;

    # The label attribute has no 'Find' option!!
    return if $attrList->get($iter, KEY_REAL) eq 'label';

    my $fileChooser = Gtk2::FileChooserDialog->new(
      $UIMsg->Render('Select File'), undef,
      'open',
      'gtk-ok' => 'ok',
      'gtk-cancel' => 'cancel');
    $fileChooser->set_current_folder($wd);
    $fileChooser->set_position('center');

    if ($attrList->get($iter, KEY_REAL) eq 'icon') {
      # Set filter on file chooser as well -- only image formats
      my $filter = Gtk2::FileFilter->new; $filter->add_pixbuf_formats;
      $fileChooser->set_filter($filter);
      # Icons will have preview
      my $preview = Gtk2::Image->new;
      $fileChooser->set_preview_widget($preview);
      $fileChooser->signal_connect('update-preview' => \&updateIconPreview,
                                   $preview);
    }

    $fileChooser->show_all;
    my $response = $fileChooser->run;
    if ($response eq 'ok') {
      my $file = $fileChooser->get_filename;
      $wd = (splitpath($file))[1]; # Store new current directory
      $attrList->set($iter, VALUE, $file); # Set element in store
    }

    $fileChooser->destroy;

    return;
  }
}

sub editMenuTreeElement {
  my ($menuTree, $menuTreeElement) = @_;

  # Dialog
  my $dialog = Gtk2::Dialog->new_with_buttons(
    $UIMsg->Render('Edit Properties'),
    undef, [],
    'gtk-ok' => 'ok',
    'gtk-cancel' => 'cancel');
  $dialog->set_position('center');

  # Attributes list
  my $attrListView = makeAttrListView($menuTree, $menuTreeElement);
  $dialog->vbox->pack_start($attrListView, TRUE, TRUE, PAD);

  # Operations (actually, only one)
  my $operationHButtonBox = Gtk2::HButtonBox->new;
  $operationHButtonBox->set_layout('spread');
  my @operations = ( [ 'gtk-find', \&findPathOrIcon ] );
  # Add them, well...it...one by one
  for my $operation (@operations) {
    my $button = Gtk2::Button->new_from_stock($operation->[0]);
    $button->signal_connect(clicked => $operation->[1], $attrListView);
    $operationHButtonBox->pack_start($button, FALSE, FALSE, PAD);
  }
  $dialog->vbox->pack_start($operationHButtonBox, FALSE, FALSE, PAD);

  # Self-documentation
  my $shortDescription = Gtk2::Expander->new(
    $UIMsg->Render('Edit Properties Short Description'));
  my $longDescription = Gtk2::Label->new(
    $UIMsg->Render('Edit Properties Long Description'));
  $longDescription->set_line_wrap(TRUE);
  $shortDescription->add($longDescription);
  $dialog->vbox->pack_start($shortDescription, FALSE, FALSE, PAD);

  $dialog->show_all;

  DIALOG: {
    my $response = $dialog->run; # Do it Jah!
    if ($response eq 'ok') {
      # Error checks
      my @checks = (
        [ 'Missing Fields' => 
            sub { attrListFull($attrListView->get_model) } ],
        [ 'Element Not Unique' => 
            sub { attrListUnique($menuTree, $menuTreeElement, 
                                 $attrListView->get_model) } ],
        [ 'Path Invalid' => # Must pass -x test
            sub { attrListPathValid($attrListView->get_model) } ],
        [ 'Icon Invalid' => # Must pass -r test
            sub { attrListIconValid($attrListView->get_model) } ]);

      # Run error checks
      for my $check (@checks) {
        unless (&{ $check->[1] }) { # $check->[1] is the subref
          # Report error
          my $error = Gtk2::MessageDialog->new(
            undef, [], qw/error ok/,
            $ErrorMsg->Render($check->[0])); # $check->[0] is the message key
          $error->set_position('center'); $error->run; $error->destroy;
          redo DIALOG; # Do over
        }
      }

      # If we made it here, that means we get to store the attributes loaded
      # in the editing window
      setMenuTreeElement($menuTree, $menuTreeElement, $attrListView->get_model);
    }

    $dialog->destroy;
    return ($response eq 'ok') ? 1 : 0;
  }
}

sub mainMenu {
  my $dialog = Gtk2::Dialog->new_with_buttons(
    $UIMsg->Render('Title'),
    undef, [],
    'gtk-apply' => 'apply',
    'gtk-ok' => 'ok',
    'gtk-close' => 'close');
  $dialog->set_position('center');
  $dialog->set_default_size(400, 300);

  ## Action area
  # TreeView of menu -- but, first, get the menu definition
  my $menu = findMenu; my $modified = FALSE;
  $menu = VASM::Catalog::Menu->new unless defined $menu;
  my $menuTreeView = makeMenuTreeView($menu, \$modified);
  
  # Pack the tree view into a scrolled window. The addition of scrollbars
  # will be done automatically in either direction
  my $scrolledView = Gtk2::ScrolledWindow->new;
  $scrolledView->set_policy(qw/automatic/ x 2);
  $scrolledView->add($menuTreeView);
  $dialog->vbox->pack_start($scrolledView, TRUE, TRUE, PAD);
  
  # Operations
  my $operationHButtonBox = Gtk2::HButtonBox->new;
  $operationHButtonBox->set_layout('spread');
  my @operations = ( [ 'gtk-add' => \&menuAdd ],
                     [ 'gtk-remove' => \&menuTreeElementRemove ],
                     [ 'gtk-go-back' => \&menuTreeElementBack ],
                     [ 'gtk-go-forward' => \&menuTreeElementForward ],
                     [ 'gtk-go-up' => \&menuTreeElementUp ],
                     [ 'gtk-go-down' => \&menuTreeElementDown ] );
  # Add them, one by one
  for my $operation (@operations) {
    my $button = Gtk2::Button->new_from_stock($operation->[0]);
    $button->signal_connect(clicked => $operation->[1], $menuTreeView);
    $operationHButtonBox->pack_start($button, TRUE, TRUE, PAD);
  }
  $dialog->vbox->pack_start($operationHButtonBox, FALSE, FALSE, PAD);

  # Expander with detailed description
  my $shortDescription = Gtk2::Expander->new(
    $UIMsg->Render('Main Menu Short Description'));
  my $longDescription = Gtk2::Label->new(
    $UIMsg->Render('Main Menu Long Description'));
  $longDescription->set_line_wrap(TRUE);
  $shortDescription->add($longDescription);
  $dialog->vbox->pack_start($shortDescription, FALSE, FALSE, PAD);

  $dialog->show_all;

  DIALOG: {
    my $response = $dialog->run;
    if ($response eq 'apply') {
      $menu = generateMenu($menuTreeView->get_model);
      writeMenu($menu);
      $modified = FALSE; # Having saved the menu, $modified is now FALSE
      redo DIALOG;
    } elsif ($response eq 'ok') {
      $menu = generateMenu($menuTreeView->get_model);
      writeMenu($menu);
    } elsif ($response eq 'close' and $modified) {
      # Confirm this
      my $confirm = Gtk2::MessageDialog->new(
        undef, [], qw/question yes-no/,
        $UIMsg->Render('Confirm Cancellation'));
      $confirm->set_position('center');
      my $response = $confirm->run; $confirm->destroy;
      # If answered no, stay here
      $response eq 'no' and redo DIALOG;
    }
  }

  $dialog->destroy;
  exit 0;
}

mainMenu();
