# -*- project-name: VASM -*-
package VASM::Resource::Catalog::Message;

use strict;
use warnings;
use base qw/Exporter/;
use VASM::Catalog::Message;
use VASM::Catalog::Error;
use VASM::Resource;
use Carp;

our @EXPORT = qw/findMessageCatalog findErrorCatalog writeMessageCatalog/;
our $VERSION = '1.06';

sub findCatalog {
  my %args = @_;

  my @path = @{ $args{Path} }; # Look under the resource path given
  my @locale = parseLocale(); # Finagle locale from %ENV

  # Now, try to find the first matching Message.xml in the hierarchy; the
  # search can go as far back as the base 'Message' directory itself:
  my $fh;
  LOOP: {
    do {
      $fh = (findDataResource(@path, @locale, 'Message.xml'))[0];
      last LOOP if defined $fh;
    } while (pop @locale);
  }

  # Return nothing unless a file was found, otherwise:
  croak 'Could not find a matching message catalog!' unless $fh;

  # Set the IO::File to UTF-8 for XML
  $fh->binmode(':utf8');

  # Switch catalog type depending on request
  my $catalog; 
  if ($args{Type} eq 'Message') {
    $catalog = VASM::Catalog::Message->new($fh);
  } elsif ($args{Type} eq 'Error') {
    $catalog = VASM::Catalog::Error->new($fh);
  }
  # And clean up
  $fh->close;

  return $catalog;
}

sub findMessageCatalog {
  return findCatalog(Path => [ @_ ], Type => 'Message');
}

sub findErrorCatalog {
  return findCatalog(Path => [ @_ ], Type => 'Error');
}

sub writeMessageCatalog {
  my ($instance, @path) = @_;
  my @locale = parseLocale(); # Finagle locale from %ENV
  
  # Generate a new IO::File for the catalog to be written. makeDataResource
  # (actually, makeResource) will croak if unsuccessful, so no error checking
  # is really necessary.
  my $fh = (makeDataResource(@path, @locale, 'Message.xml'))[0];
  $instance->Write($fh); # Now let the instance do its thing
  $fh->close; # ...close the filehandle and...done
  
  return;
}

# Because most of the normal divisions of locale support (e.g., LC_COLLATE,
# LC_NUMERIC) do not apply here, we will only honor a certain subset of
# environment variables, in this order: LC_ALL, LC_MESSAGES, LANG. (I believe
# this is the correct protocol; please correct me if I'm wrong.) Because VASM
# strictly uses UTF-8 for all messages, any codeset portion of the variable is
# ignored altogether.
sub parseLocale {
  # Loop through these variables in turn:
  for my $var (@ENV{qw/LC_ALL LC_MESSAGES LANG/}) {
    defined $var and return
      eval {
        my $locale = ($var =~ m!(\p{IsLower}{2}(?:_\p{IsUpper}{2})?)!)[0];
        # Split language and country (ex: en_GB vs. en_US)
        split /_/, $locale;
      }
  }

  # We buck the trend a little by using resource inheritance instead of the
  # default 'C' locale
  return;
}

1;

__END__

=head1 NAME

VASM::Resource::Catalog::Message - load message catalogs with the VASM
resources framework

=head1 SYNOPSIS

    use VASM::Resource::Catalog::Message;
    
    # Say LANG=de, and there is a catalog in
    # /usr/share/vasm/Some/Project/Message/de/Message.xml
    my $catalog = findMessageCatalog(qw/Some Project Message/);
    # See VASM::Catalog::Message
    $catalog->Render('Hello');
    # => Guten Tag!

    # Write out the message catalog back out -- writeMessageCatalog will
    # interpolate 'de' between the path given above and the catalog name
    writeMessageCatalog($catalog, qw/Some Project Message/);

=head1 DESCRIPTION

VASM::Resource::Catalog::Message enables the programmer to use
VASM::Catalog::Message i18n message catalogs with the resource hierarchy
defined by L<VASM::Resource>. It will search for a message catalog under the a
given resource path, in the fashion prescribed by the XDG Base Directory
standard, used by VASM::Resource. More specifically, VASM::Catalog::Message
will attempt to discern the current locale, and consequently, the appropriate
message file from environment variables that signify the locale (see
L<"VARIABLES"> below). If a message catalog for a given language/country pair
cannot be found, the search will default to a catalog for the language
independent of any national dialect, and, failing that, to a default catalog
most likely written in English.

=head1 FUNCTIONS

=over

=item findMessageCatalog(@path)

findMessageCatalog accepts a list describing a resource path and seeks a
message catalog therein, according to the rules described above. If
successful, it will return an instance of VASM::Catalog::Message after using
it to digest the catalog. Otherwise, it will return nothing.

=item findErrorCatalog(@path)

findErrorCatalog generally acts the same as findMessageCatalog, except that it
promotes the message catalog file to a VASM::Catalog::Error object.

=item writeMessageCatalog($catalog, @path)

This function writes out a message catalog instance to a given resource path,
and requires no counterpart for error messages, as the written format of a
message catalog does not distinguish between the two. Just as
findMessageCatalog and findErrorCatalog attempt to interpolate the current
locale name between the resource path and file, so will writeMessageCatalog
interpolate the current locale between the argument resource path and the file
name, Message.xml. In this way, reads will always correspond to writes,
assuming a consistent value for the locale.

=back

=head1 VARIABLES

These environment variables are searched in order for locale information, and
their values must conform to the POSIX convention consisting of a lowercase
two-letter language code optionally followed by an underscore and an uppercase
two-letter country code, e.g., hi_IN. Encoding information qualifiers such as
'.utf8' are ignored, because the VASM project strictly adheres to UTF-8:

=over

=item *

LC_ALL

=item *

LC_MESSAGES

=item *

LANG

=back

=head1 AUTHORS

hanumizzle L<mailto:hanumizzle@gmail.com> wrote
VASM::Resource::Catalog::Message. Further considerations by cintyram.

=cut
