# -*- project-name: VASM -*-
package VASM::Resource::Catalog::Menu;

use strict;
use warnings;
use base qw/Exporter/;
use VASM::Catalog::Menu;
use VASM::Resource;
use Carp;

our @EXPORT = qw/findMenu writeMenu translateMenu/;
our $VERSION = '1.06';

sub findMenu {
  my @locale = parseLocale(); # Finagle locale from %ENV

  # Now, try to find the first matching Message.xml in the hierarchy; the
  # search can go as far back as the base 'Message' directory itself:
  my $fh;
  LOOP: {
      do {
        $fh = (findConfigResource(qw/Menu File/, @locale, 'Menu.xml'))[0];
        last LOOP if defined $fh;
      } while (pop @locale);
    }

  # Return nothing unless a file was found, otherwise:
  # carp 'Could not find a matching menu file!' 
  return unless defined $fh;
  # Digest file, if possible, and return the new menu definition. If no file
  # could be found, just instantiate a blank menu object.
  $fh->binmode(':utf8'); # Set the IO::File to UTF-8 for XML
  my $menu = VASM::Catalog::Menu->new($fh);
  $fh->close;
  
  return $menu;
}

sub writeMenu {
  my ($instance) = @_;
  my @locale = parseLocale(); # Finagle locale from %ENV

  # Generate a new IO::File for the catalog to be written. makeConfigResource
  # (actually, makeResource) will croak if unsuccessful, so no error checking
  # is really necessary
  my $menuFile = (makeConfigResource(qw/Menu File/, @locale, 'Menu.xml'))[0];
  $instance->Write($menuFile); # Now let the instance do its thing
  $menuFile->close; # ...close the filehandle and...done
  
  return;
}

# If this accepts an instance of the menu class as a first argument, then why
# not make it OO? Because there is nothing fundamentally OO in nature about
# VASM::Resource. Otherwise, I would have
sub translateMenu {
  my ($instance, $wm) = @_;

  # Check if the appropriate method exists
  my $method = "Translate$wm";
  croak 'No translator exists for this window manager!'
    unless $instance->can($method);
  
  # Write out the menu
  # What about locales?
  my $menuFH = (makeConfigResource(qw/Menu Translation File/, $wm))[0];
  $instance->$method($menuFH);
  $menuFH->close; # ...and done

  return;
}

# Cut and pasted from VASM::Resource::Message:

# Because most of the normal divisions of locale support (e.g., LC_COLLATE,
# LC_NUMERIC) do not apply here, we will only honor a certain subset of
# environment variables, in this order: LC_ALL, LC_MESSAGES, LANG. (I believe
# this is the correct protocol; please correct me if I'm wrong.) Because VASM
# strictly uses UTF-8 for all messages, any codeset portion of the variable is
# ignored altogether.
sub parseLocale {
  # Loop through these variables in turn:
  for my $var (@ENV{qw/LC_ALL LC_MESSAGES LANG/}) {
    defined $var and return
      eval {
        my $locale = ($var =~ m!(\p{IsLower}{2}(?:_\p{IsUpper}{2})?)!)[0];
        # Split language and country (ex: en_GB vs. en_US)
        split /_/, $locale;
      }
  }

  # We buck the trend a little by using resource inheritance instead of the
  # default 'C' locale
  return;
}
  
1;

=head1 NAME

VASM::Resource::Catalog::Menu

=head1 SYNOPSIS

=head1 DESCRIPTION

=head1 FUNCTIONS

=head1 VARIABLES

=head1 AUTHORS

hanumizzle L<mailto:hanumizzle@gmail.com> wrote VASM::Resource::Catalog::Menu.

=cut
