# -*- project-name: VASM -*-
package VASM::Resource;

use strict;
use warnings;
use base qw/Exporter/;
use Carp;
use IO::File;
use File::Spec::Functions;
use File::Path;
use File::BaseDir qw/xdg_config_files
                     xdg_data_files
                     xdg_config_home
                     xdg_data_home/;
our @EXPORT = qw/findConfigResource
                 findDataResource
                 makeConfigResourcePath
                 makeDataResourcePath
                 makeConfigResource
                 makeDataResource/;
our $VERSION = '2.0';

sub findResource {
  my %args = @_;

  croak 'Type and path arguments must be given!'
    unless (grep { defined $args{$_} } qw/type path/) == 2;

  # Select a function according to the mode of retrieval:
  my $fileName;
  if ($args{type} eq 'data') {
    $fileName = (xdg_data_files(catfile('vasm', @{ $args{path} })))[0];
  } elsif ($args{type} eq 'config') {
    $fileName = (xdg_config_files(catfile('vasm', @{ $args{path} })))[0];
  }

  if (defined $fileName) { # If successful
    if (wantarray) {
      # Return a corresponding IO::File (usually the case)
      my $fh = IO::File->new($fileName);
      croak $fileName, ": $!" unless defined $fh;
      # If called in list context, this returns the IO::File object and
      # filename. Otherwise, it returns the $fh alone
      return ($fh, $fileName);
    } else { # Otherwise, just return the filename
      return $fileName;
    }
  }

  return; # Fall-through
}

sub findConfigResource {
  croak 'Path must be given!' unless @_;
  return findResource(type => 'config', path => \@_);
}

sub findDataResource {
  croak 'Path must be given!' unless @_;
  return findResource(type => 'data', path => \@_);
}

sub makeResourcePath {
  my %args = @_;

  croak 'All arguments must be given!'
    unless (grep { defined $args{$_} } qw/type path/) == 2;

  # Select a function according to mode of internment:
  my $path; # The path being created
  if ($args{type} eq 'data') {
    $path = catfile(xdg_data_home(), 'vasm', @{ $args{path} });
  } elsif ($args{type} eq 'config') {
    $path = catfile(xdg_config_home(), 'vasm', @{ $args{path} });
  }

  # Make sure all the required directories exist for a given resource; if not,
  # attempt to create them with mode 0700 (as per the XDG BaseDir spec). If
  # this fails, mkpath will die. I think we should allow it.
  mkpath($path, undef, 0700) unless -d $path;

  return $path;
}

sub makeDataResourcePath {
  croak 'Path must be given!' unless @_;
  return makeResourcePath(type => 'data', path => \@_);
}

sub makeConfigResourcePath {
  croak 'Path must be given!' unless @_;
  return makeResourcePath(type => 'config', path => \@_);
}

# Returns a filehandle with the UTF-8 IO layer, representing the file at the
# given resource path. Of course, this file is always in either the config or
# data user-specific resource tree.
sub makeResource {
  my %args = @_;

  # Check required args
  croak 'All arguments must be given!'
    unless (grep { defined $args{$_} } qw/type path/) == 2;
  
  my $base = pop @{ $args{path} }; # Remove the file from $args{path}
  my $path = makeResourcePath(type => $args{type}, # Find the needed path
                              path => $args{path});
  
  # Open the file with the UTF-8 IO layer. Here is an unfortunate blemish in
  # my design...there is really no clean way to i18nify (or whatever the verb
  # is) a failure in the operation, so failure will croak in English.
  # Hopefully, this subroutine acts in a fairly limited context, so we
  # shouldn't have to worry too much.
  my $fileName = catfile($path, $base);
  # Set up the umask to produce -rw------- files and save the old one
  my $old_umask = umask; umask 0077;
  # Create the IO::File object
  my $fh = IO::File->new($fileName, '>');
  # Revert to the old umask
  umask $old_umask;
  croak $fileName, ": $!" unless $fh;

  # /Successful/ fall-through; return the filename (and the IO::File, in list
  # context)
  if (wantarray) {
    return ($fh, $fileName)
  } else {
    $fh->close; return $fileName;
  }
}

sub makeConfigResource {
  croak 'Path must be given!' unless @_;
  return makeResource(type => 'config', path => \@_);
}

sub makeDataResource {
  croak 'Path must be given!' unless @_;
  return makeResource(type => 'data', path => \@_);
}

1;

__END__

=head1 NAME

VASM::Resource - resource location facility compliant with XDG Base Directory
specification

=head1 SYNOPSIS

    use VASM::Resource; 

    # Look for:
    # 
    # ~/.config/vasm/Foo/Bar/Baz/config.xml
    # /etc/xdg/vasm/Foo/Bar/Baz/config.xml
    #
    # ...in that order.
    my $configFH = (findConfigResource(qw/Foo Bar Baz config.xml/))[0];
    
    # You get the idea...
    my $dataFH = (findDataResource(qw/Some Other Stuff foo.png/))[0];

    # You can also get the name as well as the handle...something like:
    # /home/hanuman/.config/vasm/Foo/Bar/Baz/config.xml
    my $configName = findConfigResource(qw/Foo Bar Baz config.xml/);

    # Create a new config resource; this function internally uses
    # makeResourcePath. makeConfigResource will return the name in list
    # context, too!
    my $newConfigFH = (makeConfigResource(qw/Foo Bar Qux config.xml/))[0];
    ...
    $newConfigFH->close; # You will have to close it, though

=head1 DESCRIPTION

VASM::Resource provides for location of resource files in directories
containing hierarchies of resources, and returns IO::File objects to their
contents (and optionally names); these collections are searched in order, and
matches which most closely match the given request for a resource, found in
the user-specific hierarchy, are preferred over more general system default
matches. If no matches could be found, an appropriate error message will be
printed, albeit limited to English. In the interest of alliance with standards
(and enlightened laziness) VASM::Resource complies with the XDG Base Directory
specification.

=head1 FUNCTIONS

The 'config' functions are given as examples, but the 'data' functions act in
the same way, taking the obvious differences out of account. In truth, they
are merely wrappers for generic routines that handle either logical group of
information.

=over

=item findConfigResource(@path, $file)

findConfigResource accepts a plain list as an argument, indicating a directory
pathway and file, with which it locates a given configuration resource file as
well as it can, returning the absolute path to the file, or a list consisting
of an IO::File object representing the file and the filename, when called in
list context. VASM::Resource will conduct a search according to this formula:
XDG config dirs + 'vasm' + path. In other words, findConfigResource will look
under each XDG configuration directory, under the prefix directory 'vasm', for
the given path and file. If unsuccessful in finding a matching file,
findConfigResource will give up and return nothing. If unsuccessful in finding
a matching file, findConfigResource will croak with an appropriate error
message.

=item makeConfigResourcePath(@path)

makeConfigResourcePath accepts a list defining a resource path, and will
attempt to create it, if even necessary, using the serial permission 0700
(exclusive read, write, and execute rights for the owner). If unsuccessful,
the underlying mkpath function will croak out a diagnostic message. Otherwise,
it will return the absolute path to the new (or existent) directory.

=item makeConfigResource(@path, $file)

Given a list defining a resource path and a target file, makeConfigResource
will create any intervening directories, if necessary, and then instantiate an
IO::File object associated with the given file for writing. If called in list
context, makeConfigResource will return the IO::File object and the absolute
path to the filename; in scalar context, it will close the filehandle (think
of it like the idiomatic use of the 'touch' program to create blank files) and
simply return the name. If unsuccessful in creating an the file,
makeConfigResource will croak.

=back

=head1 ENVIRONMENT

=over

=item XDG_CONFIG_HOME

Directory relative to which user-specific configuration files should be
written; defaults to $HOME/.config.

=item XDG_DATA_HOME

Directory relative to which user-specific data should be written; defaults to
'$HOME/.local/share'.

=item XDG_CONFIG_DIRS

Colon-delimited, preference-ordered list of directores relative to which
configuration files should be searched after $XDG_CONFIG_HOME; defaults to
'/etc/xdg'.

=item XDG_DATA_DIRS

Colon-delimited, preference-ordered list of directores relative to which
configuration files should be searched after $XDG_DATA_HOME; defaults to
'/usr/local/share:/usr/share'.

=back

=head1 AUTHORS

hanumizzle L<mailto:hanumizzle@gmail.com> wrote VASM::Resource.

=head1 SEE ALSO

=over

=item *

XDG Base Directory specification:
L<http://standards.freedesktop.org/basedir-spec/>, implemented by
L<File::Basedir>.

=back

=cut
