# -*- project-name: VASM -*-
package VASM::Catalog::Menu;

use strict;
use warnings;
use VASM::Tree;
use base qw/VASM::Catalog::Menu::Translation/;
use Carp;
use XML::Parser;
use XML::Writer;

our $VERSION = '1.07';

sub new {
  my ($self, $file) = @_; 
  my $instance = { menu => VASM::Tree->new };
  bless $instance, $self;
  
  # Intern the contents of the IO::File $file if given
  $instance->Parse($file) if defined $file;

  return $instance;
}

sub Store {
  my ($self, $attrs, @path) = (shift, pop, @_);

  # At the very least, there must be a @path with one element and associated
  # properties
  croak 'Attributes hashref invalid!' unless @path and defined %$attrs;

  # Check contents thereof?

  # Notice that $attrs is copied when interned
  $self->{menu}->Store(@path, { %$attrs });
  
  return;
}

sub Retrieve {
  my ($self, @path) = @_;
  
  croak 'Path argument must be given to Retrieve!' unless @path;
  my $attrs = $self->{menu}->Retrieve(@path);
  # Here again, we copy the hash so that it may not be modified
  return { %$attrs } if defined $attrs;

  return;
}

sub Children {
  my ($self, @path) = @_;
  
  return $self->{menu}->Children(@path);
}

sub Parse {
  my ($self, $file) = @_;
  # See Parse in VASM::Message for lament
  my $parser = XML::Parser->new(Style => 'Tree', ErrorContext => 3);
  my $tree = $parser->parse($file);

  # This absolutely /has to/ recurse, and we obviously can't have multiple
  # instances of XML parsers running around. (I could check the information
  # from the caller built-in and special case it, but dollars to doughnuts
  # says I get reborn as Brian Peppers if God exists.)
  $self->parseHeavy($tree->[1]);
}

sub parseHeavyFolder {
  my ($self, $tags, @path) = (shift, pop, @_);
  my $attrs = $tags->[0];

  # Label and icon attributes must be defined
  for my $attr (qw/label icon/) {
    croak "$attr attribute undefined in folder tag!"
      unless length $attrs->{$attr};
  }

  # Check for an existing element at this path
  croak 'Two or more elements at the same name and level!'
    if defined $self->Retrieve(@path, $attrs->{label});
  # Store the folder
  $self->Store(@path, $attrs->{label}, { icon => $attrs->{icon} });
  # Recurse
  $self->parseHeavy($tags, @path, $attrs->{label});

  return;
}

sub parseHeavyItem {
  my ($self, $tags, @path) = (shift, pop, @_);
  my $attrs = $tags->[0];
  
  # Certain attributes must be defined
  for my $attr (qw/label path icon/) {
    croak "$attr attribute undefined in item tag!"
      unless length $attrs->{$attr};
  }

  # Check for an existing element at this path
  croak 'Two or more elements at the same name and level!'
    if defined $self->Retrieve(@path, $attrs->{label});
  # Store the item
  $self->Store(@path, $attrs->{label}, { path => $attrs->{path}, 
                                         icon => $attrs->{icon} });

  return;
}

sub parseHeavy {
  # $tags is the chunk of parsed XML under examination, and @path is a
  # stack representing the current folder hierarchy
  my ($self, $tags, @path) = @_;

  for my $index (grep { $_ % 2 } (0..$#{ $tags })) {
    next if $tags->[$index] eq '0'; # Skip text elements

    if ($tags->[$index] eq 'folder') {
      $self->parseHeavyFolder(@path, $tags->[$index + 1]);
    } elsif ($tags->[$index] eq 'item') {
      $self->parseHeavyItem(@path, $tags->[$index + 1]);
    }
  }

  return;
}

sub Write {
  my ($self, $file) = @_;

  croak 'Handle argument must be given to Write!' unless defined $file;

  # Now create an XML::Writer instance and turn that puppy out
  my $xmlFH = XML::Writer->new(OUTPUT => $file, ENCODING => 'utf-8',
                               DATA_MODE => 1, DATA_INDENT => 2);

  # Open up the document
  $xmlFH->xmlDecl;
  $xmlFH->startTag('catalog');
  $xmlFH->comment('Generated by VASM');

  # Now let writeHeavy do its stuff
  $self->writeHeavy($xmlFH); 

  # End the document...
  $xmlFH->endTag('catalog'); $xmlFH->end;

  return;
}

sub writeHeavy {
  my ($self, $xmlFH, @path) = @_;

  # Do something here
  for my $child ($self->Children(@path)) {
    my $attrs = $self->Retrieve(@path, $child);
    # Produce an item tag if appropriate
    if (defined $attrs->{path}) {
      $xmlFH->emptyTag('item', label => $child, %{ $attrs });
    }
    # Otherwise, recurse if the child in question has children of its own, in
    # between producing the appropriate opening and closing tags
    elsif ($self->Children(@path, $child)) {
      $xmlFH->startTag('folder', label => $child, icon => $attrs->{icon});
      $self->writeHeavy($xmlFH, @path, $child);
      $xmlFH->endTag('folder');
    }
    # Must be an empty folder! We will preserve it nevertheless
    else {
      $xmlFH->emptyTag('folder', label => $child, icon => $attrs->{icon});
    }
  }
}

1;

__END__

=head1 NAME

VASM::Catalog::Menu - general window manager menu definitions

=head1 SYNOPSIS

=head1 DESCRIPTION

=head1 METHODS

=head1 AUTHORS

hanumizzle L<hanumizzle@gmail.com> wrote VASM::Catalog::Menu. Further
considerations by cintyram.

=cut

