#!/usr/bin/perl
# -*- project-name: VASM -*-

use strict;
use warnings;
use constant { FALSE => 0, TRUE => 1 };
use constant { DATA => 0, LABEL => 1 };
use constant { VALUE => 0, KEY => 1, KEY_REAL => 2 };
use constant { FOLDER => 0, ITEM => 1 };
use constant { PAD => 2 };
use VASM::Resource::Catalog::Message;
use VASM::Resource::Catalog::Menu;
use File::Spec::Functions qw/curdir rel2abs splitpath/;
use Gtk2 '-init';

# Message files
my $CommonMsg = findMessageCatalog(qw/Common Message/);
my $UIMsg = findMessageCatalog(qw/Menu UI Gtk2 Message/);
my $ErrorMsg = findErrorCatalog(qw/Menu UI Error/);

sub makeMenuTree {
  my ($menu) = @_;

  # Set up a tree store corresponding to the menu; the single Glib::String
  # column indicates a particular folder or item name in the menu object
  my $menuTree = Gtk2::TreeStore->new('Glib::Scalar', 'Glib::String');

  # Add a top level imaginary root
  my $root = $menuTree->append(undef);
  $menuTree->set($root, LABEL, $UIMsg->Render('Window Manager Menu'));
  $menuTree->set($root, DATA, {});
  
  # makeMenuTreeHeavy needs to be factored out, as it is a /recursive/
  # function that will fill in the tree store itself
  makeMenuTreeHeavy($menu, $menuTree, $root);

  return $menuTree;
}

sub makeMenuTreeHeavy {
  my ($menu, $menuTree, @iters) = @_;
  
  # Get the current 'stack' of folders
  my @path = map { $menuTree->get($_, LABEL) } @iters[1..$#iters];

  # Create more folders under it
  for my $child ($menu->Children(@path)) {
    # If an item (only items have the path attribute)
    if ($menu->Retrieve(@path, $child)->{path}) {
      # Advance the pointer at the current level
      my $iter = $menuTree->append($iters[-1] or undef);
      # Set the item name and properties at the current path
      $menuTree->set($iter, DATA, $menu->Retrieve(@path, $child));
      $menuTree->set($iter, LABEL, $child);
    }
    # If a folder (by exclusion)
    else {
      push @iters, $menuTree->append($iters[-1] or undef);
      # Set the folder name at the current path
      $menuTree->set($iters[-1], LABEL, $child);
      $menuTree->set($iters[-1], DATA, $menu->Retrieve(@path, $child));
      makeMenuTreeHeavy($menu, $menuTree, @iters); # Recurse
      pop @iters; # Move back up a level
    }
  }
}

sub makeMenuTreeView {
  my ($menu, $modified) = @_;

  my $treeStore = makeMenuTree($menu); # The actual data
  # If the tree is changed, set the modified flag to TRUE
  $treeStore->signal_connect('row-changed' => sub { $$modified = TRUE });
  # The rest of this is a 'cookbook' recipe to make a tree view correspond
  # one-for-one with the data in the tree store
  my $treeView = Gtk2::TreeView->new_with_model($treeStore);
  $treeView->set_headers_visible(FALSE);
  $treeView->set_reorderable(TRUE);
  my $col = Gtk2::TreeViewColumn->new;
  my $renderer = Gtk2::CellRendererText->new;
  $col->pack_start($renderer, FALSE);
  $col->add_attribute($renderer, text => LABEL);
  $treeView->append_column($col);

  # Attach a signal handler for double clicks on row
  $treeView->signal_connect(
    'row-activated' => sub {
      my ($store, $iter) = $_[0]->get_selection->get_selected;
      editMenuElement($store, $iter) if $store->iter_parent($iter);
    });

  return $treeView;
}

sub makeEditList {
  my %attrs = @_; my $iter;
  my $attrList = Gtk2::ListStore->new(qw/Glib::String/ x 3);

  for my $attr (qw/label path icon/) {
    # If the folder or item has this attr...
    if (defined $attrs{$attr}) {
      $iter = $attrList->append;
      # Name of attribute, translated
      $attrList->set($iter, KEY, $CommonMsg->Render(ucfirst $attr));
      # Used for ID
      $attrList->set($iter, KEY_REAL, $attr);
      # Value
      $attrList->set($iter, VALUE, $attrs{$attr});
    }
  }

  return $attrList;
}

sub makeEditListView {
  my ($menuStore, $menuElement) = @_;

  my $listStore = makeEditList(getMenuElementAttrs($menuStore, $menuElement));
  my $view = Gtk2::TreeView->new_with_model($listStore);
  $view->grab_focus; # Using keyboard shortcuts becomes easier

  my @fields = qw/Attribute Value/;
  for my $field (KEY, VALUE) {
    my $col = Gtk2::TreeViewColumn->new;
    # Set 'Attribute' and 'Value' headers
    $col->set_title($CommonMsg->Render($fields[$field]));
    my $renderer = Gtk2::CellRendererText->new;
    # The 'value' column is editable
    if ($field eq VALUE) {
      $renderer->set(editable => TRUE);
      $renderer->signal_connect(edited => \&cellEdited, $listStore);
    }
    $col->pack_start($renderer, FALSE);
    $col->add_attribute($renderer, text => $field);
    $view->append_column($col);
  }

  return $view;
}

sub getMenuElementAttrs {
  my ($menuStore, $menuElement) = @_;
  my %attrs;
  
  $attrs{label} = $menuStore->get($menuElement, LABEL);
  my $menuAttrs = $menuStore->get($menuElement, DATA);
  # Add attributes from the DATA column of the row if defined
  for my $attr (qw/path icon/) {
    $attrs{$attr} = $menuAttrs->{$attr} if defined $menuAttrs->{$attr};
  }
  
  return %attrs;
}

sub getEditListAttrs {
  my ($listStore) = @_; my %attrs;
  my $iter = $listStore->get_iter_first;
  
  # Iterate over the list store and grab pairs of KEY_REAL => VALUE, the
  # non-translated field name and its value
  do {
    $attrs{$listStore->get($iter, KEY_REAL)} = $listStore->get($iter, VALUE);
  } while ($iter = $listStore->iter_next($iter));

  return %attrs;
}

sub attrListUnique {
  my ($menuStore, $menuElement, $attrList) = @_;
  my %attrList = getEditListAttrs($attrList);
  # The first row at the same level as $menuElement
  my $iter = $menuStore->iter_children($menuStore->iter_parent($menuElement));

  do {{
    # Ignore comparisons between the element being edited and itself
    next if $menuStore->get_string_from_iter($menuElement) eq
            $menuStore->get_string_from_iter($iter);
    my %menuAttrs = getMenuElementAttrs($menuStore, $iter);
    # Return false if the label for the edited menu element is not unique
    return FALSE if $menuAttrs{label} eq $attrList{label};
  }} while ($iter = $menuStore->iter_next($iter));
  
  # Otherwise, it falls through and returns true
  return TRUE;
}

sub attrListFull {
  my ($attrList) = @_;
  my %attrList = getEditListAttrs($attrList);
  
  # Return TRUE if the number of elements in $attrList with a length of 1 or
  # more is equal to the number of keys
  return TRUE if (grep { length } values %attrList) == keys %attrList;
  return FALSE;
}

sub setMenuElement {
  my ($menuStore, $menuElement, $attrList) = @_;
  my %attrList = getEditListAttrs($attrList);

  # Set the label first
  $menuStore->set($menuElement, LABEL, $attrList{label});
  # Now set all that other stuff
  delete $attrList{label};
  $menuStore->set($menuElement, DATA, \%attrList);
}

sub addMenu {
  my ($menuStore, $parent) = $_[1]->get_selection->get_selected;

  # Check for item
  if ($menuStore->get($parent, DATA)->{path}) {
    my $error = Gtk2::MessageDialog->new(
      undef, [], qw/error ok/,
      $ErrorMsg->Render('Attempted Addition to Item'));
    $error->set_position('center'); $error->run; $error->destroy;
  } else { # If it is a folder...
    addMenuElement($menuStore, $parent);
  }

  return;
}

sub removeMenu {
  my $view = $_[1];
  my $menuStore = $view->get_model;
  my $iter = $view->get_selection->get_selected;

  # Don't delete r00t!!!
  unless ($menuStore->iter_parent($iter)) {
    # Report error
    my $error = Gtk2::MessageDialog->new(
      undef, [], qw/error ok/,
      $ErrorMsg->Render('Attempted Deletion of Root'));
      $error->set_position('center'); $error->run; $error->destroy;
    return;
  }

  # Confirm the deletion, if the iter has children
  if ($menuStore->iter_children($iter)) {
    my $confirmation = Gtk2::MessageDialog->new(
      undef, [], qw/question yes-no/,
      $UIMsg->Render('Confirm Deletion'));
    $confirmation->set_position('center');

    # Get a reply
    $confirmation->show_all;
    my $response = $confirmation->run;
    $confirmation->destroy;
    
    return unless $response eq 'yes';
  }
  
  $menuStore->remove($iter);
  return;
}

sub upMenu {
  my $view = $_[1];
  my $menuStore = $view->get_model;
  my $iter = $view->get_selection->get_selected;
  my $prev = eval {
    my $path = $menuStore->get_path($iter);
    return $menuStore->get_iter($path) if $path->prev;
  };
  $menuStore->swap($prev, $iter) if $prev;
  
  return;
}

sub downMenu {
  my $view = $_[1];
  my $menuStore = $view->get_model;
  my $iter = $view->get_selection->get_selected;
  my $next = eval {
    my $path = $menuStore->get_path($iter); $path->next;
    return $menuStore->get_iter($path);
  };
  $menuStore->swap($iter, $next) if $next;
  
  return;
}

sub cellEdited {
  my ($path, $text, $listStore) = @_[1..3];

  # Just pass the text through
  $listStore->set($listStore->get_iter_from_string($path), VALUE, $text);

  return;
}

sub updateIconPreview {
  my ($fileChooser, $preview) = @_;
  
  my $file = $fileChooser->get_preview_filename;
  # For some reason, this callback gets spurious bullshit from time to time
  if (defined $file and -f $file) {
    my $pixbuf = Gtk2::Gdk::Pixbuf->new_from_file_at_size($file, 64, 64);
    $preview->set_from_pixbuf($pixbuf);
    $fileChooser->set_preview_widget_active(defined $pixbuf ? TRUE : FALSE);
  } else {
    $fileChooser->set_preview_widget_active(FALSE);
  }

  return;
}

sub identifyMenuRow {
  my ($menuStore, $iter) = @_; my @path;
  
  # Walk from branch to trunk, unshifting the 'real' column at each point. The
  # result will define the path in the tree store
  do {
    unshift @path, $menuStore->get($iter, LABEL)
      if $menuStore->iter_parent($iter);
    # The top-level 'Window Manager Menu' entry does not count
  } while ($iter = $menuStore->iter_parent($iter));

  return @path;
}

sub generateMenu {
  my ($menuStore) = @_;
  my $menu = VASM::Catalog::Menu->new;

  # Intern each element of the menu description from $menuStore
  $menuStore->foreach(
    sub {
      my ($menuStore, $iter) = @_[0,2];

      # Textual description of the path
      my @path = identifyMenuRow($menuStore, $iter);
      # Handle top-level special case
      return FALSE unless @path;
      # Element attributes
      my %attrs = getMenuElementAttrs($menuStore, $iter); delete $attrs{label};
      # Store the resulting data
      $menu->Store(@path, \%attrs);

      return FALSE;
    });

  return $menu;
}

sub addMenuElement {
  my ($menuStore, $parent) = @_;

  # Dialog
  my $dialog = Gtk2::Dialog->new_with_buttons(
    $UIMsg->Render('Select Type'),
    undef, [],
    'gtk-ok' => 'ok',
    'gtk-cancel' => 'cancel');
  $dialog->set_position('center');

  # Radio buttons
  # Speaking of radio: http://wdiy.org -- do give it a look!
  my $selected = 'Item';

  # Folder
  my $radio = Gtk2::RadioButton->new(undef, $CommonMsg->Render('Folder'));
  $radio->signal_connect(toggled => sub { $selected = 'folder' });
  $dialog->vbox->pack_start($radio, TRUE, TRUE, PAD);
  # Item (default)
  $radio = Gtk2::RadioButton->new($radio, $CommonMsg->Render('Item'));
  $radio->signal_connect(toggled => sub { $selected = 'item' });
  $radio->set_active(TRUE);
  $dialog->vbox->pack_start($radio, TRUE, TRUE, PAD);

  # Expander with detailed description
  my $shortDescription = Gtk2::Expander->new(
    $UIMsg->Render('Select Type Short Description'));
  my $longDescription = Gtk2::Label->new(
    $UIMsg->Render('Select Type Long Description'));
  $longDescription->set_line_wrap(TRUE);
  $shortDescription->add($longDescription);
  $dialog->vbox->pack_start($shortDescription, FALSE, FALSE, PAD);

  $dialog->show_all;
  my $response = $dialog->run;
  $dialog->destroy;
  
  return unless $response eq 'ok';

  # Otherwise, call a function according to the selected radio button
  my $newMenuElement = $menuStore->append($parent);
  # Fill in some blank 'default' values
  $menuStore->set($newMenuElement, LABEL, '');
  if ($selected eq 'folder') { 
    $menuStore->set($newMenuElement, DATA, { icon => '' }); 
  } elsif ($selected eq 'item') {
    $menuStore->set($newMenuElement, DATA, { path => '', icon => '' });
  }
  
  # Remove the new iter unless the user didn't decide to cancel
  $menuStore->remove($newMenuElement)
    unless editMenuElement($menuStore, $newMenuElement);
  
  return;
}

BEGIN {
  my $wd = rel2abs(curdir);
  
  sub findPathOrIcon {
    my $view = $_[1];
    my ($store, $iter) = $view->get_selection->get_selected;

    # The label attribute has no 'Find' option!!
    return if $store->get($iter, KEY_REAL) eq 'label';

    my $fileChooser = Gtk2::FileChooserDialog->new(
      $UIMsg->Render('Select File'), undef,
      'open',
      'gtk-ok' => 'ok',
      'gtk-cancel' => 'cancel');
    $fileChooser->set_current_folder($wd);
    $fileChooser->set_position('center');

    if ($store->get($iter, KEY_REAL) eq 'icon') {
      # Set filter on file chooser as well -- only image formats
      my $filter = Gtk2::FileFilter->new; $filter->add_pixbuf_formats;
      $fileChooser->set_filter($filter);
      # Icons will have preview
      my $preview = Gtk2::Image->new;
      $fileChooser->set_preview_widget($preview);
      $fileChooser->signal_connect('update-preview' => \&updateIconPreview,
                                   $preview);
    }

    $fileChooser->show_all;
    my $response = $fileChooser->run;
    if ($response eq 'ok') {
      my $file = $fileChooser->get_filename;
      $wd = (splitpath($file))[1]; # Store new current directory
      $store->set($iter, VALUE, $file); # Set element in store
    }

    $fileChooser->destroy;

    return;
  }
}

sub editMenuElement {
  my ($menuStore, $menuElement) = @_;

  # Dialog
  my $dialog = Gtk2::Dialog->new_with_buttons(
    $UIMsg->Render('Edit Properties'),
    undef, [],
    'gtk-ok' => 'ok',
    'gtk-cancel' => 'cancel');
  $dialog->set_position('center');

  # Attributes list
  my $attrListView = makeEditListView($menuStore, $menuElement);
  $dialog->vbox->pack_start($attrListView, TRUE, TRUE, PAD);

  # Operations (actually, only one)
  my $operationHButtonBox = Gtk2::HButtonBox->new;
  $operationHButtonBox->set_layout('spread');
  my @operations = ( [ 'gtk-find', \&findPathOrIcon ] );
  # Add them, well...it...one by one
  for my $operation (@operations) {
    my $button = Gtk2::Button->new_from_stock($operation->[0]);
    $button->signal_connect(clicked => $operation->[1], $attrListView);
    $operationHButtonBox->pack_start($button, FALSE, FALSE, PAD);
  }
  $dialog->vbox->pack_start($operationHButtonBox, FALSE, FALSE, PAD);

  # Self-documentation
  my $shortDescription = Gtk2::Expander->new(
    $UIMsg->Render('Edit Properties Short Description'));
  my $longDescription = Gtk2::Label->new(
    $UIMsg->Render('Edit Properties Long Description'));
  $longDescription->set_line_wrap(TRUE);
  $shortDescription->add($longDescription);
  $dialog->vbox->pack_start($shortDescription, FALSE, FALSE, PAD);

  $dialog->show_all;

  DIALOG: {
    my $response = $dialog->run; # Do it Jah!
    if ($response eq 'ok') {
      # Are all fields full?
      unless (attrListFull($attrListView->get_model)) {
        # Report error
        my $error = Gtk2::MessageDialog->new(
          undef, [], qw/error ok/,
          $ErrorMsg->Render('Missing Fields'));
        $error->set_position('center'); $error->run; $error->destroy;
        redo DIALOG; # Do over
      }
      # Is the element unique?
      unless (attrListUnique($menuStore, $menuElement, 
                             $attrListView->get_model)) {
        # Report error
        my $error = Gtk2::MessageDialog->new(
          undef, [], qw/error ok/,
          $ErrorMsg->Render('Element Not Unique'));
        $error->set_position('center'); $error->run; $error->destroy;
        redo DIALOG; # Do over
      }
      # If we made it here, that means we get to store the attributes loaded
      # in the editing window
      setMenuElement($menuStore, $menuElement, $attrListView->get_model);
    }

    $dialog->destroy;
    return ($response eq 'ok') ? 1 : 0;
  }
}

sub mainMenu {
  my $dialog = Gtk2::Dialog->new_with_buttons(
    $UIMsg->Render('Title'),
    undef, [],
    'gtk-apply' => 'apply',
    'gtk-ok' => 'ok',
    'gtk-close' => 'close');
  $dialog->set_position('center');
  $dialog->set_default_size(400, 300);

  ## Action area
  # TreeView of menu -- but, first, get the menu definition
  my $menu = findMenu(); my $modified = FALSE;
  my $menuView = makeMenuTreeView($menu, \$modified);
  
  # Pack the tree view into a scrolled window. The addition of scrollbars
  # will be done automatically in either direction
  my $scrolledView = Gtk2::ScrolledWindow->new;
  $scrolledView->set_policy(qw/automatic/ x 2);
  $scrolledView->add($menuView);
  $dialog->vbox->pack_start($scrolledView, TRUE, TRUE, PAD);
  
  # Operations
  my $operationHButtonBox = Gtk2::HButtonBox->new;
  $operationHButtonBox->set_layout('spread');
  my @operations = ( [ 'gtk-add' => \&addMenu ],
                     [ 'gtk-remove' => \&removeMenu ],
                     [ 'gtk-go-up' => \&upMenu ],
                     [ 'gtk-go-down' => \&downMenu ] );
  # Add them, one by one
  for my $operation (@operations) {
    my $button = Gtk2::Button->new_from_stock($operation->[0]);
    $button->signal_connect(clicked => $operation->[1], $menuView);
    $operationHButtonBox->pack_start($button, TRUE, TRUE, PAD);
  }
  $dialog->vbox->pack_start($operationHButtonBox, FALSE, FALSE, PAD);

  # Expander with detailed description
  my $shortDescription = Gtk2::Expander->new(
    $UIMsg->Render('Main Menu Short Description'));
  my $longDescription = Gtk2::Label->new(
    $UIMsg->Render('Main Menu Long Description'));
  $longDescription->set_line_wrap(TRUE);
  $shortDescription->add($longDescription);
  $dialog->vbox->pack_start($shortDescription, FALSE, FALSE, PAD);

  $dialog->show_all;

  DIALOG: {
    my $response = $dialog->run;
    if ($response eq 'apply') {
      $menu = generateMenu($menuView->get_model);
      writeMenu($menu);
      $modified = FALSE; # Having saved the menu, $modified is now FALSE
      redo DIALOG;
    } elsif ($response eq 'ok') {
      $menu = generateMenu($menuView->get_model);
      writeMenu($menu);
    } elsif ($response eq 'close' and $modified) {
      # Confirm this
      my $confirm = Gtk2::MessageDialog->new(
        undef, [], qw/question yes-no/,
        $UIMsg->Render('Confirm Cancellation'));
      $confirm->set_position('center');
      my $response = $confirm->run; $confirm->destroy;
      # If answered no, stay here
      $response eq 'no' and redo DIALOG;
    }
  }

  $dialog->destroy;
  exit 0;
}

mainMenu();
