#!/usr/bin/perl
# -*- project-name: VASM -*-

use strict;
use warnings;
use constant { FALSE => 0, TRUE => 1 };
use constant { LABEL => 0, LABEL_REAL => 0, LABEL_SHOWN => 1 };
use constant { PAD => 2 };
use VASM::Tree;
use VASM::Resource;
use VASM::Resource::Catalog::Message;
use VASM::Resource::Catalog::Action;
use Gtk2 '-init';
use File::Spec::Functions qw/curdir rel2abs splitpath/;
use File::MimeInfo::Magic;
use Archive::Tar;

# Message files
my $CommonMsg = findMessageCatalog(qw/Common Message/);
my $UIMsg = findMessageCatalog(qw/Action UI Gtk2 Message/);
my $ErrorMsg = findErrorCatalog(qw/Action UI Error/);

sub makeActionTree {
  # Get directories for file and URI schemata
  my $fileDirectory = mediaDirectory('File');
  my $uriDirectory = mediaDirectory('URI');

  # Convert the file directory to data more readily usable by the function
  my $fileTypeTree = VASM::Tree->new;
  for my $mime ($fileDirectory->List) {
    $fileTypeTree->Store((split m!/!, $mime), TRUE);
  }

  # Now, set up a tree store corresponding to them and add the 'Template'
  # top-level choice as well. The first Glib::String is the item as it appears
  # on the screen, according to translation and additional notes. The second
  # is the 'real name', usable to the program.
  my $actionTree = Gtk2::TreeStore->new('Glib::String', 'Glib::String');
  my @iters;

  # File
  push @iters, $actionTree->append(undef);
  $actionTree->set($iters[-1],
                   LABEL_REAL, 'File',
                   LABEL_SHOWN, $CommonMsg->Render('File')
                 );
  for my $class (sort $fileTypeTree->Children) {
    # Add the class under 'File'
    push @iters, $actionTree->append($iters[-1]);
    $actionTree->set($iters[-1],
                     LABEL_REAL, $class,
                     LABEL_SHOWN, $class);
    # Add each member of the class (e.g., File -> image -> jpeg)
    for my $member (sort $fileTypeTree->Children($class)) {
      push @iters, $actionTree->append($iters[-1]);
      my $mime = join('/', $class, $member);
      # The type will appear with a parenthetical note describing it in
      # further detail
      $actionTree->set(
        $iters[-1],
        LABEL_REAL, $member,
        LABEL_SHOWN, $member . ' (' . $fileDirectory->Render($mime) . ')',
      );
    } continue { pop @iters }
  } continue { pop @iters }
  pop @iters;

  # URI
  push @iters, $actionTree->append(undef);
  $actionTree->set($iters[-1],
                   LABEL_REAL, 'URI',
                   LABEL_SHOWN, $CommonMsg->Render('URI'));
  for my $scheme (sort $uriDirectory->List) {
    push @iters, $actionTree->append($iters[-1]);
    $actionTree->set(
      $iters[-1],
      LABEL_REAL, $scheme,
      LABEL_SHOWN, $scheme . ' (' . $uriDirectory->Render($scheme) . ')');
  } continue { pop @iters }
  pop @iters;

  # Template
  push @iters, $actionTree->append(undef);
  $actionTree->set($iters[-1],
                   LABEL_REAL, 'Template',
                   LABEL_SHOWN, $CommonMsg->Render('Template'));
  pop @iters; # Neat freak

  return $actionTree;
}

sub makeActionTreeView {
  my $treeStore = makeActionTree(); # The actual data
  # The rest of this is a 'cookbook' recipe to make a tree view correspond
  # one-for-one with the data in the tree store.
  my $treeView = Gtk2::TreeView->new_with_model($treeStore);
  $treeView->set_headers_visible(FALSE);
  my $col = Gtk2::TreeViewColumn->new;
  my $renderer = Gtk2::CellRendererText->new;
  $col->pack_start($renderer, FALSE);
  $col->add_attribute($renderer, text => LABEL_SHOWN);
  $treeView->append_column($col);
  
  # Attach a signal handler for double clicks on rows
  $treeView->signal_connect(
    'row-activated' => sub { 
      dispatchMediaType(identifyTreeRow($_[0]))
    });

  return $treeView;
}

sub makeMediaList {
  my ($catalog) = @_; my $iter;
  my $mediaList = Gtk2::ListStore->new('Glib::String');
  my @programs = $catalog->Dump;

  if (@programs) { # Add each program in the catalog to the list store
    for my $program (@programs) {
      $iter = $mediaList->append;
      $mediaList->set($iter, LABEL, $program);
    }
  } else { # Empty or non-existent catalog
    $iter = $mediaList->append;
    $mediaList->set($iter, LABEL, ''); # Add a new element
  }
  
  return $mediaList;
}

sub makeMediaListView {
  my ($catalog) = @_;

  # Cookbook recipe that instantiates a list view store from an actions
  # catalog and imposes an editable view component on it
  my $store = makeMediaList($catalog);
  my $view = Gtk2::TreeView->new_with_model($store);
  $view->set_headers_visible(FALSE);
  $view->set_reorderable(TRUE); # Allow drag and drop :D
  $view->grab_focus; # Using keyboard shortcuts becomes easier 
  my $col = Gtk2::TreeViewColumn->new;
  my $renderer = Gtk2::CellRendererText->new;
  $renderer->set(editable => TRUE);
  $renderer->signal_connect(edited => \&cellEdited, $store);
  $col->pack_start($renderer, FALSE);
  $col->add_attribute($renderer, text => LABEL);
  $view->append_column($col);

  return $view;
}

sub cellEdited {
  my ($pathString, $text, $store) = @_[1..3];

  # Just pass the text through
  $store->set($store->get_iter_from_string($pathString), LABEL, $text);
  
  return;
}

sub identifyTreeRow {
  my ($store, $iter) = $_[0]->get_selection->get_selected; my @path;

  # Walk from branch to trunk, unshifting the 'real' column at each point. The
  # result will define the path in the tree store.
  do {
    unshift @path, $store->get($iter, LABEL_REAL);
  } while ($iter = $store->iter_parent($iter));

  return @path;
}

sub addProgram {
  my $view = $_[1];
  my $store = $view->get_model;
  my $iter = $store->insert_after(scalar $view->get_selection->get_selected);
  $store->set($iter, LABEL, ''); # Add a new element

  return;
}

sub removeProgram {
  my $view = $_[1];
  my $store = $view->get_model;
  my $iter = $view->get_selection->get_selected;
  $store->remove($iter) if $iter;

  return;
}

sub upProgram {
  my $view = $_[1];
  my $store = $view->get_model;
  my $iter = $view->get_selection->get_selected;
  my $prev = eval {
    my $path = $store->get_path($iter);
    return $store->get_iter($path) if $path->prev;
  };
  $store->swap($prev, $iter) if $prev;
  
  return;
}

sub downProgram {
  my $view = $_[1];
  my $store = $view->get_model;
  my $iter = $view->get_selection->get_selected;
  my $next = eval {
    my $path = $store->get_path($iter); $path->next;
    return $store->get_iter($path);
  };
  $store->swap($iter, $next) if $next;
  
  return;
}

sub generateCatalog {
  my ($store) = @_; my $iter;
  my $catalog = VASM::Catalog::Action->new;

  # Add an element in the catalog for each member in the list store
  $store->foreach(
    sub { $catalog->Push($_[0]->get($_[2], LABEL)); return FALSE }
  );
  
  return $catalog;
}

sub mediaListFull {
  my ($listStore) = @_; my $iter = $listStore->get_iter_first;
  
  # Return FALSE unless all elements of the list store are not blank
  do {
    return FALSE unless length $listStore->get($iter, LABEL);
  } while ($iter = $listStore->iter_next($iter));

  return TRUE; # Fall through
}

sub dispatchMediaType {
  my @path = @_;

  # Fully qualified path was given
  if (($path[0] eq 'File' and @path == 3) or
      ($path[0] eq 'URI' and @path == 2)) {
    configureMedium(@path);
  } elsif ($path[0] eq 'Template') { # Select a template
    selectTemplate();
  }
  
  return;
}

sub mainMenu {
  # Dialog
  my $dialog = Gtk2::Dialog->new_with_buttons(
    $UIMsg->Render('Title'),
    undef, [], 
    'gtk-close' => 'close');
  $dialog->set_position('center');
  $dialog->set_default_size(400, 300);
    
  ## Action area
  # Tree with media
  my $actionTree = makeActionTreeView();
  # Pack the tree view into a scrolled window. The addition of scrollbars
  # will be done automatically in either direction
  my $scrolledTree = Gtk2::ScrolledWindow->new;
  $scrolledTree->set_policy('automatic', 'automatic');
  $scrolledTree->add($actionTree);
  $dialog->vbox->pack_start($scrolledTree, TRUE, TRUE, PAD);
  # Expander with detailed description
  my $shortDescription = Gtk2::Expander->new(
    $UIMsg->Render('Main Menu Short Description'));
  my $longDescription = Gtk2::Label->new(
    $UIMsg->Render('Main Menu Long Description'));
  $longDescription->set_line_wrap(TRUE);
  $shortDescription->add($longDescription);
  $dialog->vbox->pack_start($shortDescription, FALSE, FALSE, PAD);

  # Do it Jah!
  $dialog->show_all;
  $dialog->run;
  $dialog->destroy;
  
  exit 0;
}

sub configureMedium {
  my @path = @_;
  my $mediaType = (@path == 3) ? join '/', @path[1..2] : $path[1];

  # First, load the action catalog, if possible
  my $catalog = findActionCatalog(@path);

  # Dialog
  my $dialog = Gtk2::Dialog->new_with_buttons(
    $mediaType,
    undef, [],
    'gtk-ok' => 'ok',
    'gtk-cancel' => 'cancel');
  $dialog->set_position('center');

  # Media list
  my $mediaList = makeMediaListView($catalog);
  $dialog->vbox->pack_start($mediaList, TRUE, TRUE, PAD);

  # Operations
  my $operationHButtonBox = Gtk2::HButtonBox->new;
  $operationHButtonBox->set_layout('spread');
  my @operations = ( [ 'gtk-add' => \&addProgram ],
                     [ 'gtk-remove' => \&removeProgram ],
                     [ 'gtk-go-up' => \&upProgram ],
                     [ 'gtk-go-down' => \&downProgram ] );
  # Add them, one by one
  for my $operation (@operations) {
    my $button = Gtk2::Button->new_from_stock($operation->[0]);
    $button->signal_connect(clicked => $operation->[1], $mediaList);
    $operationHButtonBox->pack_start($button, FALSE, FALSE, PAD);
  }
  $dialog->vbox->pack_start($operationHButtonBox, FALSE, FALSE, PAD);

  # Description
  my $shortDescription = Gtk2::Expander->new(
      $UIMsg->Render(ID => 'Medium Configurator Short Description',
                     Arguments => [ $mediaType ]));
  my $longDescription = Gtk2::Label->new(
    $UIMsg->Render('Medium Configurator Long Description'));
  $longDescription->set_line_wrap(TRUE);
  $shortDescription->add($longDescription);
  $dialog->vbox->pack_start($shortDescription, FALSE, FALSE, PAD);

  $dialog->show_all;

  DIALOG: {
    my $response = $dialog->run;
    # Write out the catalog if so desired
    if ($response eq 'ok') {
      # Are all fields full?
      unless (mediaListFull($mediaList->get_model)) {
        # Report error
        my $error = Gtk2::MessageDialog->new(
          undef, [], qw/error ok/,
          $ErrorMsg->Render('Missing Fields'));
        $error->set_position('center'); $error->run; $error->destroy;
        redo DIALOG; # Do over
      }
      $catalog = generateCatalog($mediaList->get_model);
      writeActionCatalog($catalog, @path);
    }
  }
  $dialog->destroy;

  return;
}

sub confirmTemplate {
  my ($template) = @_;

  # Confirm the installation
  my $confirmation = Gtk2::MessageDialog->new(
    undef, [], qw/question yes-no/,
    $UIMsg->Render('Confirm Installation'));
  $confirmation->set_position('center');
  my $response = $confirmation->run; $confirmation->destroy;

  # Go ahead and install if confirmed
  installTemplate($template) if $response eq 'yes';
}

sub installTemplate {
  my ($template) = @_;

  # Check the MIME type of the file
  my $mimetype = mimetype($template);

  # Raise an error if needed
  unless ($mimetype =~ m!^application/x-(?:compressed-)?tar$!) {
    my $error = Gtk2::MessageDialog->new(undef, [], qw/error ok/,
                                         $ErrorMsg->Carp('Invalid Archive'));
    $error->set_position('center');
    $error->run;
    $error->destroy;
    return;
  }

  # Find the resource path and create it if necessary
  my $root = makeConfigResourcePath('Action');

  # Change directory and extract the template archive
  chdir $root; Archive::Tar->extract_archive($template);

  # Confirm the installation (to the user)
  my $confirmation = Gtk2::MessageDialog->new(
    undef, [], qw/info ok/,
    $UIMsg->Render('Installation Complete'));
  $confirmation->set_position('center');
  $confirmation->run;
  $confirmation->destroy;

  return;
}

BEGIN {
  my $wd = rel2abs(curdir);
  
  sub selectTemplate {
    my $fileChooser = Gtk2::FileChooserDialog->new(
      $UIMsg->Render('Select Template'), undef, 
      'open', 
      'gtk-ok' => 'ok',
      'gtk-cancel' => 'cancel');
    $fileChooser->set_current_folder($wd);
    $fileChooser->set_position('center');
    my $filter = Gtk2::FileFilter->new;
    # MIME-type selection seems unfortunately unreliable
    for my $ext (qw/tgz tar tar.gz/) {
       $filter->add_pattern("*.$ext");
    }
    $fileChooser->set_filter($filter);

    $fileChooser->show_all;
    my $response = $fileChooser->run;
    # If a given file is accepted...
    if ($response eq 'ok') {
      my $template = $fileChooser->get_filename;
      confirmTemplate($template); # Confirm install
      $wd = (splitpath($template))[1]; # Store new current directory
    }
    $fileChooser->destroy;

    return;
  }
}
  
mainMenu();
