#!/usr/bin/perl
# -*- project-name: VASM -*-

use strict;
use warnings;
use VASM::Resource;
use VASM::Resource::Catalog::Message;
use VASM::Resource::Catalog::Action;
use File::Spec::Functions qw/curdir rel2abs splitpath/;
use UI::Dialog::Backend::CDialog;
use File::MimeInfo::Magic;
use Archive::Tar;

# Message files
my $CommonMsg = findMessageCatalog(qw/Common Message/);
my $UIMsg = findMessageCatalog(qw/Action UI CDialog Message/);
my $ErrorMsg = findErrorCatalog(qw/Action UI Error/);

# Dialog instance
my $d = UI::Dialog::Backend::CDialog->new(
  backtitle => $CommonMsg->Render('VASM Backtitle'),
  title => $UIMsg->Render('Title'),
  'ok-label' => $CommonMsg->Render('OK'),
  'cancel-label' => $CommonMsg->Render('Exit'),
  'yes-label' => $CommonMsg->Render('Yes'),
  'no-label' => $CommonMsg->Render('No'),
  width => 75,
  height => 20,
  menuheight => 10
);

sub mainMenu {
  while (1) {
    # Prompt choice
    my $choice = $d->menu(
      text => $UIMsg->Render('Main Menu Description'),
      list =>
        [ map 
            { $UIMsg->Render($_),
              $UIMsg->Render("$_ Description") }
              qw/File URI Template/ ]);

    # Act on input
    if ($d->state eq 'OK') {
      if ($choice eq 'File' or $choice eq 'URI') {
        mediaMenu($choice);
      } elsif ($choice eq 'Template') {
        selectTemplate();
      }
    } elsif ($d->state eq 'CANCEL') {
      exit 0;
    }
  }
}

sub mediaMenu {
  my ($choice) = @_;
  my $directory = mediaDirectory($choice);

  while (1) {
    ### Read
    my @path = ($choice,
                split # For MIME types
                  m!/!,
                $d->menu(text => $UIMsg->Render('Media Menu Description'),
                         'extra-button' => 1,
                         'extra-label' => $CommonMsg->Render('Back'),
                         list =>
                           [ map 
                               { $_, $directory->Render($_) }
                               sort $directory->List ] ));

    ### Evaluate
    if ($d->state eq 'OK') {
      operationMenu(@path);
    } elsif ($d->state eq 'EXTRA') {
      return;
    } elsif ($d->state eq 'CANCEL') {
      exit 0;
    }
  }
}

sub operationMenu {
  my @path = @_;
  my $mediaType = (@path == 3) ? join '/', @path[1..2] : $path[1];

  while (1) {
    ### Read
    my $operation =
      $d->menu(text => $UIMsg->Render(ID => 'Operation Menu Description',
                                      Arguments => [ $mediaType ]),
               'extra-button' => 1,
               'extra-label' => $CommonMsg->Render('Back'),
               list =>
                 [ map
                     { $UIMsg->Render("$_ Operation"),
                       $UIMsg->Render("$_ Operation Description") }
                     qw/List Add Clear/ ] );

    ### Evaluate
    if ($d->state eq 'OK') {
      if ($operation eq 'List') {
        listMedium(@path);
      } elsif ($operation eq 'Add') {
        addMedium(@path);
      } elsif ($operation eq 'Clear') {
        clearMedium(@path);
      }
    } elsif ($d->state eq 'EXTRA') {
      return;
    } elsif ($d->state eq 'CANCEL') {
      exit 0;
    }
  }
}

sub listMedium {
  my @path = @_;
  my $mediaType = (@path == 3) ? join '/', @path[1..2] : $path[1];
  
  # Try to find a matching actions catalog
  my $catalog = findActionCatalog(@path);

  if (defined $catalog) {
    $d->msgbox(text => $UIMsg->Render(ID => 'Commands Listed',
                                      Arguments => [ $mediaType ])
                      . ":\\n\n\\n\n" # Yes, I know this this is ugly
                      . join("\\n\n", $catalog->Dump));
  } else { # Could not find a catalog!
    $d->msgbox(text => $ErrorMsg->Carp(ID => 'No Action Catalog',
                                       Arguments => [ $mediaType ]));
  }

  return;
}

sub addMedium {
  my @path = @_;

  # First, load the action catalog, if possible
  my $catalog = findActionCatalog(@path);
  
  # If it didn't exist, that's OK; just instantiate a blank one
  $catalog = VASM::Catalog::Action->new unless defined $catalog;

  my $command = $d->inputbox(text => $UIMsg->Render('Add Command'),
                             'cancel-label' => $CommonMsg->Render('Back'));

  if ($d->state eq 'OK') {
    # Emit warning message and return unless the command is a valid value
    $d->msgbox(text => $ErrorMsg->Render('Null Command String')) and return
      unless length $command;

    $catalog->Unshift($command); # Unshift it onto the catalog instance
    writeActionCatalog($catalog, @path); # Now write the instance out
  } 
  
  return;
}

sub clearMedium {
  my @path = @_;
  my $mediaType = (@path == 3) ? join '/', @path[1..2] : $path[1];
  
  # Try to find a matching actions catalog
  my $catalog = findActionCatalog(@path);

  if (defined $catalog) {
    # Confirm this destructive operation
    my $confirm = $d->yesno(text => $UIMsg->Render(ID => 'Confirm Clear',
                                                   Arguments =>
                                                     [ $mediaType ]));
    if ($confirm) { # "Do it!" -- Yellowman
      $catalog->Clear;
      writeActionCatalog($catalog, @path);
    }
  } else { # Could not find a catalog!
    $d->msgbox(text => $ErrorMsg->Carp(ID => 'No Action Catalog',
                                       Arguments => [ $mediaType ]));
  }
}

BEGIN {
  # selectTemplate has its own 'static' variable for the current working
  # directory; see perlfaq7
  my $wd = rel2abs(curdir);
  
  sub selectTemplate {
    # Get the filename of a template archive; start at current directory
    my $template = $d->fselect(path => $wd);
    
    if ($d->state eq 'OK') {
      # Confirm this destructive operation
      my $confirm = $d->yesno(text => $UIMsg->Render('Confirm Install'));
      
      if ($confirm) { # Do it...
        # Store new current working directory
        $wd = (splitpath($wd))[1];
          
        # Check the MIME type of the file
        my $mimetype = mimetype($template);
          
        $d->msgbox(text => $ErrorMsg->Carp('Invalid Archive')) and return
          unless $mimetype =~ m!^application/x-(?:compressed-)?tar$!;

        # Find the resource path and create it if necessary
        my $root = makeConfigResourcePath('Action');

        # Change directory and extract the template archive
        chdir $root; Archive::Tar->extract_archive($template);

        # Confirm the installation (to the user)
        $d->msgbox(text => $UIMsg->Render('Installation Complete'));
      }
    }
    
    return;
  }
}

# Start the application
mainMenu();
