# -*- project-name: VASM -*-
package VASM::Resource::Catalog::Menu;

use strict;
use warnings;
use base qw/Exporter/;
use Carp;
use VASM::Catalog::Menu;
use VASM::Resource;

our @EXPORT = qw/findMenu writeMenu/;
our $VERSION = '1.03';

sub findMenu {
  my @locale = parseLocale(); # Finagle locale from %ENV

  # Now, try to find the first matching Message.xml in the hierarchy; the
  # search can go as far back as the base 'Message' directory itself:
  my $file;
  LOOP: {
      do {
        $file = (findConfigResource('Menu', @locale, 'Menu.xml'))[0];
        last LOOP if defined $file;
      } while (pop @locale);
    }

  carp 'Could not find a matching menu file!'
    unless defined $file; # Return nothing unless a file was found, otherwise:
  # Digest file, if possible, and return the new menu definition. If no file
  # could be found, just instantiate a blank menu object.
  my $menu = VASM::Catalog::Menu->new;
  if (defined $file) {
    open my $fh, '<:utf8', $file or croak $file, ": $!";
    $menu->Parse($fh);
    close $fh;
  }
  
  return $menu;
}

sub writeMenu {
  my ($instance) = @_;
  my @locale = parseLocale(); # Finagle locale from %ENV

  # Generate a new IO::File for the catalog to be written. makeConfigResource
  # (actually, makeResource) will croak if unsuccessful, so no error checking
  # is really necessary
  my $file = makeConfigResource('Menu', @locale, 'Menu.xml');
  $instance->Write($file); # Now let the instance do its thing
  $file->close; # ...close the filehandle and...done
  
  return;
}

# Cut and pasted from VASM::Resource::Message:

# Because most of the normal divisions of locale support (e.g., LC_COLLATE,
# LC_NUMERIC) do not apply here, we will only honor a certain subset of
# environment variables, in this order: LC_ALL, LC_MESSAGES, LANG. (I believe
# this is the correct protocol; please correct me if I'm wrong.) Because VASM
# strictly uses UTF-8 for all messages, any codeset portion of the variable is
# ignored altogether.
sub parseLocale {
  # Loop through these variables in turn:
  for my $var (@ENV{qw/LC_ALL LC_MESSAGES LANG/}) {
    defined $var and return
      eval {
        my $locale = ($var =~ m!(\p{IsLower}{2}(?:_\p{IsUpper}{2})?)!)[0];
        # Split language and country (ex: en_GB vs. en_US)
        split /_/, $locale;
      }
  }

  # We buck the trend a little by using resource inheritance instead of the
  # default 'C' locale
  return;
}
  
1;

=head1 NAME

VASM::Resource::Catalog::Menu

=head1 SYNOPSIS

=head1 DESCRIPTION

=head1 FUNCTIONS

=head1 VARIABLES

=head1 AUTHORS

hanumizzle L<mailto:hanumizzle@gmail.com> wrote VASM::Resource::Catalog::Menu.

=cut

