#!/usr/bin/perl
# -*- project-name: VASM -*-

use strict;
use warnings;
use constant { FALSE => 0, TRUE => 1 };
use constant { DATA => 0, LABEL => 1 };
use constant { VALUE => 0, KEY => 1, KEY_REAL => 2 };
use constant { FOLDER => 0, ITEM => 1 };
use constant { PAD => 2 };
use VASM::Resource::Message;
use VASM::Resource::Menu;
use File::Spec::Functions qw/curdir rel2abs splitpath/;
use Gtk2 '-init';

# Message files
my $CommonMsg = findMessageCatalog(qw/Common Message/);
my $UIMsg = findMessageCatalog(qw/Menu UI Gtk2 Message/);
my $ErrorMsg = findErrorCatalog(qw/Menu UI Error/);

# Add 'Edit' button along with others

sub makeMenuTree {
  my ($menu) = @_;

  # Set up a tree store corresponding to the menu; the single Glib::String
  # column indicates a particular folder or item name in the menu object
  my $menuTree = Gtk2::TreeStore->new('Glib::Scalar', 'Glib::String');

  # Add a top level imaginary root
  my $root = $menuTree->append(undef);
  $menuTree->set($root, LABEL, $UIMsg->Render('Window Manager Menu'));
  $menuTree->set($root, DATA, {});
  
  # makeMenuTreeHeavy needs to be factored out, as it is a /recursive/
  # function that will fill in the tree store itself
  makeMenuTreeHeavy($menu, $menuTree, $root);

  return $menuTree;
}

sub makeMenuTreeHeavy {
  my ($menu, $menuTree, @iters) = @_;
  
  # Get the current 'stack' of folders
  my @path = map { $menuTree->get($_, LABEL) } @iters[1..$#iters];

  # Create more folders under it
  for my $child ($menu->Children(@path)) {
    if ($menu->Children(@path, $child)) {
      push @iters, $menuTree->append($iters[-1] or undef);
      # Set the folder name at the current path
      $menuTree->set($iters[-1], LABEL, $child);
      $menuTree->set($iters[-1], DATA, $menu->Retrieve(@path, $child));
      makeMenuTreeHeavy($menu, $menuTree, @iters); # Recurse
      pop @iters; # Move back up a level
    }
  }

  # Now, introduce the items after the folders
  for my $child ($menu->Children(@path)) {
    # Only menu items should have the path property
    if ($menu->Retrieve(@path, $child)->{path}) {
      # Advance the pointer at the current level
      my $iter = $menuTree->append($iters[-1] or undef);
      # Set the item name and properties at the current path
      $menuTree->set($iter, DATA, $menu->Retrieve(@path, $child));
      $menuTree->set($iter, LABEL, $child);
    }
  }
}

sub makeMenuTreeView {
  my ($menu, $dialog) = @_;

  my $treeStore = makeMenuTree($menu); # The actual data
  # The rest of this is a 'cookbook' recipe to make a tree view correspond
  # one-for-one with the data in the tree store.
  my $treeView = Gtk2::TreeView->new_with_model($treeStore);
  $treeView->set_headers_visible(FALSE);
  $treeView->set_reorderable(TRUE);
  my $col = Gtk2::TreeViewColumn->new;
  my $renderer = Gtk2::CellRendererText->new;
  $col->pack_start($renderer, FALSE);
  $col->add_attribute($renderer, text => LABEL);
  $treeView->append_column($col);

  # Attach a signal handler for double clicks on row
  $treeView->signal_connect(
    'row-activated' => sub {
      my ($store, $iter) = $_[0]->get_selection->get_selected;
      editElement($store, $iter) if $store->iter_parent($iter);
    });

  return $treeView;
}

sub addMenuElement {
  my ($store, $parent) = @_;

  # Dialog
  my $dialog = Gtk2::Dialog->new_with_buttons(
    $UIMsg->Render('Select Type'),
    undef, [],
    'gtk-ok' => 'ok',
    'gtk-cancel' => 'cancel');
  $dialog->set_position('center');

  # Radio buttons
  # Speaking of radio: http://wdiy.org -- do give it a look!
  my $selected = 'Item';

  # Folder
  my $radio = Gtk2::RadioButton->new(undef, $CommonMsg->Render('Folder'));
  $radio->signal_connect(toggled => sub { $selected = 'folder' });
  $dialog->vbox->pack_start($radio, TRUE, TRUE, PAD);
  # Item (default)
  $radio = Gtk2::RadioButton->new($radio, $CommonMsg->Render('Item'));
  $radio->signal_connect(toggled => sub { $selected = 'item' });
  $radio->set_active(TRUE);
  $dialog->vbox->pack_start($radio, TRUE, TRUE, PAD);

  # Expander with detailed description
  my $shortDescription = Gtk2::Expander->new(
    $UIMsg->Render('Select Type Short Description'));
  my $longDescription = Gtk2::Label->new(
    $UIMsg->Render('Select Type Long Description'));
  $longDescription->set_line_wrap(TRUE);
  $shortDescription->add($longDescription);
  $dialog->vbox->pack_start($shortDescription, FALSE, FALSE, PAD);

  $dialog->show_all;
  my $response = $dialog->run;
  $dialog->destroy;
  
  return unless $response eq 'ok';

  # Otherwise, call a function according to the selected radio button
  my $newElement = $store->append($parent);
  # Fill in some blank 'default' values
  $store->set($newElement, LABEL, '');
  if ($selected eq 'folder') { 
    $store->set($newElement, DATA, { icon => '' }); 
  } elsif ($selected eq 'item') {
    $store->set($newElement, DATA, { path => '', icon => '' });
  }
  
  # Remove the new iter unless the user didn't decide to cancel
  $store->remove($newElement) unless editElement($store, $newElement);
  
  return;
}

sub getElementAttrs {
  my ($store, $element) = @_;
  my %attrs;
  
  $attrs{label} = $store->get($element, LABEL);

  my $storeAttrs = $store->get($element, DATA);
  # Add attributes from the DATA column of the row if defined
  for my $attr (qw/path icon/) {
    $attrs{$attr} = $storeAttrs->{$attr} if defined $storeAttrs->{$attr};
  }
  
  return %attrs;
}

sub makeAttrList {
  my %attrs = @_; my $iter;
  my $attrList = Gtk2::ListStore->new(qw/Glib::String/ x 3);

  for my $attr (qw/label path icon/) {
    # If the folder or item has this attr...
    if (defined $attrs{$attr}) {
      $iter = $attrList->append;
      # Name of attribute, translated
      $attrList->set($iter, KEY, $CommonMsg->Render(ucfirst $attr));
      # Used for ID
      $attrList->set($iter, KEY_REAL, $attr);
      # Value
      $attrList->set($iter, VALUE, $attrs{$attr});
    }
  }

  return $attrList;
}

sub makeAttrListView {
  my (%attrs) = @_;
  
  my $store = makeAttrList(%attrs);
  my $view = Gtk2::TreeView->new_with_model($store);
  $view->grab_focus; # Using keyboard shortcuts becomes easier

  my @fields = qw/Attribute Value/;
  for my $field (KEY, VALUE) {
    my $col = Gtk2::TreeViewColumn->new;
    # Set 'Attribute' and 'Value' headers
    $col->set_title($CommonMsg->Render($fields[$field]));
    my $renderer = Gtk2::CellRendererText->new;
    # The 'value' column is editable
    if ($field eq VALUE) {
      $renderer->set(editable => TRUE);
      $renderer->signal_connect(edited => \&cellEdited, $store);
    }
    $col->pack_start($renderer, FALSE);
    $col->add_attribute($renderer, text => $field);
    $view->append_column($col);
  }

  return $view;
}

sub cellEdited {
  my ($pathString, $text, $store) = @_[1..3];

  # Just pass the text through if not blank
  if (length $text) {
    $store->set($store->get_iter_from_string($pathString), VALUE, $text);
  } else {
    # Notify user of empty list element
    my $emptyMessage = Gtk2::MessageDialog->new(
      undef, [], qw/error ok/,
      $ErrorMsg->Render('Null Attribute String'));
    $emptyMessage->set_position('center');
    $emptyMessage->run;
    $emptyMessage->destroy;
  }

  return;
}

sub updateIconPreview {
  my ($fileChooser, $preview) = @_;
  
  my $file = $fileChooser->get_preview_filename;
  # For some reason, this callback gets spurious bullshit from time to time
  if (defined $file and -f $file) {
    my $pixbuf = Gtk2::Gdk::Pixbuf->new_from_file_at_size($file, 64, 64);
    $preview->set_from_pixbuf($pixbuf);
    $fileChooser->set_preview_widget_active(defined $pixbuf ? TRUE : FALSE);
  } else {
    $fileChooser->set_preview_widget_active(FALSE);
  }

  return;
}

BEGIN {
  my $wd = rel2abs(curdir);
  
  sub findPathOrIcon {
    my $view = $_[1];
    my ($store, $iter) = $view->get_selection->get_selected;

    # The label attribute has no 'Find' option!!
    return if $store->get($iter, KEY_REAL) eq 'label';

    my $fileChooser = Gtk2::FileChooserDialog->new(
      $UIMsg->Render('Select File'), undef,
      'open',
      'gtk-ok' => 'ok',
      'gtk-cancel' => 'cancel');
    $fileChooser->set_current_folder($wd);
    $fileChooser->set_position('center');

    if ($store->get($iter, KEY_REAL) eq 'icon') {
      # Set filter on file chooser as well
      my $filter = Gtk2::FileFilter->new; $filter->add_pixbuf_formats;
      $fileChooser->set_filter($filter);
      # Icons will have preview
      my $preview = Gtk2::Image->new;
      $fileChooser->set_preview_widget($preview);
      $fileChooser->signal_connect('update-preview' => \&updateIconPreview,
                                   $preview);
    }

    $fileChooser->show_all;
    my $response = $fileChooser->run;
    if ($response eq 'ok') {
      my $file = $fileChooser->get_filename;
      $wd = (splitpath($file))[1]; # Store new current directory
      $store->set($iter, VALUE, $file); # Set element in store
    }

    $fileChooser->destroy;

    return;
  }
}

sub editElement {
  my ($store, $element) = @_;

  # Dialog
  my $dialog = Gtk2::Dialog->new_with_buttons(
    $UIMsg->Render('Edit Properties'),
    undef, [],
    'gtk-ok' => 'ok',
    'gtk-cancel' => 'cancel');
  $dialog->set_position('center');

  # Attributes list
  # Change this?
  my $attrList = makeAttrListView(getElementAttrs($store, $element));
  $dialog->vbox->pack_start($attrList, TRUE, TRUE, PAD);

  # Operations (actually, only one)
  my $operationHButtonBox = Gtk2::HButtonBox->new;
  $operationHButtonBox->set_layout('spread');
  my @operations = ( [ 'gtk-find', \&findPathOrIcon ] );
  # Add them, well...it...one by one
  for my $operation (@operations) {
    my $button = Gtk2::Button->new_from_stock($operation->[0]);
    $button->signal_connect(clicked => $operation->[1], $attrList);
    $operationHButtonBox->pack_start($button, FALSE, FALSE, PAD);
  }
  $dialog->vbox->pack_start($operationHButtonBox, FALSE, FALSE, PAD);

  # Self-documentation
  my $shortDescription = Gtk2::Expander->new(
    $UIMsg->Render('Edit Properties Short Description'));
  my $longDescription = Gtk2::Label->new(
    $UIMsg->Render('Edit Properties Long Description'));
  $longDescription->set_line_wrap(TRUE);
  $shortDescription->add($longDescription);
  $dialog->vbox->pack_start($shortDescription, FALSE, FALSE, PAD);

  $dialog->show_all;
  
  my $response;
  while (1) {
    $response = $dialog->run; # Do it Jah!
    # last unless $response eq 'ok';
    # Do some sanity checks
    last if elementUnique($store, $element) and elementFull($store, $element);
  }
  $dialog->destroy;

  if ($response eq 'ok') {
    # Save attrs (if operation not cancelled)
  }
  return ($response eq 'ok') ? 1 : 0;
}

sub addMenu {
  my ($store, $parent) = $_[1]->get_selection->get_selected;

  # Check for item
  if ($store->get($parent, DATA)->{path}) {
    my $errorMessage = Gtk2::MessageDialog->new(
      undef, [], qw/error ok/,
      $ErrorMsg->Render('Attempted Addition to Item'));
    $errorMessage->set_position('center');
    $errorMessage->run;
    $errorMessage->destroy;
  } else { # If it is a folder...
    addMenuElement($store, $parent);
  }

  return;
}

sub removeMenu {
  my $view = $_[1];
  my $store = $view->get_model;
  my $iter = $view->get_selection->get_selected;

  # Don't delete r00t!!!
  return unless $store->iter_parent($iter);

  # Confirm the deletion, if the iter has children
  if ($store->iter_children($iter)) {
    my $confirmMessage = Gtk2::MessageDialog->new(
      undef, [], qw/question yes-no/,
      $UIMsg->Render('Confirm Deletion'));
    $confirmMessage->set_position('center');

    # Get a reply
    $confirmMessage->show_all;
    my $response = $confirmMessage->run;
    $confirmMessage->destroy;
    
    return unless $response eq 'yes';
  }
  
  $store->remove($iter);
  return;
}

sub upMenu {
  my $view = $_[1];
  my $store = $view->get_model;
  my $iter = $view->get_selection->get_selected;
  my $prev = eval {
    my $path = $store->get_path($iter);
    return $store->get_iter($path) if $path->prev;
  };
  $store->swap($prev, $iter) if $prev;
  
  return;
}

sub downMenu {
  my $view = $_[1];
  my $store = $view->get_model;
  my $iter = $view->get_selection->get_selected;
  my $next = eval {
    my $path = $store->get_path($iter); $path->next;
    return $store->get_iter($path);
  };
  $store->swap($iter, $next) if $next;
  
  return;
}

sub MainMenu {
  my $dialog = Gtk2::Dialog->new_with_buttons(
    $UIMsg->Render('Title'),
    undef, [],
    'gtk-save' => 'ok',
    'gtk-cancel' => 'cancel');
  $dialog->set_position('center');
  $dialog->set_default_size(400, 300);

  ## Action area
  # TreeView of menu -- but, first, get the menu definition
  my $menu = findMenu();
  my $menuTree = makeMenuTreeView($menu, $dialog);
  
  # Pack the tree view into a scrolled window. The addition of scrollbars
  # will be done automatically in either direction
  my $scrolledTree = Gtk2::ScrolledWindow->new;
  $scrolledTree->set_policy('automatic', 'automatic');
  $scrolledTree->add($menuTree);
  $dialog->vbox->pack_start($scrolledTree, TRUE, TRUE, PAD);
  
  # Operations
  my $operationHButtonBox = Gtk2::HButtonBox->new;
  $operationHButtonBox->set_layout('spread');
  my @operations = ( [ 'gtk-add' => \&addMenu ],
                     [ 'gtk-remove' => \&removeMenu ],
                     [ 'gtk-go-up' => \&upMenu ],
                     [ 'gtk-go-down' => \&downMenu ] );
  # Add them, one by one
  for my $operation (@operations) {
    my $button = Gtk2::Button->new_from_stock($operation->[0]);
    $button->signal_connect(clicked => $operation->[1], $menuTree);
    $operationHButtonBox->pack_start($button, TRUE, TRUE, PAD);
  }
  $dialog->vbox->pack_start($operationHButtonBox, FALSE, FALSE, PAD);

  # Expander with detailed description
  my $shortDescription = Gtk2::Expander->new(
    $UIMsg->Render('Main Menu Short Description'));
  my $longDescription = Gtk2::Label->new(
    $UIMsg->Render('Main Menu Long Description'));
  $longDescription->set_line_wrap(TRUE);
  $shortDescription->add($longDescription);
  $dialog->vbox->pack_start($shortDescription, FALSE, FALSE, 0);

  $dialog->show_all;
  my $response = $dialog->run;
  # Do something with the response here:
  # cancel or delete-event
  # save
  $dialog->destroy;

  exit 0;
}

MainMenu();
