#!/usr/bin/perl
# -*- project-name: VASM -*-

use strict;
use warnings;
use constant { FALSE => 0, TRUE => 1 };
use constant { DATA => 0, LABEL => 1 };
use constant { FOLDER => 0, ITEM => 1 };
use VASM::Resource::Message;
use VASM::Resource::Menu;
use Gtk2 '-init';

# Message files
my $CommonMsg = findMessageCatalog(qw/Common Message/);
my $UIMsg = findMessageCatalog(qw/Menu UI Gtk2 Message/);
my $ErrorMsg = findErrorCatalog(qw/Menu UI Error/);

# Add 'Edit' button along with others

sub makeMenuTree {
  my ($menu) = @_;

  # Set up a tree store corresponding to the menu; the single Glib::String
  # column indicates a particular folder or item name in the menu object
  my $menuTree = Gtk2::TreeStore->new('Glib::Scalar', 'Glib::String');

  # Add a top level imaginary root
  my $root = $menuTree->append(undef);
  $menuTree->set($root, LABEL, $UIMsg->Render('Window Manager Menu'));
  $menuTree->set($root, DATA, {});
  
  # makeMenuTreeHeavy needs to be factored out, as it is a /recursive/
  # function that will fill in the tree store itself
  makeMenuTreeHeavy($menu, $menuTree, $root);

  return $menuTree;
}

sub makeMenuTreeHeavy {
  my ($menu, $menuTree, @iters) = @_;
  
  # Get the current 'stack' of folders
  my @path = map { $menuTree->get($_, LABEL) } @iters[1..$#iters];

  # Create more folders under it
  for my $child ($menu->Children(@path)) {
    if ($menu->Children(@path, $child)) {
      push @iters, $menuTree->append($iters[-1] or undef);
      # Set the folder name at the current path
      $menuTree->set($iters[-1], LABEL, $child);
      $menuTree->set($iters[-1], DATA, $menu->Retrieve(@path, $child));
      makeMenuTreeHeavy($menu, $menuTree, @iters); # Recurse
      pop @iters; # Move back up a level
    }
  }

  # Now, introduce the items after the folders
  for my $child ($menu->Children(@path)) {
    # Only menu items should have the path property
    if ($menu->Retrieve(@path, $child)->{path}) {
      # Advance the pointer at the current level
      my $iter = $menuTree->append($iters[-1] or undef);
      # Set the item name and properties at the current path
      $menuTree->set($iter, DATA, $menu->Retrieve(@path, $child));
      $menuTree->set($iter, LABEL, $child);
    }
  }
}

sub makeMenuTreeView {
  my ($menu) = @_;

  my $treeStore = makeMenuTree($menu); # The actual data
  # The rest of this is a 'cookbook' recipe to make a tree view correspond
  # one-for-one with the data in the tree store.
  my $treeView = Gtk2::TreeView->new_with_model($treeStore);
  $treeView->set_headers_visible(FALSE);
  $treeView->set_reorderable(TRUE);
  my $col = Gtk2::TreeViewColumn->new;
  my $renderer = Gtk2::CellRendererText->new;
  $col->pack_start($renderer, FALSE);
  $col->add_attribute($renderer, text => LABEL);
  $treeView->append_column($col);

  # Attach a signal handler for double clicks on row
  #$treeView->signal_connect('row-activated' => \&menuTreeRowActivated,
  #                          $dialog);
  
  return $treeView;
}

sub addMenuFolderOrItem {
  my ($store, $parent) = @_;

  # Dialog
  my $dialog = Gtk2::Dialog->new_with_buttons(
    $UIMsg->Render('Select Type'),
    undef, [],
    'gtk-ok' => 'ok',
    'gtk-cancel' => 'cancel');
  $dialog->set_position('center');

  # Radio buttons
  # Speaking of radio: http://wdiy.org -- do give it a look!
  my $selected = 'Item';

  # Folder
  my $radio = Gtk2::RadioButton->new(undef, $CommonMsg->Render('Folder'));
  $radio->signal_connect(toggled => sub { $selected = 'folder' });
  $dialog->vbox->pack_start($radio, TRUE, TRUE, 0);
  # Item (default)
  $radio = Gtk2::RadioButton->new($radio, $CommonMsg->Render('Item'));
  $radio->signal_connect(toggled => sub { $selected = 'item' });
  $radio->set_active(TRUE);
  $dialog->vbox->pack_start($radio, TRUE, TRUE, 0);

  # Separator
  $dialog->vbox->pack_start(Gtk2::HSeparator->new(), FALSE, FALSE, 0);

  # Expander with detailed description
  my $shortDescription = Gtk2::Expander->new(
    $UIMsg->Render('Select Type Short Description'));
  my $longDescription = Gtk2::Label->new(
    $UIMsg->Render('Select Type Long Description'));
  $longDescription->set_line_wrap(TRUE);
  $shortDescription->add($longDescription);
  $dialog->vbox->pack_start($shortDescription, FALSE, FALSE, 0);

  $dialog->show_all;
  my $response = $dialog->run;
  $dialog->destroy;
  
  return unless $response eq 'ok';

  # Otherwise, call a function according to the selected radio button
  if ($selected eq 'folder') { addMenuFolder($store, $parent) }
  elsif ($selected eq 'item') { addMenuItem($store, $parent) }
  
  return;
}

sub editElement {
  my ($folder, $type) = @_;

  # Ask for folder name
  my $dialog = Gtk2::Dialog->new_with_buttons(
    $UIMsg->Render('Edit Properties'),
    undef, [],
    'gtk-ok' => 'ok',
    'gtk-cancel' => 'cancel');
  $dialog->set_position('center');

  ## Action area
  my ($editHButtonBox, %entries);
  # Name Entry
  $editHButtonBox = Gtk2::HButtonBox->new;
  $entries{name}->{label} = Gtk2::Label->new($CommonMsg->Render('Name'));
  $entries{name}->{entry} = Gtk2::Entry->new;
  $editHButtonBox->pack_start($entries{name}->{label}, FALSE, FALSE, 0);
  $editHButtonBox->pack_start($entries{name}->{entry}, FALSE, FALSE, 0);
  $dialog->vbox->pack_start($editHButtonBox, TRUE, TRUE, 0);
  # Path Entry
  if ($type eq ITEM) {
    $editHButtonBox = Gtk2::HButtonBox->new;
    $entries{path}->{label} = Gtk2::Button->new($CommonMsg->Render('Path'));
    $entries{path}->{entry} = Gtk2::Entry->new;
    # Bind action to path -- get/set entry value
    $editHButtonBox->pack_start($entries{path}->{label}, FALSE, FALSE, 0);
    $editHButtonBox->pack_start($entries{path}->{entry}, FALSE, FALSE, 0);
    $dialog->vbox->pack_start($editHButtonBox, TRUE, TRUE, 0);
  }
  # Icon Entry
  $editHButtonBox = Gtk2::HButtonBox->new;
  $entries{icon}->{label} = Gtk2::Button->new($CommonMsg->Render('Icon'));
  $entries{icon}->{entry} = Gtk2::Entry->new;
  # Bind action to icon
  $editHButtonBox->pack_start($entries{icon}->{label}, FALSE, FALSE, 0);
  $editHButtonBox->pack_start($entries{icon}->{entry}, FALSE, FALSE, 0);
  $dialog->vbox->pack_start($editHButtonBox, TRUE, TRUE, 0);
  # Description Entry
  if ($type eq ITEM) {
    $editHButtonBox = Gtk2::HButtonBox->new;
    $entries{description}->{label} = 
      Gtk2::label->new($CommonMsg->Render('Description'));
    $entries{description}->{entry} = Gtk2::Entry->new;
    $editHButtonBox->pack_start($entries{description}->{label}, 
                                FALSE, FALSE, 0);
    $editHButtonBox->pack_start($entries{description}->{entry}, 
                                FALSE, FALSE, 0);
    $dialog->vbox->pack_start($editHButtonBox, TRUE, TRUE, 0);
  }
  # Self-Documentation
  
  $dialog->show_all;

  while (1) {
    # Run the dialog, check for cancel...otherwise, set attributes
    $dialog->run;
    $dialog->destroy;

    # Checking
    #   - unique at this level
    #   - names not blank
    # 'last' if all true
    last;
  }
  
  return;
}

sub addMenuItem {
  my ($parent) = @_;
}

sub addMenuFolder {
  my ($store, $parent) = @_;
  
  # Add a new folder beneath the parent
  my $newFolder = $store->append($parent);

  # Edit it
  editElement($newFolder, FOLDER);

  return;
}
  
sub addMenu {
  my $view = $_[1];
  my $store = $view->get_model;
  my $parent = $view->get_selection->get_selected;

  # Check for item
  if ($store->get($parent, DATA)->{path}) {
    my $errorMessage = Gtk2::MessageDialog->new(
      undef, [], qw/error ok/,
      $ErrorMsg->Render('Attempted Addition to Item'));
    $errorMessage->set_position('center');
    $errorMessage->run;
    $errorMessage->destroy;
  } else { # If it is a folder...
    addMenuFolderOrItem($store, $parent);
  }

  return;
}

sub removeMenu {
  my $view = $_[1];
  my $store = $view->get_model;
  my $iter = $view->get_selection->get_selected;

  # Don't delete r00t!!!
  return unless $store->iter_parent($iter);

  # Confirm the deletion, if the iter has children
  if ($store->iter_children($iter)) {
    my $confirmMessage = Gtk2::MessageDialog->new(
      undef, [], qw/question yes-no/,
      $UIMsg->Render('Confirm Deletion'));
    $confirmMessage->set_position('center');

    # Get a reply
    $confirmMessage->show_all;
    my $response = $confirmMessage->run;
    $confirmMessage->destroy;
    
    return unless $response eq 'yes';
  }
  
  $store->remove($iter);
  return;
}

sub upMenu {
  my $view = $_[1];
  my $store = $view->get_model;
  my $iter = $view->get_selection->get_selected;
  my $prev = eval {
    my $path = $store->get_path($iter);
    return $store->get_iter($path) if $path->prev;
  };
  $store->swap($prev, $iter) if $prev;
  
  return;
}

sub downMenu {
  my $view = $_[1];
  my $store = $view->get_model;
  my $iter = $view->get_selection->get_selected;
  my $next = eval {
    my $path = $store->get_path($iter); $path->next;
    return $store->get_iter($path);
  };
  $store->swap($iter, $next) if $next;
  
  return;
}

sub MainMenu {
  my $dialog = Gtk2::Dialog->new_with_buttons(
    $UIMsg->Render('Title'),
    undef, [],
    'gtk-save' => 'ok',
    'gtk-cancel' => 'cancel');
  $dialog->set_position('center');
  $dialog->set_default_size(400, 300);

  ## Action area
  # TreeView of menu -- but, first, get the menu definition
  my $menu = findMenu();
  my $menuTree = makeMenuTreeView($menu);
  
  # Pack the tree view into a scrolled window. The addition of scrollbars
  # will be done automatically in either direction
  my $scrolledTree = Gtk2::ScrolledWindow->new;
  $scrolledTree->set_policy('automatic', 'automatic');
  $scrolledTree->add($menuTree);
  $dialog->vbox->pack_start($scrolledTree, TRUE, TRUE, 0);
  
  # Separator
  $dialog->vbox->pack_start(Gtk2::HSeparator->new(), FALSE, FALSE, 0);
  
  # Operations
  my $operationHButtonBox = Gtk2::HButtonBox->new;
  $operationHButtonBox->set_layout('spread');
  my @operations = ( [ 'gtk-add' => \&addMenu ],
                     [ 'gtk-remove' => \&removeMenu ],
                     [ 'gtk-go-up' => \&upMenu ],
                     [ 'gtk-go-down' => \&downMenu ] );
  # Add them, one by one
  for my $operation (@operations) {
    my $button = Gtk2::Button->new_from_stock($operation->[0]);
    $button->signal_connect(clicked => $operation->[1], $menuTree);
    $operationHButtonBox->pack_start($button, TRUE, TRUE, 0);
  }
  $dialog->vbox->pack_start($operationHButtonBox, FALSE, FALSE, 0);

  # Separator
  $dialog->vbox->pack_start(Gtk2::HSeparator->new(), FALSE, FALSE, 0);

  # Expander with detailed description
  my $shortDescription = Gtk2::Expander->new(
    $UIMsg->Render('Main Menu Short Description'));
  my $longDescription = Gtk2::Label->new(
    $UIMsg->Render('Main Menu Long Description'));
  $longDescription->set_line_wrap(TRUE);
  $shortDescription->add($longDescription);
  $dialog->vbox->pack_start($shortDescription, FALSE, FALSE, 0);

  $dialog->show_all;
  my $response = $dialog->run;
  # Do something with the response here:
  # cancel or delete-event
  # save
  $dialog->destroy;

  exit 0;
}

MainMenu();
