# -*- project-name: VASM -*-
package VASM::Resource;

use strict;
use warnings;
use base qw/Exporter/;
use Carp;
use IO::Handle;
use File::Path;
use File::BaseDir qw/xdg_config_files
                     xdg_data_files
                     xdg_config_home
                     xdg_data_home/;
our @EXPORT = qw/findConfigResource
                 findDataResource
                 makeConfigResourcePath
                 makeDataResourcePath
                 makeConfigResource
                 makeDataResource/;
our $VERSION = '1.08';

sub findResource {
  my %args = @_;

  croak 'All arguments must be given to findResource!'
    unless (grep { defined $args{$_} } qw/Type Path/) == 2;

  # Select a function according to the mode of retrieval:
  my $closure;
  if ($args{Type} eq 'data') { $closure = \&xdg_data_files }
  elsif ($args{Type} eq 'config') { $closure = \&xdg_config_files }

  my @results = $closure->(join('/', 'vasm', @{ $args{Path} }));
  @results and return @results;

  return; # Fall-through
}

sub findConfigResource {
  return findResource(Type => 'config', Path => \@_);
}

sub findDataResource {
  return findResource(Type => 'data', Path => \@_);
}

sub makeResourcePath {
  my (%args) = @_; my $closure;

  croak 'All arguments must be given to makeResourcePath!'
    unless (grep { defined $args{$_} } qw/Type Path/) == 2;

  # Select a function according to mode of internment:
  if ($args{Type} eq 'data') { $closure = \&xdg_data_home }
  elsif ($args{Type} eq 'config') { $closure = \&xdg_config_home }

  # Create the full pathname then
  my $path = join('/', $closure->(), 'vasm', @{ $args{Path} });

  # Make sure all the required directories exist for a given resource; if not,
  # attempt to create them with mode 0700 (as per the XDG BaseDir spec). If
  # this fails, mkpath will die. I think we should allow it.
  mkpath($path, undef, 0700) unless -d $path;

  return $path;
}

sub makeDataResourcePath {
  return makeResourcePath(Type => 'data', Path => \@_);
}

sub makeConfigResourcePath {
  return makeResourcePath(Type => 'config', Path => \@_);
}

# Returns a filehandle with the UTF-8 IO layer, representing the file at the
# given resource path. Of course, this file is always in either the config or
# data user-specific resource tree.
sub makeResource {
  my %args = @_;

  # Check required args
  croak 'All arguments must be given to makeResource!'
    unless (grep { defined $args{$_} } qw/Type Path/) == 2;
  
  my $file = pop @{ $args{Path} }; # Remove the file from $args{Path}
  my $path = makeResourcePath(Type => $args{Type}, # Find the needed path
                              Path => $args{Path});
  
  # Open the file with the UTF-8 IO layer. Here is an unfortunate blemish in
  # my design...there is really no clean way to i18nify (or whatever the verb
  # is) a failure in the operation, so failure will croak in English.
  # Hopefully, this subroutine acts in a fairly limited context, so we
  # shouldn't have to worry too much.
  my $fullPath = join('/', $path, $file);
  open my $fh, '>:utf8', $fullPath or croak $fullPath, ": $!";

  # If the above succeeds, this should succeed; all it is is OO abstraction
  # hocus pocus for a Real Man's Filehandle.
  my $handle = IO::Handle->new_from_fd(fileno($fh), 'w');

  return $handle; # /Successful/ fall-through; return the filehandle
}

sub makeConfigResource {
  return makeResource(Type => 'config', Path => \@_);
}

sub makeDataResource {
  return makeResource(Type => 'data', Path => \@_);
}

1;

__END__

=head1 NAME

VASM::Resource - resource location facility compliant with XDG Base Directory
specification

=head1 SYNOPSIS

    use VASM::Resource; 

    # Look for:
    # 
    # ~/.config/vasm/Foo/Bar/Baz/config.xml
    # /etc/xdg/vasm/Foo/Bar/Baz/config.xml
    #
    # ...in that order.
    my $configResource =
      findConfigResource(qw/Foo Bar Baz config.xml/);
    
    # You get the idea...
    my $dataResource =
      findDataResource(qw/Some Other Stuff foo.png/);

=head1 DESCRIPTION

VASM::Resource provides for location of resource files in directories
containing hierarchies of resources; these collections are searched in order,
and matches which most closely match the given request for a resource, found
in the user-specific hierarchy, are preferred over more general system default
matches. If no matches could be found, an appropriate error message will be
printed, albeit limited to English. In the interest of alliance with standards
(and enlightened laziness) VASM::Resource complies with the XDG Base Directory
specification.

=head1 FUNCTIONS

=over

=item findConfigResource

findConfigResource accepts a plain list as an argument, indicating a directory
pathway and file, with which it locates a given configuration resource file as
well as it can. VASM::Resource will conduct a search according to this
formula: XDG config dirs + 'vasm' + path. In other words, findConfigResource
will look under each XDG configuration directory, under the prefix directory
'vasm', for the given path and file. If unsuccessful, findConfigResource will
give up and return nothing.

=item findDataResource

...is the same as the above, except for data resources instead of
configuration resources.

=back

=head1 VARIABLES

=over

=item XDG_CONFIG_HOME

Directory relative to which user-specific configuration files should be
written; defaults to $HOME/.config.

=item XDG_DATA_HOME

Directory relative to which user-specific data should be written; defaults to
'$HOME/.local/share'.

=item XDG_CONFIG_DIRS

Colon-delimited, preference-ordered list of directores relative to which
configuration files should be searched after $XDG_CONFIG_HOME; defaults to
'/etc/xdg'.

=item XDG_DATA_DIRS

Colon-delimited, preference-ordered list of directores relative to which
configuration files should be searched after $XDG_DATA_HOME; defaults to
'/usr/local/share:/usr/share'.

=back

=head1 AUTHORS

hanumizzle L<mailto:hanumizzle@gmail.com> wrote VASM::Resource.

=head1 SEE ALSO

=over

=item *

XDG Base Directory specification:
L<http://standards.freedesktop.org/basedir-spec/>, implemented by
L<File::Basedir>.

=back

=cut
