# -*- project-name: VASM -*-
package VASM::Menu;

use strict;
use warnings;
use base qw/VASM::Tree VASM::Menu::Translators/;
use Carp;
use XML::Parser;
use XML::Writer;

our $VERSION = '1.02';

sub Store {
  my ($self, $attrs, @path) = (shift, pop, @_);

  # At the very least, there must be a @path with one element and associated
  # properties
  croak 'Attributes hashref invalid!' unless @path and defined %$attrs;

  # Check contents thereof?

  # Notice that $attrs is copied when interned
  $self->SUPER::Store(@path, { %$attrs });
  
  return;
}

sub Retrieve {
  my ($self, @path) = @_;
  
  croak 'Path argument must be given to Retrieve!' unless @path;
  my $attrs = $self->SUPER::Retrieve(@path);
  # Here again, we copy the hash so that it may not be modified
  return { %$attrs } if defined $attrs;

  return;
}

sub Parse {
  my ($self, $handle) = @_;
  # See Parse in VASM::Message for lament
  my $parser = XML::Parser->new(Style => 'Tree', ErrorContext => 3);
  my $tree = $parser->parse($handle);

  # This absolutely /has to/ recurse, and we obviously can't have multiple
  # instances of XML parsers running around. (I could check the information
  # from the caller built-in and special case it, but dollars to doughnuts
  # says I get reborn as Brian Peppers if God exists.)
  $self->parseHeavy($tree->[1]);
}

sub parseHeavyMenu {
  my ($self, $tags, @path) = (shift, pop, @_);
  my $attrs = $tags->[0];

  # Label attribute must be defined
  croak 'Label attribute undefined in menu tag!' unless defined $attrs->{label};
  # Check for an existing element at this path
  croak 'Two or more elements at the same name and level!'
    if defined $self->Retrieve(@path, $attrs->{label});
  # Store the folder
  croak 'Icon attribute undefined in menu tag!'
    unless defined $attrs->{icon};
  $self->Store(@path, $attrs->{label}, { icon => $attrs->{icon} });
  # Recurse
  $self->parseHeavy($tags, @path, $attrs->{label});
}

sub parseHeavyItem {
  my ($self, $tags, @path) = (shift, pop, @_);
  my $attrs = $tags->[0];
  
  # Label attribute must be defined
  croak 'Label attribute undefined in item tag!'
    unless defined $attrs->{label};
  # Check for an existing element at this path
  croak 'Two or more elements at the same name and level!'
    if defined $self->Retrieve(@path, $attrs->{label});

  # Collect properties from under the <item> tag
  my $propHash = {};
  for my $index (grep { $_ % 2 } (0..$#{ $tags })) {
    next if $tags->[$index] eq '0';

    # Add the attribute to the hash of item attributes if valid
    if (grep { $_ eq $tags->[$index] } qw/path icon description/) {
      croak "Item property $tags->[$index] undefined!"
        unless defined $tags->[$index + 1]->[2];
      $propHash->{$tags->[$index]} = $tags->[$index + 1]->[2];
    }
  }

  # Intern it unless $propsHash lacks important fields
  croak 'Item properties hash incomplete!' unless keys %$propHash == 3;
  $self->Store(@path, $tags->[0]->{label}, $propHash);
}

sub parseHeavy {
  # $tags is the chunk of parsed XML under examination, and @path is a
  # stack representing the current folder hierarchy
  my ($self, $tags, @path) = @_;

  for my $index (grep { $_ % 2 } (0..$#{ $tags })) {
    next if $tags->[$index] eq '0'; # Skip text elements

    if ($tags->[$index] eq 'menu') {
      $self->parseHeavyMenu(@path, $tags->[$index + 1]);
    } elsif ($tags->[$index] eq 'item') {
      $self->parseHeavyItem(@path, $tags->[$index + 1]);
    }
  }

  return 1; # Success!
}

sub Write {
  my ($self, $handle) = @_;

  croak 'Handle argument must be given to Write!' unless defined $handle;

  # Now create an XML::Writer instance and turn that puppy out
  my $xml = XML::Writer->new(OUTPUT => $handle, ENCODING => 'utf-8',
                             DATA_MODE => 1, DATA_INDENT => 2);

  # Open up the document
  $xml->xmlDecl;
  $xml->startTag('menu');
  $xml->comment('Generated by VASM');

  $self->writeHeavy($xml); # Now let writeHeavy do its stuff

  $xml->endTag('menu'); $xml->end; # End the document...

  return 1; # Success!
}

sub writeHeavy {
  my ($self, $xml, @path) = @_;

  for my $child ($self->Children(@path)) {
    # Recurse if the child in question has children of its own, in between
    # producing the appropriate opening and closing tags
    if ($self->Children(@path, $child)) {
      $xml->startTag('menu',
                     label => $child,
                     icon => $self->Retrieve(@path, $child)->{icon});
      $self->writeHeavy($xml, @path, $child);
      $xml->endTag('menu');
    }

    # Produce 'item' stanzas where items are defined
    if (my $attrs = $self->Retrieve(@path, $child)) {
      $xml->startTag('item', label => $child);
      for my $key (qw/path icon description/) {
        $xml->dataElement($key, $attrs->{$key});
      }
      $xml->endTag('item');
    }
  }
}

1;

__END__

=head1 NAME

VASM::Menu - general window manager menu definitions

=head1 SYNOPSIS

=head1 DESCRIPTION

=head1 METHODS

=head1 AUTHORS

hanumizzle L<hanumizzle@gmail.com> wrote VASM::Menu.

=cut

