#!/usr/bin/perl
# -*- project-name: VASM -*-

use strict;
use warnings;
use constant { FALSE => 0, TRUE => 1 };
use constant { LABEL => 0, LABEL_REAL => 0, LABEL_SHOWN => 1 };
use VASM::Tree;
use VASM::Resource;
use VASM::Resource::Message;
use VASM::Resource::Action;
use Gtk2 '-init';
use File::MimeInfo::Magic;
use Archive::Tar;

# Message files
my $CommonMsg = findMessageCatalog(qw/Common Message/);
my $UIMsg = findMessageCatalog(qw/Action UI Gtk2 Message/);
my $ErrorMsg = findErrorCatalog(qw/Action UI Error/);

sub makeActionTree {
  # Get directories for file and URI schemata
  my $fileDirectory = mediaDirectory('File');
  my $uriDirectory = mediaDirectory('URI');

  # Convert the file directory to data more readily usable by the function
  my $fileTypeTree = VASM::Tree->new;
  for my $mime ($fileDirectory->List) {
    $fileTypeTree->Store((split m!/!, $mime), 1);
  }

  # Now, set up a tree store corresponding to them and add the 'Template'
  # top-level choice as well. The first Glib::String is the item as it appears
  # on the screen, according to translation and additional notes. The second
  # is the 'real name', usable to the program.
  my $actionTree = Gtk2::TreeStore->new('Glib::String', 'Glib::String');
  my @iters;

  # File
  push @iters, $actionTree->append(undef);
  $actionTree->set($iters[-1],
                   LABEL_REAL, 'File',
                   LABEL_SHOWN, $UIMsg->Render('File')
                 );
  for my $class (sort $fileTypeTree->Children) {
    # Add the class under 'File'
    push @iters, $actionTree->append($iters[-1]);
    $actionTree->set($iters[-1],
                     LABEL_REAL, $class,
                     LABEL_SHOWN, $class);
    # Add each member of the class (e.g., File -> image -> jpeg)
    for my $member (sort $fileTypeTree->Children($class)) {
      push @iters, $actionTree->append($iters[-1]);
      my $mime = join('/', $class, $member);
      # The type will appear with a parenthetical note describing it in
      # further detail
      $actionTree->set(
        $iters[-1],
        LABEL_REAL, $member,
        LABEL_SHOWN, $member . ' (' . $fileDirectory->Render($mime) . ')',
      );
    } continue { pop @iters }
  } continue { pop @iters }
  pop @iters;

  # URI
  push @iters, $actionTree->append(undef);
  $actionTree->set($iters[-1],
                   LABEL_REAL, 'URI',
                   LABEL_SHOWN, $UIMsg->Render('URI'));
  for my $scheme (sort $uriDirectory->List) {
    push @iters, $actionTree->append($iters[-1]);
    $actionTree->set(
      $iters[-1],
      LABEL_REAL, $scheme,
      LABEL_SHOWN, $scheme . ' (' . $uriDirectory->Render($scheme) . ')');
  } continue { pop @iters }
  pop @iters;

  # Template
  push @iters, $actionTree->append(undef);
  $actionTree->set($iters[-1],
                   LABEL_REAL, 'Template',
                   LABEL_SHOWN, $UIMsg->Render('Template'));
  pop @iters; # Neat freak

  return $actionTree;
}

sub makeActionTreeView {
  my $treeStore = makeActionTree(); # The actual data
  # The rest of this is a 'cookbook' recipe to make a tree view correspond
  # one-for-one with the data in the tree store.
  my $treeView = Gtk2::TreeView->new_with_model($treeStore);
  $treeView->set_headers_visible(FALSE);
  my $col = Gtk2::TreeViewColumn->new;
  my $renderer = Gtk2::CellRendererText->new;
  $col->pack_start($renderer, FALSE);
  $col->add_attribute($renderer, text => LABEL_SHOWN);
  $treeView->append_column($col);
  
  # Attach a signal handler for double clicks on rows
  $treeView->signal_connect('row-activated' => \&actionTreeRowActivated);

  return $treeView;
}

sub actionTreeRowActivated {
  my $dialog = $_[3];

#  my @path = identifyTreeRow($store);
#  dispatchMediaType(@path);

  return;
}

sub makeMediaList {
  my ($catalog) = @_; my $iter;
  my $mediaList = Gtk2::ListStore->new('Glib::String');
  my @programs = $catalog->Dump;

  if (@programs) { # Add each program in the catalog to the list store
    for my $program (@programs) {
      $iter = $mediaList->append;
      $mediaList->set($iter, LABEL, $program);
    }
  } else { # Empty or non-existent catalog
    $iter = $mediaList->append;
    $mediaList->set($iter, LABEL, '<edit me>'); # Add a new element
  }
  
  return $mediaList;
}

sub makeMediaListView {
  my ($catalog) = @_;

  # Cookbook recipe that instantiates a list view store from an actions
  # catalog and imposes an editable view component on it
  my $listStore = makeMediaList($catalog);
  my $listView = Gtk2::TreeView->new_with_model($listStore);
  $listView->set_headers_visible(FALSE);
  $listView->set_reorderable(TRUE); # Allow drag and drop :D
  $listView->grab_focus; # Using keyboard shortcuts becomes easier 
  my $col = Gtk2::TreeViewColumn->new;
  my $renderer = Gtk2::CellRendererText->new;
  $renderer->set(editable => TRUE);
  $renderer->signal_connect(edited => \&listCellEdited, $listStore);
  $col->pack_start($renderer, FALSE);
  $col->add_attribute($renderer, text => LABEL);
  $listView->append_column($col);

  return $listView;
}

sub listCellEdited {
  my ($pathString, $text, $store) = @_[1..3];

  # Just pass the text through if not blank
  if (length($text)) {
    $store->set($store->get_iter_from_string($pathString), LABEL, $text);
  } else {
    # Notify user of empty list element
    my $emptyMessage = Gtk2::MessageDialog->new(
      undef, [], qw/error ok/,
      $ErrorMsg->Render('Null Command String'));
    $emptyMessage->set_position('center');
    $emptyMessage->run;
    $emptyMessage->destroy;
  }

  return;
}

sub identifyTreeRow {
  my $view = $_[0]; my @path;
  my $store = $view->get_model;
  my $iter = $view->get_selection->get_selected;

  # Walk from branch to trunk, unshifting the 'real' column at each point. The
  # result will define the path in the tree store.
  do {
    unshift @path, $store->get($iter, LABEL_REAL);
  } while ($iter = $store->iter_parent($iter));

  return @path;
}

sub addProgram {
  my $view = $_[1];
  my $store = $view->get_model;
  my $iter = $store->insert_after(scalar $view->get_selection->get_selected);
  $store->set($iter, LABEL, '<edit me>'); # Add a new element

  return;
}

sub removeProgram {
  my $view = $_[1];
  my $store = $view->get_model;
  my $iter = $view->get_selection->get_selected;
  $store->remove($iter) if $iter;

  return;
}

sub upProgram {
  my $view = $_[1];
  my $store = $view->get_model;
  my $iter = $view->get_selection->get_selected;
  my $prev = eval {
    my $path = $store->get_path($iter);
    return $store->get_iter($path) if $path->prev;
  };
  $store->swap($prev, $iter) if $prev;
  
  return;
}

sub downProgram {
  my $view = $_[1];
  my $store = $view->get_model;
  my $iter = $view->get_selection->get_selected;
  my $next = eval {
    my $path = $store->get_path($iter); $path->next;
    return $store->get_iter($path);
  };
  $store->swap($iter, $next) if $next;
  
  return;
}

sub generateCatalog {
  my ($catalog, $store) = @_; my $iter;

  $catalog->Clear; # Wipe out the catalog first
  # Add an element in the catalog for each member in the list store
  $store->foreach(
    sub { $catalog->Push($_[0]->get($_[2], LABEL)); return FALSE }
  );
  
  return;
}

sub dispatchMediaType {
  my @path = @_;

  # Fully qualified path was given
  if (($path[0] eq 'File' and @path == 3) or
      ($path[0] eq 'URI' and @path == 2)) {
    MediumConfigure(@path);
  } elsif ($path[0] eq 'Template') { # Select a template
    TemplateSelect();
  }
  
  return;
}

sub MainMenu {
  # Dialog
  my $dialog = Gtk2::Dialog->new_with_buttons(
    $UIMsg->Render('Title'),
    undef, [], 
    'gtk-close' => 'close');
  $dialog->set_position('center');
  $dialog->set_default_size(400, 300);
    
  ## Action area
  # Tree with media
  my $actionTree = makeActionTreeView();

  # Pack the tree view into a scrolled window. The addition of scrollbars
  # will be done automatically in either direction
  my $scrolledTree = Gtk2::ScrolledWindow->new;
  $scrolledTree->set_policy('automatic', 'automatic');
  $scrolledTree->add($actionTree);
  $dialog->vbox->pack_start($scrolledTree, TRUE, TRUE, 0);

  # Separator
  $dialog->vbox->pack_start(Gtk2::HSeparator->new(), FALSE, FALSE, 0);

  # Expander with detailed description
  my $shortDescription = Gtk2::Expander->new(
    $UIMsg->Render('Main Menu Short Description'));
  my $longDescription = Gtk2::Label->new(
    $UIMsg->Render('Main Menu Long Description'));
  $longDescription->set_line_wrap(TRUE);
  $shortDescription->add($longDescription);
  $dialog->vbox->pack_start($shortDescription, FALSE, FALSE, 0);
  $dialog->show_all;

  while (1) {
    my $response = $dialog->run; # Run has its own main loop
    # Terminate here if so desired
    last if $response eq 'close' or $response eq 'delete-event';

    my @path = identifyTreeRow($actionTree); # Grab the selected row
    dispatchMediaType(@path); # Dispatch the selection
  }

  $dialog->destroy; # ...and destroy the dialog
  exit 0;
}

sub MediumConfigure {
  my @path = @_;
  my $mediaType = (@path == 3) ? join '/', @path[1..2] : $path[1];

  # First, load the action catalog, if possible
  my $catalog = findActionCatalog(@path);

  # Dialog
  my $dialog = Gtk2::Dialog->new_with_buttons(
    $mediaType,
    undef, [],
    'gtk-ok' => 'ok',
    'gtk-cancel' => 'cancel');
  $dialog->set_position('center');

  # Media list
  my $mediaList = makeMediaListView($catalog);
  $dialog->vbox->pack_start($mediaList, TRUE, TRUE, 0);

  # HSeparator
  $dialog->vbox->pack_start(Gtk2::HSeparator->new(), FALSE, FALSE, 0);

  # Operations
  my $operationHBox = Gtk2::HBox->new(TRUE, 0);
  my @operations = ( [ 'gtk-add' => \&addProgram ],
                     [ 'gtk-remove' => \&removeProgram ],
                     [ 'gtk-go-up' => \&upProgram ],
                     [ 'gtk-go-down' => \&downProgram ] );
  # Add them, one by one
  for my $operation (@operations) {
    my $button = Gtk2::Button->new_from_stock($operation->[0]);
    $button->signal_connect(clicked => $operation->[1], $mediaList);
    $operationHBox->pack_start($button, TRUE, TRUE, 0);
  }
  $dialog->vbox->pack_start($operationHBox, FALSE, FALSE, 0);

  # Description
  my $shortDescription = Gtk2::Expander->new(
      $UIMsg->Render(ID => 'Medium Configurator Short Description',
                     Arguments => [ $mediaType ]));
  my $longDescription = Gtk2::Label->new(
    $UIMsg->Render('Medium Configurator Long Description'));
  $longDescription->set_line_wrap(TRUE);
  $shortDescription->add($longDescription);
  $dialog->vbox->pack_start($shortDescription, FALSE, FALSE, 0);

  $dialog->show_all;
  my $response = $dialog->run;
  # Write out the catalog if so desired
  if ($response eq 'ok') {
    generateCatalog($catalog, $mediaList->get_model);
    writeActionCatalog($catalog, @path);
  }
  $dialog->destroy;

  return;
}

sub templateConfirm {
  my ($template) = @_;

  # Confirm the installation
  my $confirmMessage = Gtk2::MessageDialog->new(
    undef, [], qw/question yes-no/,
    $UIMsg->Render('Confirm Installation'));
  $confirmMessage->set_position('center');
       
  # Get a reply
  $confirmMessage->show_all;
  my $response = $confirmMessage->run;
  $confirmMessage->destroy;

  # Go ahead an install if confirmed
  return templateInstall($template) if $response eq 'yes';
  # Otherwise, we don't need to do this again
  return;
}

sub templateInstall {
  my ($template) = @_;

  # Check the MIME type of the file
  my $mimetype = mimetype($template);

  # Raise an error if needed
  unless ($mimetype =~ m!^application/x-(?:compressed-)?tar$!) {
    my $errorMessage =
      Gtk2::MessageDialog->new(undef, [], qw/error ok/,
                               $ErrorMsg->Carp('Invalid Archive'));
    $errorMessage->set_position('center');
    $errorMessage->run;
    $errorMessage->destroy;
    return 1; # Redo!
  }

  # Find the resource path and create it if necessary
  my $root = makeConfigResourcePath('Action');

  # Change directory and extract the template archive
  chdir $root; Archive::Tar->extract_archive($template);

  # Confirm the installation (to the user)
  my $confirmMessage =
    Gtk2::MessageDialog->new(undef, [], qw/info ok/,
                             $UIMsg->Render('Installation Complete'));
  $confirmMessage->set_position('center');
  $confirmMessage->run;
  $confirmMessage->destroy;

  return; # We can quit TemplateSelect then
}

sub TemplateSelect {
  my $fileSelector =
    Gtk2::FileSelection->new($UIMsg->Render('Select Template'));
  $fileSelector->set_position('center');
  my $template;

  # If the user clicks OK, store the filename through closure goodness.
  # Otherwise, the function will return
  $fileSelector->ok_button->signal_connect(
    clicked => sub { $template = $fileSelector->get_filename;
                     Gtk2->main_quit });
  $fileSelector->cancel_button->signal_connect(
    clicked => sub { undef $template; Gtk2->main_quit });
  $fileSelector->signal_connect(
    destroy => sub { Gtk2->main_quit; return FALSE });
  $fileSelector->show_all;
  
  while (1) {
    Gtk2->main; # Wait for response
    last unless defined $template; # Otherwise...
    # Redo if templateConfirm returns false
    last unless templateConfirm($template); 
  }

  $fileSelector->destroy;
  return;
}

MainMenu();
