# -*- project-name: VASM -*-
package VASM::Menu;

use strict;
use warnings;
use base qw/VASM::Tree VASM::Menu::Translators/;
use Carp;
use XML::Parser;
use XML::Writer;

our $VERSION = '1.01';

sub Store {
  my ($self, $attrs, $item, @folders) = (shift, pop, pop, @_);

  # At the very least, there must be an item name and associated properties
  croak 'Attributes hashref invalid!' unless defined $item and defined %$attrs;

  # Notice that $attrs is copied when interned
  $self->SUPER::Store(@folders, $item, { %$attrs });
  
  return;
}

sub Retrieve {
  my ($self, @path) = @_;
  
  croak 'Path argument must be given to Retrieve!' unless @path;
  my $attrs = $self->SUPER::Retrieve(@path);
  # Here again, we copy the hash so that it may not be modified
  return { %$attrs } if defined $attrs;

  return;
}

sub Parse {
  my ($self, $handle) = @_;
  # See Parse in VASM::Message for lament
  my $parser = XML::Parser->new(Style => 'Tree', ErrorContext => 3);
  my $tree = $parser->parse($handle);

  # This absolutely /has to/ recurse, and we obviously can't have multiple
  # instances of XML parsers running around. (I could check the information
  # from the caller built-in and special case it, but dollars to doughnuts
  # says I get reborn as Brian Peppers if God exists.)
  $self->parseHeavy($tree->[1]);
}

sub parseHeavy {
  # $tags is the chunk of parsed XML under examination, and @folders is a
  # stack representing the current folder hierarchy
  my ($self, $tags, @folders) = @_;

  for my $index (grep { $_ % 2 } (0..$#{ $tags })) {
    next if $tags->[$index] eq '0'; # Skip text elements

    if ($tags->[$index] eq 'menu') {
      # Label attribute must be defined
      croak 'Label attribute undefined in menu tag!'
        unless defined $tags->[$index + 1]->[0]->{label};
      $self->parseHeavy($tags->[$index + 1],
                        @folders, $tags->[$index + 1]->[0]->{label});
    } elsif ($tags->[$index] eq 'item') {
      croak 'Label attribute undefined in item tag!'
        unless defined $tags->[$index + 1]->[0]->{label};

      my $attrs = $tags->[$index + 1]; my $attrHash = {};
      for my $index (grep { $_ % 2 } (0..$#{ $attrs })) {
        next if $attrs->[$index] eq '0';

        # Add the attribute to the hash of item attributes if valid
        if (grep { $_ eq $attrs->[$index] } qw/path icon description/) {
          croak 'Item attribute undefined!'
            unless defined $attrs->[$index + 1]->[2];
          $attrHash->{$attrs->[$index]} = $attrs->[$index + 1]->[2];
        }
      }

      # Intern it unless $attrHash lacks important fields
      croak 'Item attributes hash incomplete!' unless keys %$attrHash == 3;
      $self->Store(@folders, $tags->[$index + 1]->[0]->{label}, $attrHash);
    }
  }

  return 1; # Success!
}

sub Write {
  my ($self, $handle) = @_;

  croak 'Handle argument must be given to Write!' unless defined $handle;

  # Now create an XML::Writer instance and turn that puppy out
  my $xml = XML::Writer->new(OUTPUT => $handle, ENCODING => 'utf-8',
                             DATA_MODE => 1, DATA_INDENT => 2);

  # Open up the document
  $xml->xmlDecl;
  $xml->startTag('menu');
  $xml->comment('Generated by VASM');

  $self->writeHeavy($xml); # Now let writeHeavy do its stuff

  $xml->endTag('menu'); $xml->end; # End the document...

  return 1; # Success!
}

sub writeHeavy {
  my ($self, $xml, @folders) = @_;

  for my $child ($self->Children(@folders)) {
    # Recurse if the child in question has children of its own, in between
    # producing the appropriate opening and closing tags
    if ($self->Children(@folders, $child)) {
      $xml->startTag('menu', label => $child);
      $self->writeHeavy($xml, @folders, $child);
      $xml->endTag('menu');
    }

    # Produce 'item' stanzas where items are defined
    if (my $attrs = $self->Retrieve(@folders, $child)) {
      $xml->startTag('item', label => $child);
      for my $key (qw/path icon description/) {
        $xml->dataElement($key, $attrs->{$key});
      }
      $xml->endTag('item');
    }
  }
}

1;

__END__

=head1 NAME

VASM::Menu - general window manager menu definitions

=head1 SYNOPSIS

=head1 DESCRIPTION

=head1 METHODS

=head1 AUTHORS

hanumizzle L<hanumizzle@gmail.com> wrote VASM::Menu.

=cut

