# -*- project-name: VASM -*-
package VASM::Resource::Menu;

use strict;
use warnings;
use base qw/Exporter/;
use Carp;
use VASM::Menu;
use VASM::Resource;

our $VERSION = '1.0';

sub findMenu {
  my @path = @_; # Resource path for menu
  my @locale = parseLocale(); # Finagle locale from %ENV

  # Now, try to find the first matching Message.xml in the hierarchy; the
  # search can go as far back as the base 'Message' directory itself:
  my $file;
  do {{
    $file = (findConfigResource(@path, @locale, 'Menu.xml'))[0];
    last if defined $file;
  }} while (pop @locale);

  croak 'Could not find a matching menu file!'
    unless $file; # Return nothing unless a file was found, otherwise:
  # Digest file and return the new menu definition
  open my $fh, '<:utf8', $file or croak $file, ": $!"; my $menu;
  $menu = VASM::Menu->new; $menu->Parse($fh);
  close $fh;

  return $menu;
}

sub writeMenu {
  my ($instance, @path) = @_;
  
  # Generate a new IO::Handle for the catalog to be written.
  # makeDataResource (actually, makeResource) will croak if unsuccessful, so
  # no error checking is really necessary.
  my $handle = makeDataResource(@path, 'Menu.xml');
  $instance->Write($handle); # Now let the instance do its thing
  $handle->close; # ...close the filehandle and...done
  
  return;
}

# Cut and pasted from VASM::Resource::Message:

# Because most of the normal divisions of locale support (e.g., LC_COLLATE,
# LC_NUMERIC) do not apply here, we will only honor a certain subset of
# environment variables, in this order: LC_ALL, LC_MESSAGES, LANG. (I believe
# this is the correct protocol; please correct me if I'm wrong.) Because VASM
# strictly uses UTF-8 for all messages, any codeset portion of the variable is
# ignored altogether.
sub parseLocale {
  # Loop through these variables in turn:
  for my $var (@ENV{qw/LC_ALL LC_MESSAGES LANG/}) {
    defined $var and return
      eval {
        my $locale = ($var =~ m!(\p{IsLower}{2}(?:_\p{IsUpper}{2})?)!)[0];
        # Split language and country (ex: en_GB vs. en_US)
        split /_/, $locale;
      }
  }

  # We buck the trend a little by using resource inheritance instead of the
  # default 'C' locale
  return;
}
  
1;

=head1 NAME

VASM::Resource::Menu

=head1 SYNOPSIS

=head1 DESCRIPTION

=head1 FUNCTIONS

=head1 VARIABLES

=head1 AUTHORS

hanumizzle L<mailto:hanumizzle@gmail.com> wrote VASM::Resource::Menu.

=cut

