# -*- project-name: VASM -*-
package VASM::Resource::Action;

use strict;
use warnings;
use base qw/Exporter/;
use Carp;
use VASM::Action;
use VASM::Resource;
use VASM::Resource::Message;
use File::MimeInfo::Magic;
use IO::Handle;
use URI;

our @EXPORT = qw/findActionCatalog
                 findActionCatalogByMedium
                 mediaDirectory
                 writeActionCatalog/;
our $VERSION = '1.08';

sub findActionCatalog {
  # The resource path in which to find the action catalog
  my @path = @_;

  # Find the matching file with the highest precedence
  my $file = (findConfigResource('Action', @path, 'Action.xml'))[0];

  # Not necessarily an error
  carp 'Could not find a matching action catalog!'
    unless $file; # Return nothing unless a file was found, otherwise
  # Digest file and return the new catalog
  open my $fh, '<:utf8', $file or croak $file, ": $!";
  my $catalog = VASM::Action->new; $catalog->Parse($fh);
  close $fh;

  return $catalog;
}

sub findActionCatalogByMedium {
  # A URI or absolute local filename. Note that 'file:///home/hanuman' is
  # different from '/home/hanuman', because many programs still refuse to
  # accept the URI form (such as qiv, for instance).
  my ($path) = @_; my ($scheme, $mime, $catalog);

  # If $path is a valid URI, good, try to find an actions catalog for it:
  if (defined ($scheme = URI->new($path)->scheme())) {
    $catalog = findActionCatalog('URI', $scheme);
  } else { # By elimination, assume that this is a file
    # The file must exist for magic detection to work, obviously
    croak 'MIME type unavailable or file non-existent!'
      unless defined ($mime = mimetype($path));
    $catalog = findActionCatalog('File', (split m!/!, $mime));

    # If that fails, check each superclass of the mimetype
    unless (defined $catalog) {
      for my $superclass (File::MimeInfo::mimetype_isa($mime)) {
        $catalog = findActionCatalog('File', (split m!/!, $superclass));
        # Exit if satisfied
        last if defined $catalog;
      }
    }
  }

  return $catalog; # Will be undef if nothing was found
}

# Return a VASM::Message object describing all media types usable by
# VASM::Action, both for MIME and URI data:
sub mediaDirectory {
  my $catalog = findMessageCatalog(qw/Action Directory/, $_[0]);

  # The benefits of encapsulation
  return $catalog;
}

sub writeActionCatalog {
  my ($instance, @path) = @_;
  
  # Generate a new IO::Handle for the catalog to be written.
  # makeConfigResource (actually, makeResource) will croak if unsuccessful, so
  # no error checking is really necessary.
  my $handle = makeConfigResource('Action', @path, 'Action.xml');

  $instance->Write($handle); # Now let the instance do its thing

  $handle->close; # ...close the filehandle and...done
  
  return;
}

1;

__END__

=head1 NAME

VASM::Resource::Action - load action catalogs with the VASM resources
framework

=head1 SYNOPSIS

    use VASM::Resource::Action;

=head1 DESCRIPTION

=head1 FUNCTIONS

=over

=item 

=back

=head1 AUTHORS

hanumizzle L<hanumizzle@gmail.com> wrote VASM::Resource::Action.

=cut
