#!/usr/bin/perl
# -*- project-name: VASM -*-

use strict;
use warnings;
use constant { FALSE => 0, TRUE => 1 };
use constant { REAL => 0, SHOWN => 1 };
use VASM::Tree;
use VASM::Resource;
use VASM::Resource::Message;
use VASM::Resource::Action;
use Gtk2 '-init';
use File::MimeInfo::Magic;
use Archive::Tar;

# Message files
my $CommonMsg = findMessageCatalog(qw/Common Message/);
my $UIMsg = findMessageCatalog(qw/Action UI Gtk2 Message/);
my $ErrorMsg = findErrorCatalog(qw/Action UI Error/);

sub makeChoiceTree {
  # Get directories for file and URI schemata
  my $fileDirectory = mediaDirectory('File');
  my $uriDirectory = mediaDirectory('URI');

  # Convert the file directory to data more readily usable by the function
  my $fileTypeTree = VASM::Tree->new;
  for my $mime ($fileDirectory->List) {
    $fileTypeTree->Store((split m!/!, $mime), 1);
  }

  # Now, set up a tree store corresponding to them and add the 'Template'
  # top-level choice as well. The first Glib::String is the item as it appears
  # on the screen, according to translation and additional notes. The second
  # is the 'real name', usable to the program.
  my $choiceTree = Gtk2::TreeStore->new('Glib::String', 'Glib::String');
  my @iters;

  # File
  push @iters, $choiceTree->append(undef);
  $choiceTree->set($iters[-1],
                   REAL, 'File',
                   SHOWN, $UIMsg->Render('File')
                 );
  for my $class (sort $fileTypeTree->Children) {
    # Add the class under 'File'
    push @iters, $choiceTree->append($iters[-1]);
    $choiceTree->set($iters[-1],
                     REAL, $class,
                     SHOWN, $class);
    # Add each member of the class (e.g., File -> image -> jpeg)
    for my $member (sort $fileTypeTree->Children($class)) {
      push @iters, $choiceTree->append($iters[-1]);
      my $mime = join('/', $class, $member);
      # The type will appear with a parenthetical note describing it in
      # further detail
      $choiceTree->set(
        $iters[-1],
        REAL, $member,
        SHOWN, $member . ' (' . $fileDirectory->Render($mime) . ')',
      );
    } continue { pop @iters }
  } continue { pop @iters }
  pop @iters;

  # URI
  push @iters, $choiceTree->append(undef);
  $choiceTree->set($iters[-1],
                   REAL, 'URI',
                   SHOWN, $UIMsg->Render('URI'));
  for my $scheme (sort $uriDirectory->List) {
    push @iters, $choiceTree->append($iters[-1]);
    $choiceTree->set(
      $iters[-1],
      REAL, $scheme,
      SHOWN, $scheme . ' (' . $uriDirectory->Render($scheme) . ')');
  } continue { pop @iters }
  pop @iters;

  # Template
  push @iters, $choiceTree->append(undef);
  $choiceTree->set($iters[-1],
                   REAL, 'Template',
                   SHOWN, $UIMsg->Render('Template'));
  pop @iters; # Neat freak

  return $choiceTree;
}

sub makeChoiceTreeView {
  my $treeStore = makeChoiceTree(); # The actual data
  # The rest of this is a 'cookbook' recipe to make a tree view correspond
  # one-for-one with the data in the tree store.
  my $treeView = Gtk2::TreeView->new_with_model($treeStore);
  $treeView->set_headers_visible(FALSE);
  my $col = Gtk2::TreeViewColumn->new;
  my $renderer = Gtk2::CellRendererText->new;
  $col->pack_start($renderer, FALSE);
  $col->add_attribute($renderer, "text", SHOWN);
  $treeView->append_column($col);

  return $treeView;
}

sub makeMediaList {
  my ($catalog) = @_; my $iter;
  my $mediaList = Gtk2::ListStore->new('Glib::String');
  my @programs = $catalog->Dump;

  if (@programs) { # Add each program in the catalog to the list store
    for my $program (@programs) {
      $iter = $mediaList->append;
      $mediaList->set($iter, REAL, $program);
    }
  } else { # Empty or non-existent catalog
    $iter = $mediaList->append;
    $mediaList->set($iter, REAL, '<edit me>'); # Add a new element
  }
  
  return $mediaList;
}

sub makeMediaListView {
  my ($catalog) = @_;

  # Cookbook recipe that instantiates a list view store from an actions
  # catalog and imposes an editable view component on it
  my $listStore = makeMediaList($catalog);
  my $listView = Gtk2::TreeView->new_with_model($listStore);
  $listView->set_headers_visible(FALSE);
  my $col = Gtk2::TreeViewColumn->new;
  my $renderer = Gtk2::CellRendererText->new;
  $renderer->set(editable => TRUE);
  $renderer->signal_connect(edited => \&listCellEdited, $listStore);
  $col->pack_start($renderer, FALSE);
  $col->add_attribute($renderer, text => REAL);
  $listView->append_column($col);

  return $listView;
}

sub listCellEdited {
  my ($pathString, $text, $store) = @_[1..3];

  # Just pass the text through if not blank
  if (length($text)) {
    $store->set($store->get_iter_from_string($pathString), REAL, $text);
  } else {
    # Notify user of empty list element
    my $emptyMessage =
      Gtk2::MessageDialog->new(undef, qw/modal error ok/,
                               $ErrorMsg->Render('Null Command String'));
    $emptyMessage->set_position('center');
    $emptyMessage->run;
    $emptyMessage->destroy;
  }

  return;
}

sub identifyTreeRow {
  my $view = $_[0]; my @path;
  my $store = $view->get_model;
  my $iter = $view->get_selection->get_selected;

  # Walk from branch to trunk, unshifting the 'real' column at each point. The
  # result will define the path in the tree store.
  do {
    unshift @path, $store->get($iter, REAL);
  } while ($iter = $store->iter_parent($iter));

  return @path;
}

sub addProgram {
  my $view = $_[1];
  my $store = $view->get_model;
  my $iter = $store->insert_after(scalar $view->get_selection->get_selected);
  $store->set($iter, REAL, '<edit me>'); # Add a new element

  return;
}

sub removeProgram {
  my $view = $_[1];
  my $store = $view->get_model;
  my $iter = $view->get_selection->get_selected;
  $store->remove($iter) if $iter;

  return;
}

sub upProgram {
  my $view = $_[1];
  my $store = $view->get_model;
  my $iter = $view->get_selection->get_selected;
  my $prev = eval {
    my $path = $store->get_path($iter);
    return $store->get_iter($path) if $path->prev;
  };
  $store->swap($prev, $iter) if $prev;
  
  return;
}

sub downProgram {
  my $view = $_[1];
  my $store = $view->get_model;
  my $iter = $view->get_selection->get_selected;
  my $next = eval {
    my $path = $store->get_path($iter); $path->next;
    return $store->get_iter($path);
  };
  $store->swap($iter, $next) if $next;
  
  return;
}

sub generateCatalog {
  my ($catalog, $store) = @_; my $iter;

  $catalog->Clear; # Wipe out the catalog first
  # Add an element in the catalog for each member in the list store
  $store->foreach(
    sub { $catalog->Push($_[0]->get($_[2], REAL)); return FALSE }
  );
  
  return;
}

sub dispatchMediaType {
  my @path = @_;

  # Fully qualified path was given
  if (($path[0] eq 'File' and @path == 3) or
      ($path[0] eq 'URI' and @path == 2)) {
    MediumConfigure(@path);
  } elsif ($path[0] eq 'Template') { # Select a template
    TemplateSelect();
  }
  
  return;
}

sub MainMenu {
  while (1) {
    # Dialog
    my $dialog = Gtk2::Dialog->new_with_buttons(
      $UIMsg->Render('Title'),
      undef,
      [ qw/modal/ ],
      # Handles own translation...anyway, 'OK' is universal, AFAIK
      'gtk-close' => 'close',
      'gtk-ok' => 'none');
    $dialog->set_position('center');
    $dialog->set_default_size(400, 300);
    
    ## Action area
    # Tree with media
    my $choiceTree = makeChoiceTreeView();
    # Pack the tree view into a scrolled window. The addition of scrollbars
    # will be done automatically in either direction
    my $scrolledTree = Gtk2::ScrolledWindow->new;
    $scrolledTree->set_policy('automatic', 'automatic');
    $scrolledTree->add($choiceTree);
    $dialog->vbox->pack_start($scrolledTree, TRUE, TRUE, 0);
    # Separator
    $dialog->vbox->pack_start(Gtk2::HSeparator->new(), FALSE, FALSE, 0);
    # Expander with detailed description
    my $shortDescription = Gtk2::Expander->new(
      $UIMsg->Render('Main Menu Short Description'));
    my $longDescription = Gtk2::Label->new(
      $UIMsg->Render('Main Menu Long Description'));
    $longDescription->set_line_wrap(TRUE);
    $shortDescription->add($longDescription);
    $dialog->vbox->pack_start($shortDescription, FALSE, FALSE, 0);

    $dialog->show_all;
    my $response = $dialog->run; # Run has its own main loop
    # Terminate here if so desired
    exit 0 if $response eq 'close' or $response eq 'delete-event';
    my @path = identifyTreeRow($choiceTree); # Grab the selected row
    $dialog->destroy; # ...and destroy the dialog

    dispatchMediaType(@path); # Dispatch the selection
  }
}

sub MediumConfigure {
  my @path = @_;
  my $mediaType = (@path == 3) ? join '/', @path[1..2] : $path[1];

  # First, load the action catalog, if possible
  my $catalog = findActionCatalog(@path);

  # If it didn't exist, that's OK; just instantiate a blank one
  $catalog = VASM::Action->new unless defined $catalog;

  # Dialog
  my $dialog = Gtk2::Dialog->new_with_buttons(
    $mediaType,
    undef,
    [ qw/modal/ ],
    'gtk-cancel' => 'cancel',
    'gtk-ok' => 'ok');
  $dialog->set_position('center');

  # Media list
  my $mediaList = makeMediaListView($catalog);
  $mediaList->grab_focus; # Using shortcuts is damned annoying otherwise
  $dialog->vbox->pack_start($mediaList, TRUE, TRUE, 0);

  # HSeparator
  $dialog->vbox->pack_start(Gtk2::HSeparator->new(), FALSE, FALSE, 0);

  # Operations
  my $operationHBox = Gtk2::HBox->new(TRUE, 0);
  my @operations = ( [ 'gtk-add', \&addProgram ],
                     [ 'gtk-remove', \&removeProgram ],
                     [ 'gtk-go-up', \&upProgram ],
                     [ 'gtk-go-down', \&downProgram ] );
  # Add them, one by one
  for my $operation (@operations) {
    my $button = Gtk2::Button->new_from_stock($operation->[0]);
    $button->signal_connect(clicked => $operation->[1], $mediaList);
    $operationHBox->pack_start($button, TRUE, TRUE, 0);
  }
  $dialog->vbox->pack_start($operationHBox, FALSE, FALSE, 0);

  # Description
  my $shortDescription = Gtk2::Expander->new(
      $UIMsg->Render(ID => 'Medium Configurator Short Description',
                     Arguments => [ $mediaType ]));
  my $longDescription = Gtk2::Label->new(
    $UIMsg->Render('Medium Configurator Long Description'));
  $longDescription->set_line_wrap(TRUE);
  $shortDescription->add($longDescription);
  $dialog->vbox->pack_start($shortDescription, FALSE, FALSE, 0);
  $dialog->show_all;

  my $response = $dialog->run;
  # Write out the catalog if so desired
  if ($response eq 'ok') {
    generateCatalog($catalog, $mediaList->get_model);
    writeActionCatalog($catalog, @path) if $response eq 'ok';
  }
  $dialog->destroy;

  return;
}

sub TemplateSelect {
 PROMPT: 
  {
    my $fileSelector =
      Gtk2::FileSelection->new($UIMsg->Render('Select Template'));
    $fileSelector->set_position('center');
    # Start in the working directory by default
    # $fileSelector->set_filename("$ENV{PWD}/");
    my $template;

    # If the user clicks OK, store the filename through closure goodness.
    # Otherwise, the function will return
    $fileSelector->ok_button->signal_connect(
      clicked => sub { $template = $fileSelector->get_filename;
                       Gtk2->main_quit });
    $fileSelector->cancel_button->signal_connect(
      clicked => sub { Gtk2->main_quit });
    $fileSelector->signal_connect(
      destroy => sub { Gtk2->main_quit; return FALSE });
    
    $fileSelector->show_all;
    Gtk2->main;
    $fileSelector->destroy;

    if (defined $template) { # User didn't cancel
      # Find the resource path and create it if necessary
      my $root = makeConfigResourcePath('Action');

      # Confirm the installation
      my $dialog = Gtk2::Dialog->new_with_buttons(
        $UIMsg->Render('Confirm Installation Title'),
        undef,
        [ qw/modal/ ],
        'gtk-no' => 'no',
        'gtk-yes' => 'yes');
      $dialog->set_position('center');
      
      # Explanation label
      my $label = Gtk2::Label->new(
        $UIMsg->Render('Confirm Installation Label'));
      $label->set_line_wrap(TRUE);
      $dialog->vbox->pack_start($label, FALSE, FALSE, 0);

      $dialog->show_all;
      my $response = $dialog->run;
      $dialog->destroy;
      if ($response eq 'yes') { # Do it Jah!
        # Check the MIME type of the file
        my $mimetype = mimetype($template);

        # Raise an error if needed
        unless ($mimetype =~ m!^application/x-(?:compressed-)?tar$!) {
          my $errorMessage =
            Gtk2::MessageDialog->new(undef, qw/modal error ok/,
                                     $ErrorMsg->Carp('Invalid Archive'));
          $errorMessage->set_position('center');
          $errorMessage->run;
          $errorMessage->destroy;
          redo PROMPT;
        }

        # Find the resource path and create it if necessary
        my $root = makeConfigResourcePath('Action');

        # Change directory and extract the template archive
        chdir $root; Archive::Tar->extract_archive($template);

        # Confirm the installation (to the user)
        my $confirmMessage =
          Gtk2::MessageDialog->new(undef, qw/modal info ok/,
                                   $UIMsg->Render('Installation Complete'));
        $confirmMessage->set_position('center');
        $confirmMessage->run;
        $confirmMessage->destroy;
      }
    }
  }

  return;
}

MainMenu();
