# -*- project-name: VASM -*-
package VASM::Resource::Message;

use strict;
use warnings;
use base qw/Exporter/;
use Carp;
use VASM::Message;
use VASM::Error;
use VASM::Resource;

our @EXPORT = qw/findMessageCatalog findErrorCatalog/;
our $VERSION = '1.02';

sub findCatalog {
  my %args = @_;

  # Look under the resource path given.  
  my @path = @{ $args{Path} };

  # Finagle locale from %ENV
  my @locale = parseLocale();

  # Now, try to find the first matching Message.xml in the hierarchy; the
  # search can go as far back as the base 'Message' directory itself:
  my $file;
  do {{
    $file = (findDataResource(@path, @locale, 'Message.xml'))[0];
    last if defined $file;
  }} while (pop @locale);

  return unless $file; # Return nothing unless a file was found, otherwise:

  # Digest file and return the new catalog
  open my $fh, '<:utf8', $file or croak $file, ": $!"; my $catalog;
  if ($args{Type} eq 'Message') {
    $catalog = VASM::Message->new($fh);
  } elsif ($args{Type} eq 'Error') {
    $catalog = VASM::Error->new($fh);
  }
  close $fh;

  return $catalog;
}

sub findMessageCatalog {
  return findCatalog(Path => [ @_ ], Type => 'Message');
}

sub findErrorCatalog {
  return findCatalog(Path => [ @_ ], Type => 'Error');
}

sub writeMessageCatalog {
  my ($instance, @path) = @_;
  
  # Generate a new IO::Handle for the catalog to be written.
  # makeDataResource (actually, makeResource) will croak if unsuccessful, so
  # no error checking is really necessary.
  my $handle = makeDataResource(@path, 'Message.xml');

  # Now let the instance do its thing
  $instance->Write($handle);

  $handle->close; # ...close the filehandle and...done
  
  return;
}

# Because most of the normal divisions of locale support (e.g., LC_COLLATE,
# LC_NUMERIC) do not apply here, we will only honor a certain subset of
# environment variables, in this order: LC_ALL, LC_MESSAGES, LANG. (I believe
# this is the correct protocol; please correct me if I'm wrong.) Because VASM
# strictly uses UTF-8 for all messages, any codeset portion of the variable is
# ignored altogether.
sub parseLocale {
  # Loop through these variables in turn:
  for my $var (@ENV{qw/LC_ALL LC_MESSAGES LANG/}) {
    defined $var and return
      eval {
        my $locale = ($var =~ m!(\p{IsLower}{2}(?:_\p{IsUpper}{2})?)!)[0];
        # Split language and country (ex: en_GB vs. en_US)
        split /_/, $locale;
      }
  }

  # We buck the trend a little by using resource inheritance instead of the
  # default 'C' locale
  return;
}

1;

__END__

=head1 NAME

VASM::Resource::Message - load message catalogs with the VASM resources
framework

=head1 SYNOPSIS

    use VASM::Resource::Message;
    
    # Say LANG=de, and there is a catalog in
    # /usr/share/vasm/Some/Project/Message/de/Message.xml
    my $catalog = findMessageCatalog(qw/Some Project/);
    # See VASM::Message
    $catalog->Render(ID => 'Hello');
    # Guten Tag!

=head1 DESCRIPTION

VASM::Resource::Message enables the programmer to use VASM::Message i18n
message catalogs with the resource hierarchy defined by L<VASM::Resource>. It
will search for a message catalog under the directory Message in a given
resource path, in the fashion prescribed by the XDG Base Directory standard,
used by VASM::Resource. More specifically, VASM::Message will attempt to
discern the current locale, and consequently, the appropriate message file
from environment variables that signify the locale (see L<"VARIABLES"> below).
If a message catalog for a given language/country pair cannot be found, the
search will default to a catalog for the language independent of any national
dialect, and, failing that, to a default catalog most likely written in
English.

=head1 FUNCTIONS

=over

=item findMessageCatalog

findMessageCatalog accepts a list describing a resource path and seeks a
message catalog therein, according to the rules described above. If
successful, it will return an instance of VASM::Message after using it to
digest the catalog. Otherwise, it will return nothing.

=back

=head1 VARIABLES

These environment variables are searched in order for locale information, and
their values must conform to the POSIX convention consisting of a lowercase
two-letter language code optionally followed by an underscore and an uppercase
two-letter country code, e.g., hi_IN. Encoding information qualifiers such as
'.utf8' are ignored, because the VASM project strictly adheres to UTF-8:

=over

=item *

LC_ALL

=item *

LC_MESSAGES

=item *

LANG

=back

=head1 AUTHORS

hanumizzle L<mailto:hanumizzle@gmail.com> wrote VASM::Resource::Message.

=cut
