# -*- project-name: VASM -*-
package VASM::Action;

use strict;
use warnings;
use XML::Parser;
use XML::Writer;

our $VERSION = '1.05';

sub new {
  my ($self, $handle) = @_;

  my $instance = []; # Commands pushed and popped using arrayref
  bless $instance, $self;

  # Initialize the instance if a file is specified
  $instance->Parse($handle) if defined $handle;

  return $instance;
}

# Add another command string to the end of the actions list
sub Push {
  my ($self, $command) = @_;

  push @{ $self }, $command;
  return;
}

# You get the idea...
sub Unshift {
  my ($self, $command) = @_;
  
  unshift @{ $self }, $command;
  return;
}

# Remove a command string from the end of the actions list
sub Pop {
  my ($self) = @_;
  
  pop @{ $self };
  return;
}

sub Shift {
  my ($self) = @_;
  
  shift @{ $self };
  return;
}

# Just return the contents of the instance
sub Dump {
  my ($self) = @_;
  
  return @{ $self };
}

# Clear out the catalog altogether
sub Clear {
  my ($self) = @_;
  
  $#{ $self } = -1;
  
  return;
}

sub Exec {
  my ($self, @arguments) = @_; my $status;
  
  for my $program (@{ $self }) {
    $status = system sprintf $program, @arguments;
    last if $status != -1;
  }

  return $status; # Will be -1 if none found
}

sub Parse {
  my ($self, $handle) = @_;
  # If this fails, XML::Parser will emit an error message and end the program.
  # Unfortunately, this message is always English. :( There is at least an
  # error context, which can be universally understood by the programmer.
  my $parser = XML::Parser->new(Style => 'Tree', ErrorContext => 3);
  my $tree = $parser->parse($handle); my $tags = $tree->[1];
  
  # Iterate through tags under <actions>
  for my $index (grep { $_ % 2 } (0..$#{ $tags })) {
    next if $tags->[$index] eq '0'; # Skip text elements 

    # Otherwise, it /has to be/ a command string. Return if it does not
    # contain a text string or holds no content
    return unless $tags->[$index + 1]->[1] eq '0'
      and defined $tags->[$index + 1]->[2];

    # Push the new member then
    $self->Push($tags->[$index + 1]->[2]);
  }

  return 1; # Success!
}

# Write out the contents of the instance to a file, given a filehandle
sub Write {
  my ($self, $handle) = @_;

  # Now create an XML::Writer instance and turn that puppy out
  my $xml = XML::Writer->new(OUTPUT => $handle, ENCODING => 'utf-8',
                             DATA_MODE => 1, DATA_INDENT => 2);

  # Open up the document
  $xml->xmlDecl;
  $xml->startTag('actions');
  $xml->comment('Generated by VASM; hanumizzle was here'); # Had to... :D

  # Emit the 'command' tags
  for my $command (@{ $self }) {
    $xml->dataElement('command', $command);
  }

  $xml->endTag('actions'); $xml->end; # End the document...

  return 1; # Successful!
}
  
1;

__END__

=head1 NAME

VASM::Action - preference registration for diverse media types

=head1 SYNOPSIS

    use VASM::Action;
    
    # Instantiate VASM::Action object using an IO::Handle. This will call the
    # Parse method of the new instance, passing that handle in.
    my $instance = VASM::Action->new($ioHandle);
    
    # Wipe out what was there earlier
    $instance->Clear;
    
    # Add some new programs
    $instance->Push('gvim %s');
    $instance->Push('nedit %s');
    $instance->Push('notepad %s');
    
    # Change your mind?
    $instance->Pop;
    
    # Even better
    $instance->Unshift('emacsclient %s');
    
    # (Push, Pop, Unshift, and Shift are all available, although they return
    # nothing)
    
    # Print out contents of catalog
    print for $instance->Dump;

    # Execute the first command found in the catalog with an argument. This
    # will run 'emacsclient /home/hanuman/sithon-and-manola'
    $instance->Exec('/home/hanuman/sithon-and-manola');
    
    # Write out the contents of the catalog via IO::Handle 
    $instance->Write($ioHandle);

=head1 DESCRIPTION

VASM::Action defines lists of programs associated with a given media type as
viewers and/or editors. In principle, VASM::Action can be used with any media
type represented in the arguments to a Unix command, as it is completely
agnostic towards them, being responsible only for executing a command in a
list. In practice, this means action catalogs are used in conjunction with
VASM::Resource::Action to pair vanilla file pathnames and URIs (Uniform
Resource Indicators, such as http://distro.ibiblio.org) to preferred programs.
Once prepared, the Exec method of an instance will systematically run through
a list of programs and execute the first valid command with arguments, if so
desired. Exec may fail to find a single usable command line; in this case, it
will return -1, propagating the return value of Perl's built-in system
function (as of Perl 5.x; this is liable to change in Perl 6).

=head1 METHODS

=over

=item new

new constructs a new instance of the VASM::Message class, and accepts an
optional argument, indicating an IO::Handle representing an XML file, whose
catalog definition you wish to introduce to the instance upon construction.
See L</FORMAT> below.

=item Push, Pop, Unshift, and Shift

These all correspond to the built-in Perl operators of the same name, adding
and subtracting definitions as one would expect. The methods do not return
anything, though.

=item Dump

This returns a list representing the catalog as it stands at the time the
method was executed. Exposing state like this isn't very OO, but necessary in
a number of cases.

...besides, you can: #always use: #Smalltalk.

=item Clear

Clear empties any definitions in the catalog, takes no arguments, and returns
nothing. Yeah, it's a BAMF method.

=item Parse

This method accepts a single argument, an IO::Handle whose referent XML file
to intern in the instance's message collection. (See </FORMAT> below.) Because
this process is somewhat more complex than the methods previously described
and requires validation of user input (those damn translators!), Parse returns
a true value if successful.

=item Write

The Write method serves as the counterpart to the Parse method, accepting an
IO::Handle with which to serialize the invocant instance. The output produced
takes human beings into account, and will follow reasonable indenting
guidelines. Like Parse, Write also returns a true value if successful.

=item Exec

The method Exec attempts to execute each command string in the catalog, to
which it will apply the list of arguments passed in, using sprintf formatting
directives. The method will return when an attempt is successful, or when all
possibilities in the catalog have been executed. The return value is the same
as the value returned by Perl's system function, which will certainly be -1 if
no command could be found.

=back

=head1 FORMAT

The XML format for VASM::Action is even simpler than that used in
VASM::Message catalogs, essentially representing a single flat list. The only
reason it /is/ XML is because the author likes consistency. Anyway:

    <?xml version="1.0" encoding="utf-8"?>

    <actions>
      <command>emacsclient %s</command>
      <command>gvim %s</command>
    </actions>

Each command tag pairing represents a command in an VASM::Action instance. The
entries will be interned in the order they are written.

=head1 AUTHORS

hanumizzle L<hanumizzle@gmail.com> wrote VASM::Action.

=cut
