#!/usr/bin/perl
# -*- project-name: VASM -*-

use strict;
use warnings;
use UI::Dialog::Backend::CDialog;
use File::MimeInfo::Magic;
use VASM::Resource;
use VASM::Resource::Message;
use VASM::Resource::Action;
use Archive::Tar;

# Message files
my $CommonMsg = findMessageCatalog(qw/Common Message/);
my $UIMsg = findMessageCatalog(qw/Action UI CDialog/);
my $ErrorMsg = findErrorCatalog(qw/Action UI Error/);
my $Templates = findMessageCatalog(qw/Action Templates/);

# Dialog instance
my $d = UI::Dialog::Backend::CDialog->new(
  backtitle => $CommonMsg->Render(ID => 'VASM Title'),
  title => $UIMsg->Render(ID => 'Title'),
  'ok-label' => $CommonMsg->Render(ID => 'OK'),
  'cancel-label' => $CommonMsg->Render(ID => 'Exit'),
  'yes-label' => $CommonMsg->Render(ID => 'Yes'),
  'no-label' => $CommonMsg->Render(ID => 'No'),
  width => 75,
  height => 20,
  menuheight => 10
);

sub MainMenu {
  while (1) {
    ### Read
    my $choice = $d->menu(
      text => $UIMsg->Render(ID => 'Main Menu Description'),
      list =>
        [ map 
            { $UIMsg->Render(ID => $_),
              $UIMsg->Render(ID => "$_ Description") }
              qw/File URI Template/ ]);

    ### Evaluate
    if ($d->state eq 'OK') {
      if ($choice eq 'File' or $choice eq 'URI') {
        MediaMenu($choice);
      } elsif ($choice eq 'Template') {
        TemplateSelect();
      }
    } elsif ($d->state eq 'CANCEL') {
      exit 0;
    }
  }
}

sub MediaMenu {
  my ($choice) = @_;
  my $directory = mediaDirectory($choice);

  while (1) {
    ### Read
    my @path = ($choice,
                split # For MIME types
                  m!/!,
                $d->menu(
                  text => $UIMsg->Render(ID => 'Media Menu Description'),
                  'extra-button' => 1,
                  'extra-label' => $CommonMsg->Render(ID => 'Back'),
                  list =>
                    [ map 
                        {
                          $_, $directory->Render(ID => $_) }
                          sort $directory->List ] ));

    ### Evaluate
    if ($d->state eq 'OK') {
      OperationMenu(@path);
    } elsif ($d->state eq 'EXTRA') {
      return;
    } elsif ($d->state eq 'CANCEL') {
      exit 0;
    }
  }
}

sub OperationMenu {
  my @path = @_;
  my $mediaType = (@path == 3) ? join '/', @path[1..2] : $path[1];

  while (1) {
    ### Read
    my $operation =
      $d->menu(text => $UIMsg->Render(ID => 'Operation Menu Description',
                                      Arguments => [ $mediaType ]),
               'extra-button' => 1,
               'extra-label' => $CommonMsg->Render(ID => 'Back'),
               list =>
                 [ map
                     { $UIMsg->Render(ID => "$_ Operation"),
                         $UIMsg->Render(
                           ID => "$_ Operation Description") }
                     qw/List Add Clear/ ] );

    ### Evaluate
    if ($d->state eq 'OK') {
      if ($operation eq 'List') {
        MediaList(@path);
      } elsif ($operation eq 'Add') {
        MediumAdd(@path);
      } elsif ($operation eq 'Clear') {
        MediaClear(@path);
      }
    } elsif ($d->state eq 'EXTRA') {
      return;
    } elsif ($d->state eq 'CANCEL') {
      exit 0;
    }
  }
}

sub MediaList {
  my @path = @_;
  my $mediaType = (@path == 3) ? join '/', @path[1..2] : $path[1];
  
  # Try to find a matching actions catalog
  my $catalog = findActionCatalog(@path);

  if (defined $catalog) {
    $d->msgbox(text => $UIMsg->Render(ID => 'Commands Listed',
                                      Arguments => [ $mediaType ])
                      . ":\\n\n\\n\n" # Yes, I know this this is ugly
                      . join("\\n\n", $catalog->Dump));
  } else { # Could not find a catalog!
    $d->msgbox(text => $ErrorMsg->Carp(ID => 'No Action Catalog',
                                       Arguments => [ $mediaType ]));
  }

  return;
}

sub MediumAdd {
  my @path = @_;

  # First, load the action catalog, if possible
  my $catalog = findActionCatalog(@path);
  
  # If it didn't exist, that's OK; just instantiate a blank one
  $catalog = VASM::Action->new unless defined $catalog;

  PROMPT: {
      # Prompt for a new command
      my $command = $d->inputbox(text => $UIMsg->Render(ID => 'Add Command'),
                                 'cancel-label' =>
                                   $CommonMsg->Render(ID => 'Back'));
      # Process it
      if ($d->state eq 'OK') {
        # Emit warning message and redo unless the command is a valid value
        unless ($command) {
          $d->msgbox(text => $ErrorMsg->Carp(ID => 'Null Command String'));
        redo PROMPT;
        }
        
        $catalog->Unshift($command); # Unshift it onto the catalog instance
        writeActionCatalog($catalog, @path); # Now write the instance out
      } 
    }
  
  return;
}

sub MediaClear {
  my @path = @_;
  my $mediaType = (@path == 3) ? join '/', @path[1..2] : $path[1];
  
  # Try to find a matching actions catalog
  my $catalog = findActionCatalog(@path);

  if (defined $catalog) {
    # Confirm this destructive operation
    my $confirm = $d->yesno(text => $UIMsg->Render(ID => 'Confirm Clear',
                                                   Arguments =>
                                                     [ $mediaType ]));
    if ($confirm) { # "Do it!" -- Yellowman
      $catalog->Clear;
      writeActionCatalog($catalog, @path);
    }
  } else { # Could not find a catalog!
    $d->msgbox(text => $ErrorMsg->Carp(ID => 'No Action Catalog',
                                       Arguments => [ $mediaType ]));
  }
}

sub TemplateSelect {
  PROMPT: {
      # Get the filename of a template archive
      my $template = $d->fselect();

      if ($d->state eq 'OK') {
        # Find the resource path and create it if necessary
        my $root = makeConfigResourcePath('Action');
        
        # Confirm this destructive operation
        my $confirm = $d->yesno(text =>
                                  $UIMsg->Render(ID => 'Confirm Install'));
      
        if ($confirm) { # Do it...
          # Check the MIME type of the file
          my $mimetype = mimetype($template);
          
          unless ($mimetype =~ m!^application/x-(?:compressed-)?tar$!) {
            $d->msgbox(text => $ErrorMsg->Carp(ID => 'Invalid Archive'));
            redo PROMPT;
          }

          # Change directory and extract the template archive
          chdir $root; Archive::Tar->extract_archive($template);

          # Confirm the installation (to the user)
          $d->msgbox(text =>
                       $UIMsg->Render(ID => 'Installation Complete'));
        }
      }
    }
    
  return;
}

# Start the application
MainMenu();
