# -*- project-name: VASM -*-
package VASM::Resource::Action;

use strict;
use warnings;
use base qw/Exporter/;
use VASM::Action;
use VASM::Resource;
use VASM::Resource::Message;
use IO::Handle;
use URI;
use File::MimeInfo::Magic;
use XML::Writer;

our @EXPORT = qw/findActionCatalog
                 mediaTypes
                 makeActionCatalog/;
our $VERSION = '1.03';
my $prefix = 'Action';
my $catalogName = 'Action.xml';
my $indent = 2; # The Indent Level of the Gods

sub findActionCatalog {
  # A URI or absolute local filename. Note that 'file:///home/hanuman' is
  # different from '/home/hanuman', because many programs still refuse to
  # accept the URI form (such as qiv, for instance). The action is a function
  # such as 'View', or 'Edit', and optional.
  my ($path, $action) = @_; my ($scheme, $file, $mime);

  # If $path is a valid URI, good, try to find an actions catalog for it:
  if (defined ($scheme = URI->new($path)->scheme())) {
    $file = (findConfigResource(Path => [ $prefix, 'URI', $scheme, $action ],
                                Limit => 3, File => $catalogName))[0];
  } else { # By elimination, assume that this is a file
    return unless defined ($mime = mimetype($path));
    $file = (findConfigResource(
      Path => [ $prefix, 'File', (split m!/!, $mime), $action ],
      Limit => 3, File => $catalogName))[0];

    # If that fails, check each superclass of the mimetype
    unless (defined $file) {
      for my $superclass (File::MimeInfo::mimetype_isa($mime)) {
        $file = (findConfigResource(
          Path => [ $prefix, 'File', (split m!/!, $superclass), $action ],
          Limit => 3, File => $catalogName))[0];
        # Exit if satisfied
        last if defined $file;
      }
    }
  }

  return unless $file; # Return nothing unless a file was found, otherwise:
  return VASM::Action->new($file); # Digest file and return the new catalog
}

# Return a VASM::Message object describing all media types usable by
# VASM::Action, both for MIME and URI data:
sub mediaTypes {
  my $catalog = findMessageCatalog($prefix, $_[0]);

  # The benefits of encapsulation
  return $catalog;
}

# Produce an actions catalog in the user-specific config directory
sub makeActionCatalog {
  my %args = @_;

  # Instantiate an IO::Handle object representing the specified XML file
  my $handle = IO::Handle->new_from_fd(
    fileno(makeConfigResource(Path => [ $prefix, @{ $args{Path} } ],
                              File => $catalogName)),
    'w');
  return unless defined $handle; # Return unless successful

  # Now create an XML::Writer instance and turn that puppy out
  my $xml = XML::Writer->new(OUTPUT => $handle, ENCODING => 'utf-8',
                             DATA_MODE => 1, DATA_INDENT => $indent);

  # Open up the document
  $xml->xmlDecl;
  $xml->startTag('actions');
  $xml->comment('Generated by VASM; hanumizzle was here'); # Had to... :D

  # Emit the 'command' tags
  for my $command (@{ $args{Commands} }) {
    $xml->dataElement('command', $command);
  }
  
  $xml->endTag('actions'); $xml->end; # End the document...
  $handle->close; # ...close the filehandle and...done
  
  return 1; # Successful!
}

1;

__END__
