# -*- project-name: VASM -*-
package VASM::Resources::Message;

use strict;
use warnings;
use base qw/Exporter/;
use VASM::Message;
use VASM::Resources qw/findDataResource/;

our @EXPORT_OK = qw/findMessageCatalog/;
our $VERSION = '1.0';
my $suffix = 'Messages';
my $catalogName = 'Messages.xml';

sub findMessageCatalog {
  # Look under the suffix directory in the resource path given.
  my @path = (@_, $suffix);

  # Finagle locale from %ENV
  my @locale = parseLocale();

  # Now, try to find the first matching Messages.xml in the hierarchy; the
  # search can go as far back as the base 'Message' directory itself:
  my $file = (findDataResource(Path => [ @path, @locale ],
                               Limit => scalar @path,
                               File => $catalogName))[0];

  return unless $file; # Return nothing unless a file was found, otherwise:
  return VASM::Message->new($file); # Digest file and return the new catalog
}

# Because most of the normal divisions of locale support (e.g., LC_COLLATE,
# LC_NUMERIC) do not apply here, we will only honor a certain subset of
# environment variables, in this order: LC_ALL, LC_MESSAGES, LANG. (I believe
# this is the correct protocol; please correct me if I'm wrong.) Because VASM
# strictly uses UTF-8 for all messages, any codeset portion of the variable is
# ignored altogether.
sub parseLocale {
  # Loop through these variables in turn:
  for my $var (@ENV{qw/LC_ALL LC_MESSAGES LANG/}) {
    defined $var and return
      eval {
        my $locale = ($var =~ m!(\p{IsLower}{2}(?:_\p{IsUpper}{2})?)!)[0];
        # Split language and country (ex: en_GB vs. en_US)
        split /_/, $locale;
      }
  }

  # We buck the trend a little by using resource inheritance instead of the
  # default 'C' locale
  return; 
}

1;

__END__

=head1 NAME

VASM::Resources::Message - load message catalogs with the VASM resources
framework

=head1 SYNOPSIS

    use VASM::Resources::Message qw/findMessageCatalog/;
    
    # Say LANG=de, and there is a catalog in
    # /usr/share/vasm/Some/Project/Messages/de/Messages.xml
    my $catalog = findMessageCatalog(qw/Some Project/);
    # See VASM::Message
    $catalog->Render(ID => 'Hello');
    # Guten Tag!

=head1 DESCRIPTION

VASM::Resources::Message enables the programmer to use VASM::Message i18n
message catalogs with the resource hierarchy defined by L<VASM::Resources>. It
will search for a message catalog under the directory Messages in a given
resource path, in the fashion prescribed by the XDG Base Directory standard,
used by VASM::Resources. More specifically, VASM::Message will attempt to
discern the current locale, and consequently, the appropriate message file
from environment variables that signify the locale (see L<"VARIABLES"> below).
If a message catalog for a given language/country pair cannot be found, the
search will default to a catalog for the language independent of any national
dialect, and, failing that, to a default catalog most likely written in
English.

=head1 FUNCTIONS

=over 4

=item findMessageCatalog

findMessageCatalog accepts a list describing a resource path and seeks a
message catalog therein, according to the rules described above. If
successful, it will return an instance of VASM::Message after using it to
digest the catalog. Otherwise, it will return nothing.

=back

=head1 VARIABLES

These environment variables are searched in order for locale information, and
their values must conform to the POSIX convention consisting of a lowercase
two-letter language code optionally followed by an underscore and an uppercase
two-letter country code, e.g., hi_IN. Encoding information qualifiers such as
'@euro' and '.utf8' are ignored, because the VASM project strictly adheres to
UTF-8:

=over 4

=item *

LC_ALL

=item *

LC_MESSAGES

=item *

LANG

=back

=head1 AUTHORS

hanumizzle L<mailto:hanumizzle@gmail.com> wrote VASM::Resources::Message.

=cut
