# -*- project-name: VASM -*-
package VASM::Resources;

use strict;
use warnings;
use base qw/Exporter/;
# Of course, it's doubtful this will ever be used beyond Vector Linux, BUT
# File::Spec is useful in any case
use File::Spec::Functions;
use File::Path;
use File::BaseDir qw/xdg_data_files xdg_config_files/;

our @EXPORT_OK = qw/findConfigResource findDataResource/;
our $VERSION = '1.02';
my $prefix = 'vasm'; # Prefix directory for resource searches

sub findResource {
  my %args = @_;

  # If no limit on regression was given, assume there is none
  my ($limit, $file, @path) = ($args{Limit} || 0,
                               $args{File},
                               @{ $args{Path} });
  my (@results, $closure);

  # Select a function according to the mode of retrieval:
  if ($args{Type} eq 'data') { $closure = sub { xdg_data_files(@_) } }
  elsif ($args{Type} eq 'config') { $closure = sub { xdg_config_files(@_) } }
  else { return } # Shit value for $args{Type}!

  do {
    @results = $closure->(catfile($prefix, @path, $file));
    # Return the first result, if one could be found, or pop an element off
    # of @path and try again (see below)
    @results and return @results;
  } while (pop @path and @path >= $limit);

  return; # Fall-through
}

sub findConfigResource {
  my %args = @_; $args{Type} = 'config';
  
  return findResource(%args);
}

sub findDataResource {
  my %args = @_; $args{Type} = 'data';
  
  return findResource(%args);
}

# Returns a filename indicating the file at a given resource path (user
# responsible for opening). Of course, this file is always in either the
# config or data user-specific resource tree.
sub makeResource {
  my %args = @_;
  
  my ($file, @path) = ($args{File}, @{ $args{Path} });
  my $closure, $fullPath;
  
  # Select a function according to the mode of internment:
  if ($args{Type} eq 'data') { $closure = sub { xdg_data_home() } }
  elsif ($args{Type} eq 'config') { $closure = sub { xdg_config_home() } }
  else { return } # Shit value for $args{Type}!
  
  # For convenience
  $fullPath = catdir($closure->(), $prefix, @path);
  
  # Make sure all the required directories exist; if not, attempt to create
  # them with mode 0700 (as per the XDG BaseDir spec). If this fails, mkpath
  # will croak. I think we should allow it.
  mkpath($fullPath, undef, 0700) unless -d $fullPath;
  # Testing for write capability should be sufficient?
  return unless -w $fullPath;

  # Now, return the filename. Again, the user is responsible for actually
  # creating the file (not only lazy, also a more flexible scheme, and
  # consistent with the behavior of findResource to boot).
  return catfile($fullPath, $args{File});
}

sub makeConfigResource {
  my %args = @_; $args{Type} = 'config';
  
  return makeResource(%args);
}

sub makeDataResource {
  my %args = @_; $args{Type} = 'data';
  
  return makeResource(%args);
}

1;

__END__

=head1 NAME

VASM::Resources - resource location facility compliant with XDG Base Directory
specification

=head1 SYNOPSIS

    use VASM::Resources qw/findConfigResource
                           findDataResource/;

    # Look for:
    # 
    # ~/.config/vasm/Foo/Bar/Baz/config.xml
    # /etc/xdg/vasm/Foo/Bar/Baz/config.xml
    # ~/.config/vasm/Foo/Bar/config.xml
    # /etc/xdg/Foo/Bar/config.xml
    #
    # ...in that order. If no limit were given, this would
    # walk all the way up to ~/.config/vasm/config.xml and
    # /etc/xdg/vasm/config.xml
    my $configResource =
      findConfigResource(Path => [ qw/Foo Bar Baz/ ],
                         File => 'config.xml',
                         Limit => 2);
    
    # You get the idea...
    my $dataResource =
      findDataResource(Path => [ qw/Some Other Stuff/ ],
                       File => 'foo.png');

=head1 DESCRIPTION

VASM::Resources provides for location of resource files in directories
containing hierarchies of resources; these collections are searched in order,
and matches which most closely match the given request for a resource, found
lower in the filesystem hierarchy, are preferred over more general matches
higher in the hierarchy. The class implements a flexible defaults system
implementing inheritance from parent directories. In the interest of alliance
with standards (and enlightened laziness) VASM::Resources complies with the
XDG Base Directory specification.

=head1 FUNCTIONS

=over 4

=item findConfigResource

findConfigResource accepts a hash constructor as an argument, with which it
locates a given configuration resource file as well as it can. The keys Path
and File are a listref of directory names and the file itself, respectively.
VASM::Resources will conduct a search according to this formula: XDG config
dirs + 'vasm' + path + file. In other words, findConfigResource will look
under each XDG configuration directory, under the prefix directory 'vasm', for
the given path and file. The match that takes highest precedence according to
the specification will be chosen and returned. If it is not possible to find a
matching file using the path applied to the function, findConfigResource, by
default, will pop the list of directories and try again. Thusly, depth in the
resource hierarchy corresponds to the degree of specificity of any given
resource file itself. This regression can be limited with the 'Limit' key in
the arguments hash: if the number of elements in the path list becomes less
than the value of Limit, findConfigResource will give up and return nothing.

=item findDataResource

...is the same as the above, except for data resources instead of
configuration resources.

=back

=head1 VARIABLES

=over 4

=item XDG_CONFIG_HOME

Directory relative to which user-specific configuration files should be
written; defaults to $HOME/.config.

=item XDG_DATA_HOME

Directory relative to which user-specific data should be written; defaults to
'$HOME/.local/share'.

=item XDG_CONFIG_DIRS

Colon-delimited, preference-ordered list of directores relative to which
configuration files should be searched after $XDG_CONFIG_HOME; defaults to
'/etc/xdg'.

=item XDG_DATA_DIRS

Colon-delimited, preference-ordered list of directores relative to which
configuration files should be searched after $XDG_DATA_HOME; defaults to
'/usr/local/share:/usr/share'.

=back

=head1 AUTHORS

hanumizzle L<mailto:hanumizzle@gmail.com> wrote VASM::Resources.

=head1 SEE ALSO

=over

=item *

XDG Base Directory specification:
L<http://standards.freedesktop.org/basedir-spec/>, implemented by
L<File::Basedir>.

=back

=cut
