# -*- project-name: VASM -*-
package VASM::Message;

use strict;
use warnings;
use encoding 'utf8';
use XML::Parser;
use VASM::Tree;

sub new {
  # File and locale arguments?
  my ($self, %args) = @_;

  my $instance = {};
  $instance->{Catalog} = VASM::Tree->new; # The message catalog itself
  $instance->{Identifiers} = VASM::Tree->new; # Quantity identifiers

  bless $instance, $self;

  # Optional file arguments, locale: try to finagle it
  # One file type for messages, another for identifiers

  return $instance;
}

sub Store {
  # $args{qw/ID Quantity/} represents the original message in English and the
  # quantity expressed. $args{Message} is the translated message bearing all
  # these other qualities.
  my ($self, %args) = @_;

  # It is now very simple to store the message in the catalog itself.
  $self->{Catalog}->Store(@args{qw/ID Quantity/},
                          $args{Message}); # Separated for clarity
  
  return;
}

sub SetQuantity {
  my ($self, %args) = @_;

  # Put the new identifier in the Identifiers tree, which is implicitly
  # anchored on either side
  $self->{Identifiers}->Store($args{qw/Quantity/},
                              "^$args{Identifier}\$");

  return;
}
  
sub MatchQuantity {
  my ($self, $quantity) = @_;

  # Iterate over the patterns provided in the Identifiers tree
  for my $key ($self->{Identifiers}->Children()) {
    return $key if
      $quantity =~ $self->{Identifiers}->Retrieve($key);
  }
}

sub DefinedQuantity {
  my ($self, $name) = @_;
  
  return 1 if defined $self->{Identifiers}->Retrieve($name);
}

sub Render {
  my ($self, %args) = @_;
  # This will become a /linguistic/ quantity, as opposed to $args{Quantity},
  # which is a number.
  my $quantity;
  
  # If no /numerical/ quantity was given, assume -1
  $args{Quantity} = -1 unless defined $args{Quantity};

  # Here, give a linguistic identification to the raw number, and return if
  # unsuccessful
  return unless $quantity = $self->MatchQuantity($args{Quantity});

  # Return the formatted message with all remaining arguments
  my $message = $self->{Catalog}->Retrieve($args{ID}, $quantity);
  return sprintf $message,
    # This will flag 'use of undefined value' if the string expects arguments!
    defined $args{Arguments} ? @{ $args{Arguments} } : undef
      if defined $message;
}

# This deletes all the translations of a given message; any other usage would
# be kind of pointless.
sub Delete {
  my ($self, $id) = @_;

  # Prevent destruction of the whole catalog; I doubt this method will be very
  # useful anyway
  return unless defined $id;

  $self->{Catalog}->Delete($id);

  return; # A subroutine
}

# Piggyback on the facilities of XML::Parser to instantiate a message catalog
# from the contents of an XML file whose path is given as input. This is
# pretty hairy, so I suggest you read the POD for XML::Parser (specifically
# the part about the 'Tree' style of parsing) before trying to understand
# this.
sub Parse {
  my ($self, $file) = @_;
  my $parser = XML::Parser->new(Style => 'Tree');
  # Not to be confused with my $ek or my $to. If this fails, XML::Parser will
  # emit an error message and terminate the program. Unfortunately, this
  # message will always be in English. :(
  my $tree = $parser->parsefile($file); my $tags = $tree->[1];
  
  # Main loop: iterate through tags under <catalog>
  for my $index ( grep { ($_ + 2) % 2 } (0..$#{ $tags }) ) {
    next if $tags->[$index] eq '0'; # Skip empty text elements

    if ($tags->[$index] eq 'quantity') {
      # Quantity tags must have the 'match' and 'degree' attributes
      return unless defined $tags->[$index + 1]->[0]->{'match'}
        and defined $tags->[$index + 1]->[0]->{'degree'};
      # Add the new quantity definition
      $self->SetQuantity(Quantity => $tags->[$index + 1]->[0]->{'degree'},
                         Identifier => $tags->[$index + 1]->[0]->{'match'});
    } elsif ($tags->[$index] eq 'message') {
      # Message tags must have the 'id' attribute
      return unless defined $tags->[$index + 1]->[0]->{'id'};
      # The remaining attributes are quantities whose values are the
      # translated messages
      for my $attr (keys %{ $tags->[$index + 1]->[0] }) {
        next if $attr eq 'id'; # Skip the id attribute, duh
        return unless $self->DefinedQuantity($attr); # Quantity must exist
        $self->Store(ID => $tags->[$index + 1]->[0]->{'id'},
                     Quantity => $attr,
                     Message => $tags->[$index + 1]->[0]->{$attr});
      }
    } else { # Unexpected tag!
      return;
    }
  }
  
  return 1; # Success!
}
  
# Because most of the normal divisions of locale support (e.g., LC_COLLATE,
# LC_NUMERIC) do not apply here, we will only honor a certain subset of
# environment variables, in this order: LC_ALL, LC_MESSAGES, LANG. (I believe
# this is the correct protocol; please correct me if I'm wrong.) Because VASM
# strictly uses UTF-8 for all messages, any codeset portion of the variable is
# ignored altogether.
# sub findLocale {
#   # Loop through these variables in turn:
#   for my $var (@ENV{qw/LC_ALL LC_MESSAGES LANG/}) {
#     defined $var and return
#       eval {
#         $var =~ s!\..*?$!!; # Strip away any codeset information
#         # Split language and country (ex: en_GB vs. en_US)
#         my @locale = split /_/, $var;
#         return (Language => shift @locale,
#                 Country => shift @locale);
#       }
#   }
# 
#   return (Language => 'C'); # Default locale
# }

1;

__END__
