# -*- project-name: VASM -*-
package VASM::Message;

use strict;
use warnings;
use encoding 'utf8';
use XML::Parser;
use VASM::Tree;

# XML parsing will do default values for Nullar, Singular, Dual, Trial
# Retrieve will try again without Country field if first attempt at retrieval
# fails

# Methods:
# new
#   Accept file argument (method call to Parse)
#   Aggregate wrt Tree
# Parse

sub new {
  # File and locale arguments?
  my ($self, %args) = @_;

  my $instance = {};
  $instance->{Catalog} = VASM::Tree->new; # The message catalog itself
  $instance->{Identifiers} = VASM::Tree->new; # Quantity identifiers

  bless $instance, $self;

  # Optional file arguments, locale: try to finagle it
  # One file type for messages, another for identifiers

  return $instance;
}

sub Store {
  # $args{qw/ID Quantity/} represents the original message in English and the
  # quantity expressed. $args{Message} is the translated message bearing all
  # these other qualities.
  my ($self, %args) = @_;

  # It is now very simple to store the message in the catalog itself.
  $self->{Catalog}->Store(@args{qw/ID Quantity/},
                          $args{Message}); # Separated for clarity
  
  return;
}

sub SetQuantity {
  my ($self, %args) = @_;

  # Put the new identifier in the Identifiers tree, which is implicitly
  # anchored on either side
  $self->{Identifiers}->Store($args{qw/Quantity/},
                              "^$args{Identifier}\$");

  return;
}
  
sub GetQuantity {
  my ($self, $quantity) = @_;

  # Iterate over the patterns provided in the Identifiers tree
  for my $key ($self->{Identifiers}->Children()) {
    return $key if
      $quantity =~ $self->{Identifiers}->Retrieve($key);
  }

  return; # Fall-through
}

sub Render {
  my ($self, %args) = @_;
  # This will become a /linguistic/ quantity, as opposed to $args{Quantity},
  # which is a number.
  my $quantity;
  
  # If no /numerical/ quantity was given, assume -1
  $args{Quantity} = -1 unless defined $args{Quantity};

  # Here, give a linguistic identification to the raw number, and return if
  # unsuccessful
  return unless $quantity = $self->GetQuantity($args{Quantity});

  # Return the formatted message with all remaining arguments
  my $message = $self->{Catalog}->Retrieve($args{ID}, $quantity);
  return sprintf $message,
    # This will flag 'use of undefined value' if the string expects arguments
    defined $args{Arguments} ? @{ $args{Arguments} } : undef
      if defined $message;

  return; # Fall through
}

# This deletes all the translations of a given message; any other usage would
# be kind of pointless.
sub Delete {
  my ($self, $id) = @_;
  
  $self->{Catalog}->Delete($id);

  return; # A subroutine
}

# Because most of the normal divisions of locale support (e.g., LC_COLLATE,
# LC_NUMERIC) do not apply here, we will only honor a certain subset of
# environment variables, in this order: LC_ALL, LC_MESSAGES, LANG. (I believe
# this is the correct protocol; please correct me if I'm wrong.) Because VASM
# strictly uses UTF-8 for all messages, any codeset portion of the variable is
# ignored altogether.
# sub findLocale {
#   # Loop through these variables in turn:
#   for my $var (@ENV{qw/LC_ALL LC_MESSAGES LANG/}) {
#     defined $var and return
#       eval {
#         $var =~ s!\..*?$!!; # Strip away any codeset information
#         # Split language and country (ex: en_GB vs. en_US)
#         my @locale = split /_/, $var;
#         return (Language => shift @locale,
#                 Country => shift @locale);
#       }
#   }
# 
#   return (Language => 'C'); # Default locale
# }

1;

__END__
