# -*- project-name: VASM -*-
package VASM::Thingie;

use strict;
use warnings;

sub new { 
  my ($self) = @_;
  return bless { Root => {} }, $self; # Ain't Perl grand?
}

sub Retrieve {
  my ($self, @indices) = @_;

  # If any explict arguments were given, use them as indices.
  scalar @indices
    and return nodeTraverse($self->{Root}, @indices)->{Datum};
  # Otherwise, use the stack
  scalar @{ $self->{Stack} }
    and return $self->{Stack}->[-1]->{Datum};

  # Return nothing if the above conditions fall through.
  return;
}

sub Store {
  # Indices should be self-explanatory be now; $datum is the object to be
  # stored.

  my $self = shift; my ($datum, @indices);

  # There are two possible scenarios: if there be one argument, then it is the
  # datum and the stack represnts the 'implicit' indices. Otherwise, the first
  # is a listref containing the indices and the second is the datum
  scalar @_ == 1 and ($datum) = @_
    or ($datum, @indices) = (pop, @{ shift; }); # Here I get a little clever
  
  # This may be considered a subroutine; always returns nothing
  scalar @indices
    and nodeTraverse($self->{Root}, @indices)->{Datum} = $datum; return;
  scalar @{ $self->{Stack} }
    and $self->{Stack}->[-1]->{Datum} = $datum; return;
}
  
sub Push {
  my ($self, @indices) = @_;

  return push @{ $self->{Stack} },
    # If there be anything on the stack already, start at the top of it.
    # Otherwise, begin at the root.
    nodeTraverse(scalar @{ $self->{Stack} } ?
                   $self->{Stack}->[-1] : $self->{Root},
                 @indices);
}

sub Pop {
  my ($self) = @_;
  
  # Simply return if nothing remains on the stack.
  return unless scalar @{ $self->{Stack} };

  # Otherwise, pop both the stack and return the topmost datum.
  return pop( @{ $self->{Stack} } )->{Datum};
}

sub Indices {
  my $self = shift;

  # Simply return all indices that have been pushed onto the stack. This
  # information may be useful in some iterative processes. (Yeah, We Probably
  # Won't Need It, but it's just a few lines and I can always revert.)
  return map { $_->{Name} } @{ $self->{Stack} };
}

sub Children {
  # Indices is a listref in this case, for there are three possible requests
  # one can make to `Children', namely, getting the children under a) a named
  # node, b) Root, and c) the top node in the stack. The listref covers the
  # first two cases, where an empty listref represents the Root, while a
  # /lack/ of arguments covers the last case.
  my ($self, $indices) = @_;
  
  # Return children of specified node if arguments have been supplied, with
  # absolute indexing (i.e., start at Root). Otherwise, use the top of the
  # stack. Implicitly returns nothing if there are no children. (See
  # nodeChildren below.)
  defined $indices
    and return nodeChildren( scalar nodeTraverse($self->{Root}, @$indices) );
  scalar @{ $self->{Stack} } # Something has to be on the stack!
    and return nodeChildren( $self->{Stack}->[-1] );

  # If the above conditions fall-through, explictly return nothing.
  return;
}

sub indexIdentify {
  my ($index) = @_;
  
  # If $index be a reference, its referent is unequivocally a hash index.
  ref $index and return 'HASH';
  # Otherwise, if $index be a number, then we will assume it is an array index
  $index =~ m!^-?\d*$! and return 'ARRAY';
  # In any other case, it will be considered a hash index
  return 'HASH';
}

sub nodeGet {
  my ($node, $index) = @_;

  # Initialize the node, if it be necessary. There are two cases where a node
  # may be considered uninitialized: if the node itself is undefined, and in
  # cases where it has been defined with a hashref, but lacks a Children
  # member. This will fix that in either situation.
  unless (defined $node->{Children}) {
    # Find out what kind of index we have, so that we may properly create the
    # new node.
    my $type = indexIdentify($index);

    # Now create the node. We can worry about the Datum member later; it's not
    # even used in the root, anyway.
    $node->{Children} =
      eval {
        $type eq 'ARRAY' and return [];
        $type eq 'HASH' and return {};
      };
  }

  # Ok, now that we're sure the originating node exists, we may now return the
  # the specified node under it. Look familiar? 
  $index = $$index if ref $index; # Resolve reference if necessary

  # In either case, create the node if necessary, using the $index parameter
  # to initialize the Name field. (The Indices method needs to know the names
  # of the nodes.) Then return the node.
  if (ref $node->{Children} eq 'ARRAY') {
    $node->{Children}->[$index] = { Name => $index }
      unless defined $node->{Children}->[$index];
    return $node->{Children}->[$index];
  } elsif (ref $node->{Children} eq 'HASH') {
    $node->{Children}->{$index} = { Name => $index }
      unless defined $node->{Children}->{$index};
    return $node->{Children}->{$index};
  }
}

sub nodeTraverse {
  my ($node, @indices) = @_;    # The @indices through which we will traverse
  my @nodes;                    # Used to keep track of nodes

  # If there were no arguments, just return the node that was passed in.
  return $node unless @indices;
  
  for my $index (@indices) {
    $node = nodeGet($node, $index);
    push @nodes, $node;
  }
  
  # If a scalar value is desired, return the last node found. Otherwise,
  # return the whole lot.
  wantarray ? return @nodes : return $node;
}

sub nodeChildren {
  my ($node) = @_;
  
  # If the Children member of the node be an array, just return a list of all
  # the indices. In the event of a hashref Children, return the keys.
  ref $node->{Children} eq 'ARRAY' and return (0..$#{ $node->{Children} });
  ref $node->{Children} eq 'HASH' and return keys %{ $node->{Children} };
}
  
1;

__END__

=head1 NAME

VASM::Thingie - generalized support for thingies

=head1 DESCRIPTION

VASM::Thingie provides an object-oriented interface to data structures of
heterogenous, multilevel array and hash composition. We like to call such
hierarchical constructs 'thingies'.

As well as permitting the free mixture of both of Perl's mutable data types in
a complex tiered formation, a VASM::Thingie instance can manage a stack
representing a pathway through its nodes, which is useful for iterative tasks.

=head1 METHODS

=over

=item new

new produces an instance of VASM::Thingie and expects a class invocant. There
are no cloning semantics for instance invocation, which will probably raise an
error. new accepts no particular arguments beyond the implicit invocant.

    my $thingie = VASM::Thingie->new;

=item Store

The Store method acts polymorphically according to the number of arguments. If
two arguments be applied, the first is a listref of indices in the thingie,
and the second is a datum to store. The class will automatically determine the
type of each index, with a regular expression: sequences of digits will be
considered list indices, and anything else, hash keys. If necessary, it is
possible to coerce integer values to hash keys by supplying a reference,
rather than a direct value. If there be only one argument, then it is a datum
and the stack serves as the implicit indices; the datum will be stored in the
topmost node. (See Push and Pop below.)

    $thingie->Store([ qw/Lao Province/ ], 'Pakse');

    $thingie->Push([ qw/Indian State/ ]);
    $thingie->Store('Kerala');

    $thingie->Store([ \(1, 2, 3) ], 'All hash keys!');

=item Retrieve

The instance method Retrieve accepts a list of indices defining the route
through the data structure, and returns the datum at the terminal point. 

    my $datum = $thingie->Retrieve('foo', 'bar', \36, 42, 'baz');

=item Push

An instance of VASM::Thingie can maintain a stack of indices which
collectively indicate a particular member of the thingie, the bottom of the
stack naming the node directly attached to the root, the top naming the
deepest. This stack is useful for iterative processing of the thingie.

The Push method pushes any number of indices onto this internal stack and
returns the number of elements so pushed.

    my $tin = $thingie->Push(qw/Brahma Vishnu Shiva/);
    print "$tin\n"; # Ek, do, tin...

=item Pop

The Pop method pops the internal stack described above and returns the datum
of the former topmost element in the stock.

    my $datum = $thingie->Pop;

If no elements remain on the stack, Pop will return nothing.

=item Indices

This method simply returns a list of all indices that have been pushed onto
the stick. Using the above example for Push:

  local $\ = "\n";
  my @trimurti = $thingie->Indices;
  print "@trimurti"; # --> Brahma Vishnu Shiva

=item Children

The Children method, when supplied a listref of indices, will return a list of
all the children of a given node, navigating downwards from the root; an empty
listref will yield the children of the root. If the children be stored in a
hashref, this is a list of keys; for listrefs, a range of all possible
indices. With no such arguments, the children of the topmost stack member are
named. If there are no children, Children returns nothing.

=back

=head1 TODO

Map method? (General iteration.)

=head1 AUTHORS

hanumizzle <hanumizzle@gmail.com> wrote VASM::Thingie.

=cut
