/* 
**  LockMe! - Periodically locks your Palm(Pilot) at a specified time
**  Copyright (C) 1998 by Ren Witte <witte@acm.org>
**
**  $Id: LockMe.c,v 1.2 1998/07/03 00:11:36 rene Exp $
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
**  ChangeLog:
**      $Log: LockMe.c,v $
**      Revision 1.2  1998/07/03 00:11:36  rene
**      Fixed incompatibility with built-in DateBook (and other applications
**      that ignore appStopEvents):
**      LockMe now uses the sysAppLaunchCmdDisplayAlarm.
**
**      Revision 1.1  1998/06/19 02:58:07  rene
**      Locking interval is now selectable.
**      New option "Lock after reset" added.
**      LockMe! now correctly handles system resets.
**
**      Revision 1.0  1998/05/09 18:30:44  rene
**      First release.
**
**      Revision 0.1.1.1  1998/05/09 18:28:53  rene
**      Initial import.
**
*/

#include <Pilot.h>
#include <Password.h>
#include <System/SysAll.h>
#include <UI/UIAll.h>
#include "callback.h"
#include <stdio.h>
#include "LockMeRsc.h"

/* defines */
#define LOCKMEID 'LOCK'          /* LockMe! Program ID                   */
#define SECAPPID 'secr'          /* ID of the built-in security app      */
#define NUMHOUR 9                /* number of supported time deltas      */
#define PREFVERSION 2            /* LockMe!'s preferences version        */

/* (Saved) Preferences */
typedef struct {
     Boolean lockme_enabled;     /* LockMe turned on?                    */
     Boolean hide_private;       /* Hide private records at lock time?   */
     Boolean lock_after_reset;   /* Lock after a system reset?           */
     Boolean lock_set;           /* Lock currently set? (internal use)   */
     UInt time;                  /* Selected lock start time             */
     UInt delta;                 /* Selected time between locks          */
     Int lock_time;              /* Hour of the next lock (internal use) */
} LockMe_prefs_t;

/* Global variables */
static LockMe_prefs_t LockMe_prefs;
static UInt hour_conv[ NUMHOUR ] = { 1, 2, 3, 4, 6, 8, 12, 24 }; 


static void lockme( LockMe_prefs_t LM_prefs )
/* does the actual locking by calling the security app */
{
     DmSearchStateType sst;
     UInt              CardDB;
     LocalID           dDB;

     if ( LM_prefs.hide_private )
	  /* hide private records */
	  PrefSetPreference( prefHidePrivateRecords, true );
     /* only lock if password exists and not already locked */
     if ( !PrefGetPreference( prefDeviceLocked ) && PwdExists() ) {
	  DmGetNextDatabaseByTypeCreator( true, &sst, sysFileTApplication, 
					  SECAPPID, true, &CardDB, &dDB );
	  SysUIAppSwitch( CardDB, dDB, sysAppLaunchCmdSystemLock, 0 );
     } else {
	  SysSleep( false, false );
     }
}


static int schedule_lock( LockMe_prefs_t LM_prefs )
/* sets the alarm to let LockMe! go off for the selected time */
/* returns the hour for which the alarm was set               */
{
     DateTimeType      lockme_time;
     ULong             lockme_time_seconds;
     ULong             current_time_seconds;
     UInt              CardDB;
     LocalID           dDB;
     Err               err;
     DmSearchStateType searchInfo;

     current_time_seconds = TimGetSeconds();
     TimSecondsToDateTime( current_time_seconds, &lockme_time );
     lockme_time.minute = 0;
     lockme_time.second = 1;
     if( LM_prefs.lock_time < 0 ) {
          /* first lock with this start time */
          if ( lockme_time.hour >= LM_prefs.time )
               /* lock tomorrow */
               TimAdjust( &lockme_time, (Long)86400 );
          lockme_time.hour = LM_prefs.time;
     } else {
	  if( lockme_time.hour < LM_prefs.lock_time )
	       TimAdjust( &lockme_time, (Long)-86400 );
	  lockme_time.hour = LM_prefs.lock_time;
	  /* adjust the time with the delta until we have a valid lock time */
	  while( TimDateTimeToSeconds( &lockme_time ) <= current_time_seconds )
	       TimAdjust( &lockme_time, (Long)LM_prefs.delta*3600 );
     }
     lockme_time_seconds = TimDateTimeToSeconds( &lockme_time );
     DmGetNextDatabaseByTypeCreator( true, &searchInfo, sysFileTApplication,
				     LOCKMEID, true, &CardDB, &dDB);
     err = AlmSetAlarm( CardDB, dDB, 0, lockme_time_seconds, 0 );

     return lockme_time.hour;
}


static void LockMe_alarm(void)
/* LockMe! started by Alarm handler */
{
      LockMe_prefs_t    LM_prefs;
      Word              prefs_size;

      prefs_size = sizeof( LM_prefs );
      PrefGetAppPreferences( LOCKMEID, 0, &LM_prefs, &prefs_size, true );
      if( LM_prefs.lockme_enabled )
	   /* schedule another lock */
	   LM_prefs.lock_time = schedule_lock( LM_prefs );
      else
	   /* no more locks... */
	   LM_prefs.lock_set = false;

      PrefSetAppPreferences( LOCKMEID, 0, PREFVERSION, &LM_prefs, sizeof( LM_prefs ), true );
      lockme( LM_prefs );
}


static void LockMe_reset(void)
/* LockMe! started after a system reset */
{
     LockMe_prefs_t    LM_prefs;
     Word              prefs_size;
     ULong             lockme_time_seconds;
     UInt              CardDB;
     LocalID           dDB;
     Err               err;
     DmSearchStateType searchInfo;

     prefs_size = sizeof( LM_prefs );
     PrefGetAppPreferences( LOCKMEID, 0, &LM_prefs, &prefs_size, true );

     if( LM_prefs.lock_after_reset && LM_prefs.lock_set ) {
	  /* we can't lock the device just right now, */
	  /* so set an alarm and lock then...         */
	  lockme_time_seconds = TimGetSeconds() + 12; /* be nice to HackMaster */
	  DmGetNextDatabaseByTypeCreator( true, &searchInfo, sysFileTApplication,
					  LOCKMEID, true, &CardDB, &dDB);
	  err = AlmSetAlarm( CardDB, dDB, 0, lockme_time_seconds, 0 );
     } else if( LM_prefs.lock_set ) {
	  /* after a reset we have to re-set (no pun intended) our alarm */
	  schedule_lock( LM_prefs );
     }
}
     

static void MakeLabel( CharPtr label, Char format1[], Char format2[], UInt time )
/* little helper function because formats don't seem to work quite right */
{
     if( time < 10 )
	  StrPrintF( label, format1, time );
     else /* why doesn't precision (%.2d) work here? */
	  StrPrintF( label, format2, time );
}


static Boolean MainFormHandleEvent( EventPtr e )
/* form event loop */
{
    Boolean        handled = false;
    FormPtr        frm;
    UInt           i;
    UInt           selected_time;
    UInt           selected_delta;
    UInt           list_top_time;
    UInt           list_top_delta;
    Char           time_label[ 6 ];
    Char           delta_label[ 3 ];

    CALLBACK_PROLOGUE

    switch (e->eType) {
    case frmOpenEvent:
	 /* initialize the main form */
	frm = FrmGetActiveForm();
	FrmDrawForm(frm);
	/* set the checkboxes */
	CtlSetValue( FrmGetObjectPtr( frm, FrmGetObjectIndex( frm, ENABLED )), 
		     LockMe_prefs.lockme_enabled );
	CtlSetValue( FrmGetObjectPtr( frm, FrmGetObjectIndex( frm, HIDE )), 
		     LockMe_prefs.hide_private );
	CtlSetValue( FrmGetObjectPtr( frm, FrmGetObjectIndex( frm, RESET )), 
		     LockMe_prefs.lock_after_reset );
	/* set the starting time selection */
	MakeLabel( time_label, "0%d:00", "%d:00", LockMe_prefs.time );
	CtlSetLabel( FrmGetObjectPtr( frm, FrmGetObjectIndex( frm, P_SELTIME )),
		     time_label );
	LstSetSelection( FrmGetObjectPtr( frm, FrmGetObjectIndex( frm, SELTIME )),
			 LockMe_prefs.time );
	list_top_time = LockMe_prefs.time;
	if( list_top_time >= 2 )
	     list_top_time -= 2;
	else
	     list_top_time -= list_top_time;
	LstSetTopItem(  FrmGetObjectPtr( frm, FrmGetObjectIndex( frm, SELTIME )), list_top_time );
	/* set the time delta selection */
	MakeLabel( delta_label, "0%d", "%d", LockMe_prefs.delta );
	CtlSetLabel( FrmGetObjectPtr( frm, FrmGetObjectIndex( frm, P_DELTA )),
		     delta_label );
	/* ok, for the "delta" list we'll have to find the right index */
	i = 0;
	while( (i < NUMHOUR) && (hour_conv[i] != LockMe_prefs.delta ) )
	     i++;
	LstSetSelection( FrmGetObjectPtr( frm, FrmGetObjectIndex( frm, DELTA )), i );
	list_top_delta = i;
	if ( list_top_delta >= 2 )
	     list_top_delta -= 2;
	else
	     list_top_delta -= list_top_delta;
	LstSetTopItem( FrmGetObjectPtr( frm, FrmGetObjectIndex( frm, DELTA )), list_top_delta );

	handled = true;
	break;

    case menuEvent:
	MenuEraseStatus(NULL);

	switch(e->data.menu.itemID) {
	}

    	handled = true;
	break;

    case ctlSelectEvent:
	 
	switch(e->data.ctlSelect.controlID) {
	case ENABLED: /* enable/disable LockMe */
	     if (e->data.ctlSelect.on) {
		  LockMe_prefs.lockme_enabled = true;
		  LockMe_prefs.lock_set = true;
		  LockMe_prefs.lock_time = schedule_lock( LockMe_prefs );
	     } else {
		  LockMe_prefs.lockme_enabled = false;
	     }

	     handled = true;	
	     break;

	case HIDE: /* also hide private records? */
	     if (e->data.ctlSelect.on) {
		  LockMe_prefs.hide_private = true;
	     } else {
		  LockMe_prefs.hide_private = false;
	     }

	     handled = true;	
	     break;

	case RESET: /* lock after a reset? */
	     if (e->data.ctlSelect.on) {
		  LockMe_prefs.lock_after_reset = true;
	     } else {
		  LockMe_prefs.lock_after_reset = false;
	     }

	     handled = true;	
	     break;

	case LOCKNOW: /* LockMe now! */
	     lockme( LockMe_prefs );
	     handled = true;
	     break;

	}

	break;
	
    case popSelectEvent:
	 frm = FrmGetActiveForm();
	 switch(e->data.ctlSelect.controlID) {
	 case P_SELTIME:  /* time selected */
	      selected_time = LstGetSelection( FrmGetObjectPtr( frm, FrmGetObjectIndex( frm, SELTIME )));
	      LockMe_prefs.time = selected_time;

	      MakeLabel( time_label, "0%d:00", "%d:00", LockMe_prefs.time );
	      CtlSetLabel( FrmGetObjectPtr( frm, FrmGetObjectIndex( frm, P_SELTIME )), time_label );
	      LockMe_prefs.lock_time = -1;         /* new start time! */
	      if ( LockMe_prefs.lockme_enabled )
		   LockMe_prefs.lock_time = schedule_lock( LockMe_prefs );
	      handled = true;	
	      break;

	 case P_DELTA: /* new time delta selected */
	      selected_delta = LstGetSelection( FrmGetObjectPtr( frm, FrmGetObjectIndex( frm, DELTA )));
	      LockMe_prefs.delta = hour_conv[ selected_delta ];
	      MakeLabel( delta_label, "0%d", "%d", LockMe_prefs.delta );
	      CtlSetLabel( FrmGetObjectPtr( frm, FrmGetObjectIndex( frm, P_DELTA )), delta_label );
	      handled = true;	
	      break;

	 default:
	      break;
	 }
    
    default:
	 break;
    }
    
    CALLBACK_EPILOGUE

    return handled;
}


static Boolean ApplicationHandleEvent(EventPtr e)
{
    FormPtr frm;
    Word    formId;
    Boolean handled = false;

    if (e->eType == frmLoadEvent) {
	formId = e->data.frmLoad.formID;
	frm = FrmInitForm(formId);
	FrmSetActiveForm(frm);

	switch(formId) {
	case MainForm:
	    FrmSetEventHandler(frm, MainFormHandleEvent);
	    break;
	}
	handled = true;
    }

    
    return handled;
}



static Word StartApplication(void)
/* Get preferences */
{
     Word prefs_size;
 
     prefs_size = sizeof( LockMe_prefs );
     LockMe_prefs.lockme_enabled = false;
     LockMe_prefs.hide_private = false;
     LockMe_prefs.lock_after_reset = false;
     LockMe_prefs.lock_set = false;
     LockMe_prefs.time = 0;
     LockMe_prefs.delta = 24;
     LockMe_prefs.lock_time = -1;
     PrefGetAppPreferences( LOCKMEID, 0, &LockMe_prefs, &prefs_size, true );
     FrmGotoForm(MainForm);

     return 0;
}


static void StopApplication(void)
/* close forms, save app preferences */
{
     PrefSetAppPreferences( LOCKMEID, 0, PREFVERSION, &LockMe_prefs, sizeof( LockMe_prefs ), true );
     FrmSaveAllForms();
     FrmCloseAllForms();
}


static void EventLoop(void)
/* The main event loop */
{
    Word err;
    EventType e;

    do {
	EvtGetEvent(&e, evtWaitForever);
	if (! SysHandleEvent (&e))
	    if (! MenuHandleEvent (NULL, &e, &err))
		if (! ApplicationHandleEvent (&e))
		    FrmDispatchEvent (&e);
    } while (e.eType != appStopEvent);
}


DWord PilotMain(Word cmd, Ptr cmdPBP, Word launchFlags)
{
    Word err;

    if (cmd == sysAppLaunchCmdNormalLaunch) {

	err = StartApplication();
	if (err) return err;

	EventLoop();
	StopApplication();

    } else if (cmd == sysAppLaunchCmdDisplayAlarm) {
	 LockMe_alarm();
    } else if (cmd == sysAppLaunchCmdSystemReset) {
	 LockMe_reset();
    } else {	 
	return sysErrParamErr;	 
    }

    return 0;
}
