/* tri2obj.c -- read in a .ele/.node file pair generated by the triangle 
 *              program and output a simple Wavefront .obj file.
 *
 * Written by Curtis Olson, started October 1997.
 *
 * Copyright (C) 1997  Curtis L. Olson  - curt@infoplane.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: tri2obj.c,v 1.16 1998/05/23 15:20:41 curt Exp $
 * (Log is kept at end of this file)
 */


#include <stdio.h>
#include <stdlib.h>   /* for atoi() */
#include <string.h>
#include <sys/stat.h> /* for stat() */
#include <unistd.h>   /* for stat() */

#include "tri2obj.h"

#include <Include/fg_constants.h>
#include <Include/fg_types.h>
#include <Bucket/bucketutils.h>

#include <Math/fg_geodesy.h>
#include <Math/mat3.h>
#include <Math/polar3d.h>


int nodecount, tricount;
int normalcount = 0;
static fgCartesianPoint3d nodes[MAX_NODES];
static int tris[MAX_TRIS][3];

static double normals[MAX_NODES][3];

struct fgBUCKET my_index;
struct fgBUCKET ne_index, nw_index, sw_index, se_index;
struct fgBUCKET north_index, south_index, east_index, west_index;

/* convert a geodetic point lon(arcsec), lat(arcsec), elev(meter) to
 * a cartesian point */
fgCartesianPoint3d geod_to_cart(double geod[3]) {
    fgCartesianPoint3d cp;
    fgPolarPoint3d pp;
    double gc_lon, gc_lat, sl_radius;

    /* printf("A geodetic point is (%.2f, %.2f, %.2f)\n", 
	   geod[0], geod[1], geod[2]); */

    gc_lon = geod[0]*ARCSEC_TO_RAD;
    fgGeodToGeoc(geod[1]*ARCSEC_TO_RAD, geod[2], &sl_radius, &gc_lat);

    /* printf("A geocentric point is (%.2f, %.2f, %.2f)\n", gc_lon, 
	   gc_lat, sl_radius+geod[2]); */

    pp.lon = gc_lon;
    pp.lat = gc_lat;
    pp.radius = sl_radius+geod[2];
    cp = fgPolarToCart3d(pp);
    
    /* printf("A cart point is (%.8f, %.8f, %.8f)\n", cp.x, cp.y, cp.z); */

    return(cp);
}


/* given three points defining a triangle, calculate the normal */
void calc_normal(fgCartesianPoint3d p1, fgCartesianPoint3d p2, 
		 fgCartesianPoint3d p3, double normal[3])
{
    double v1[3], v2[3];
    double temp;

    v1[0] = p2.x - p1.x; v1[1] = p2.y - p1.y; v1[2] = p2.z - p1.z;
    v2[0] = p3.x - p1.x; v2[1] = p3.y - p1.y; v2[2] = p3.z - p1.z;

    MAT3cross_product(normal, v1, v2);
    MAT3_NORMALIZE_VEC(normal, temp);

/*  printf("  Normal = %.2f %.2f %.2f\n", normal[0], normal[1], normal[2]); */
}


/* return the index of all triangles containing the specified node */
void find_tris(int n, int *t1, int *t2, int *t3, int *t4, int *t5) {
    int i;

    *t1 = *t2 = *t3 = *t4 = *t5 = 0;

    i = 1;
    while ( i <= tricount ) {
        if ( (n == tris[i][0]) || (n == tris[i][1]) || (n == tris[i][2]) ) {
            if ( *t1 == 0 ) {
		*t1 = i;
            } else if ( *t2 == 0 ) {
		*t2 = i;
            } else if ( *t3 == 0 ) {
		*t3 = i;
            } else if ( *t4 == 0 ) {
		*t4 = i;
	    } else {
		*t5 = i;
	    }
        }
        i++;
    }
}


/* return the file base name ( foo/bar/file.ext = file.ext ) */
void extract_file(char *in, char *base) {
    int len, i;

    len = strlen(in);

    i = len - 1;
    while ( (i >= 0) && (in[i] != '/') ) {
	i--;
    }

    in += (i + 1);
    strcpy(base, in);
}


/* return the file path name ( foo/bar/file.ext = foo/bar ) */
void extract_path(char *in, char *base) {
    int len, i;

    len = strlen(in);
    strcpy(base, in);

    i = len - 1;
    while ( (i >= 0) && (in[i] != '/') ) {
	i--;
    }

    base[i] = '\0';
}


/* check if a file exists */
int file_exists(char *file) {
    struct stat stat_buf;
    int result;

    printf("checking %s ... ", file);

    result = stat(file, &stat_buf);

    if ( result != 0 ) {
	/* stat failed, no file */
	printf("not found.\n");
	return(0);
    } else {
	/* stat succeeded, file exists */
	printf("exists.\n");
	return(1);
    }
}


/* check to see if a shared object exists */
int shared_object_exists(char *basepath, char *ext, char *file) {
    char scene_path[256];
    long int index;

    if ( strcmp(ext, ".sw") == 0 ) {
	fgBucketGenBasePath(&my_index, scene_path);
	index = fgBucketGenIndex(&my_index);
	sprintf(file, "%s/%s/%ld.1.sw", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
	fgBucketGenBasePath(&west_index, scene_path);
	index = fgBucketGenIndex(&west_index);
	sprintf(file, "%s/%s/%ld.1.se", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
	fgBucketGenBasePath(&sw_index, scene_path);
	index = fgBucketGenIndex(&sw_index);
	sprintf(file, "%s/%s/%ld.1.ne", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
	fgBucketGenBasePath(&south_index, scene_path);
	index = fgBucketGenIndex(&south_index);
	sprintf(file, "%s/%s/%ld.1.nw", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
    }

    if ( strcmp(ext, ".se") == 0 ) {
	fgBucketGenBasePath(&my_index, scene_path);
	index = fgBucketGenIndex(&my_index);
	sprintf(file, "%s/%s/%ld.1.se", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
	fgBucketGenBasePath(&east_index, scene_path);
	index = fgBucketGenIndex(&east_index);
	sprintf(file, "%s/%s/%ld.1.sw", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
	fgBucketGenBasePath(&se_index, scene_path);
	index = fgBucketGenIndex(&se_index);
	sprintf(file, "%s/%s/%ld.1.nw", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
	fgBucketGenBasePath(&south_index, scene_path);
	index = fgBucketGenIndex(&south_index);
	sprintf(file, "%s/%s/%ld.1.ne", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
    }

    if ( strcmp(ext, ".ne") == 0 ) {
	fgBucketGenBasePath(&my_index, scene_path);
	index = fgBucketGenIndex(&my_index);
	sprintf(file, "%s/%s/%ld.1.ne", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
	fgBucketGenBasePath(&east_index, scene_path);
	index = fgBucketGenIndex(&east_index);
	sprintf(file, "%s/%s/%ld.1.nw", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
	fgBucketGenBasePath(&ne_index, scene_path);
	index = fgBucketGenIndex(&ne_index);
	sprintf(file, "%s/%s/%ld.1.sw", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
	fgBucketGenBasePath(&north_index, scene_path);
	index = fgBucketGenIndex(&north_index);
	sprintf(file, "%s/%s/%ld.1.se", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
    }

    if ( strcmp(ext, ".nw") == 0 ) {
	fgBucketGenBasePath(&my_index, scene_path);
	index = fgBucketGenIndex(&my_index);
	sprintf(file, "%s/%s/%ld.1.nw", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
	fgBucketGenBasePath(&west_index, scene_path);
	index = fgBucketGenIndex(&west_index);
	sprintf(file, "%s/%s/%ld.1.ne", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
	fgBucketGenBasePath(&nw_index, scene_path);
	index = fgBucketGenIndex(&nw_index);
	sprintf(file, "%s/%s/%ld.1.se", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
	fgBucketGenBasePath(&north_index, scene_path);
	index = fgBucketGenIndex(&north_index);
	sprintf(file, "%s/%s/%ld.1.sw", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
    }

    if ( strcmp(ext, ".south") == 0 ) {
	fgBucketGenBasePath(&my_index, scene_path);
	index = fgBucketGenIndex(&my_index);
	sprintf(file, "%s/%s/%ld.1.south", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
	fgBucketGenBasePath(&south_index, scene_path);
	index = fgBucketGenIndex(&south_index);
	sprintf(file, "%s/%s/%ld.1.north", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
    }

    if ( strcmp(ext, ".north") == 0 ) {
	fgBucketGenBasePath(&my_index, scene_path);
	index = fgBucketGenIndex(&my_index);
	sprintf(file, "%s/%s/%ld.1.north", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
	fgBucketGenBasePath(&north_index, scene_path);
	index = fgBucketGenIndex(&north_index);
	sprintf(file, "%s/%s/%ld.1.south", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
    }

    if ( strcmp(ext, ".west") == 0 ) {
	fgBucketGenBasePath(&my_index, scene_path);
	index = fgBucketGenIndex(&my_index);
	sprintf(file, "%s/%s/%ld.1.west", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
	fgBucketGenBasePath(&west_index, scene_path);
	index = fgBucketGenIndex(&west_index);
	sprintf(file, "%s/%s/%ld.1.east", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
    }

    if ( strcmp(ext, ".east") == 0 ) {
	fgBucketGenBasePath(&my_index, scene_path);
	index = fgBucketGenIndex(&my_index);
	sprintf(file, "%s/%s/%ld.1.east", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
	fgBucketGenBasePath(&east_index, scene_path);
	index = fgBucketGenIndex(&east_index);
	sprintf(file, "%s/%s/%ld.1.west", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
    }

    if ( strcmp(ext, ".body") == 0 ) {
	fgBucketGenBasePath(&my_index, scene_path);
	index = fgBucketGenIndex(&my_index);
	sprintf(file, "%s/%s/%ld.1.body", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
    }

    return(0);
}


/* given a file pointer, read all the vn (normals from it) */
void read_normals(FILE *fp) {
    char line[256];

    while ( fgets(line, 250, fp) != NULL ) {
	if ( strncmp(line, "vn ", 3) == 0 ) {
	    sscanf( line, "vn %lf %lf %lf\n", 
		    &normals[normalcount][0], 
		    &normals[normalcount][1], 
		    &normals[normalcount][2] );
	    /*
	    printf("read_normals(%d) %.2f %.2f %.2f %s", normalcount, 
		   normals[normalcount][0], normals[normalcount][1], 
		   normals[normalcount][2], line);
		   */
	    normalcount++;
	}
    }
}


/* my custom file opening routine ... don't open if a shared edge or
 * vertex alread exists */
FILE *my_open(char *basename, char *basepath, char *ext) {
    FILE *fp;
    char filename[256];

    /* check if a shared object already exists */
    if ( shared_object_exists(basepath, ext, filename) ) {
	/* not an actual file open error, but we've already got the
         * shared edge, so we don't want to create another one */
	fp = fopen(filename, "r");
	printf("Opening %s\n", filename);
	return(fp);
    } else {
	/* open the file */
	printf("not opening\n");
	return(NULL);
    }
}


/* Initialize a new mesh structure */
void triload(char *basename, char *basepath) {
    char nodename[256], elename[256];
    double n[3];
    FILE *ne, *nw, *se, *sw, *north, *south, *east, *west;
    FILE *node, *ele;
    int dim, junk1, junk2;
    int i;

    ne = my_open(basename, basepath, ".ne");
    read_normals(ne);
    fclose(ne);

    nw = my_open(basename, basepath, ".nw");
    read_normals(nw);
    fclose(nw);

    se = my_open(basename, basepath, ".se");
    read_normals(se);
    fclose(se);

    sw = my_open(basename, basepath, ".sw");
    read_normals(sw);
    fclose(sw);

    north = my_open(basename, basepath, ".north");
    read_normals(north);
    fclose(north);

    south = my_open(basename, basepath, ".south");
    read_normals(south);
    fclose(south);

    east = my_open(basename, basepath, ".east");
    read_normals(east);
    fclose(east);

    west = my_open(basename, basepath, ".west");
    read_normals(west);
    fclose(west);

    strcpy(nodename, basename);
    strcat(nodename, ".node");
    strcpy(elename, basename);
    strcat(elename, ".ele");

    printf("Loading node file:  %s ...\n", nodename);
    if ( (node = fopen(nodename, "r")) == NULL ) {
	printf("Cannot open file '%s'\n", nodename);
	exit(-1);
    }

    fscanf(node, "%d %d %d %d", &nodecount, &dim, &junk1, &junk2);

    if ( nodecount > MAX_NODES - 1 ) {
	printf("Error, too many nodes, need to increase array size\n");
	exit(-1);
    } else {
	printf("    Expecting %d nodes\n", nodecount);
    }

    for ( i = 1; i <= nodecount; i++ ) {
	fscanf(node, "%d %lf %lf %lf %d\n", &junk1, 
	       &n[0], &n[1], &n[2], &junk2);
	/* printf("%d %.2f %.2f %.2f\n", junk1, n[0], n[1], n[2]); */
	nodes[i] = geod_to_cart(n);
	/* printf("%d %.2f %.2f %.2f\n", 
	       junk1, nodes[i].x, nodes[i].y, nodes[i].z); */
    }

    fclose(node);

    printf("Loading element file:  %s ...\n", elename);
    if ( (ele = fopen(elename, "r")) == NULL ) {
	printf("Cannot open file '%s'\n", elename);
	exit(-1);
    }

    fscanf(ele, "%d %d %d", &tricount, &junk1, &junk2);

    if ( tricount > MAX_TRIS - 1 ) {
	printf("Error, too many elements, need to increase array size\n");
	exit(-1);
    } else {
	printf("    Expecting %d elements\n", tricount);
    }

    for ( i = 1; i <= tricount; i++ ) {
	fscanf(ele, "%d %d %d %d\n", &junk1, 
	       &tris[i][0], &tris[i][1], &tris[i][2]);
	/* printf("%d %d %d %d\n", junk1, tris[i][0], tris[i][1], tris[i][2]);*/
    }

    fclose(ele);
}


/* dump in WaveFront .obj format */
void dump_obj(char *basename) {
    char objname[256];
    double n1[3], n2[3], n3[3], n4[3], n5[3], norm[3], temp;
    FILE *obj;
    int i, t1, t2, t3, t4, t5, count;

    strcpy(objname, basename);
    strcat(objname, ".obj");

    printf("Dumping to file:  %s ...\n", objname);

    obj = fopen(objname, "w");

    /* dump vertices */
    printf("  writing vertices\n");
    for ( i = 1; i <= nodecount; i++ ) {
	fprintf(obj, "v %.6f %.6f %.6f\n", 
		nodes[i].x, nodes[i].y, nodes[i].z);
    }

    printf("  calculating and writing normals\n");
    printf("  First %d normals taken from shared files.\n", normalcount);
						     
    /* calculate and generate normals */
    for ( i = 1; i <= nodecount; i++ ) {

	if ( i <= normalcount ) {
	    /* use precalculated (shared) normal */
	    norm[0] = normals[i-1][0];
	    norm[1] = normals[i-1][1];
	    norm[2] = normals[i-1][2];
	} else {
	    /* printf("Finding normal\n"); */

	    find_tris(i, &t1, &t2, &t3, &t4, &t5);

	    n1[0] = n1[1] = n1[2] = 0.0;
	    n2[0] = n2[1] = n2[2] = 0.0;
	    n3[0] = n3[1] = n3[2] = 0.0;
	    n4[0] = n4[1] = n4[2] = 0.0;
	    n5[0] = n5[1] = n5[2] = 0.0;

	    count = 1;
	    calc_normal(nodes[tris[t1][0]], nodes[tris[t1][1]], 
			nodes[tris[t1][2]], n1);

	    if ( t2 > 0 ) {
		calc_normal(nodes[tris[t2][0]], nodes[tris[t2][1]], 
			    nodes[tris[t2][2]], n2);
		count = 2;
	    }

	    if ( t3 > 0 ) {
		calc_normal(nodes[tris[t3][0]], nodes[tris[t3][1]],
			    nodes[tris[t3][2]], n3);
		count = 3;
	    }

	    if ( t4 > 0 ) {
		calc_normal(nodes[tris[t4][0]], nodes[tris[t4][1]],
			    nodes[tris[t4][2]], n4);
		count = 4;
	    }

	    if ( t5 > 0 ) {
		calc_normal(nodes[tris[t5][0]], nodes[tris[t5][1]],
			    nodes[tris[t5][2]], n5);
		count = 5;
	    }

	    /* printf("  norm[2] = %.2f %.2f %.2f\n", n1[2], n2[2], n3[2]); */

	    norm[0] = ( n1[0] + n2[0] + n3[0] + n4[0] + n5[0] ) / (double)count;
	    norm[1] = ( n1[1] + n2[1] + n3[1] + n4[1] + n5[1] ) / (double)count;
	    norm[2] = ( n1[2] + n2[2] + n3[2] + n4[2] + n5[2] ) / (double)count;
	
	    /* 	printf("  count = %d\n", count); */
	    /* 	printf("  Ave. normal = %.4f %.4f %.4f\n", 
		       norm[0], norm[1], norm[2]);*/
	    MAT3_NORMALIZE_VEC(norm, temp);
	    /* 	printf("  Normalized ave. normal = %.4f %.4f %.4f\n",  */
	    /* 	       norm[0], norm[1], norm[2]); */
	}
	/* printf("%d vn %.4f %.4f %.4f\n", i, norm[0], norm[1], norm[2]); */
	fprintf(obj, "vn %.4f %.4f %.4f\n", norm[0], norm[1], norm[2]);
    }

    /* dump faces */
    printf("  writing faces\n");
    for ( i = 1; i <= tricount; i++ ) {
	fprintf(obj, "f %d %d %d\n", tris[i][0], tris[i][1], tris[i][2]);
    }

    fclose(obj);
}

int main(int argc, char **argv) {
    char basename[256], basepath[256], temp[256];
    long int tmp_index;
    int len;

    strcpy(basename, argv[1]);

    /* find the base path of the file */
    extract_path(basename, basepath);
    extract_path(basepath, basepath);
    extract_path(basepath, basepath);
    printf("%s\n", basepath);

    /* find the index of the current file */
    extract_file(basename, temp);
    len = strlen(temp);
    if ( len >= 2 ) {
	temp[len-2] = '\0';
    }
    tmp_index = atoi(temp);
    printf("%ld\n", tmp_index);
    fgBucketParseIndex(tmp_index, &my_index);

    printf("bucket = %d %d %d %d\n", 
	   my_index.lon, my_index.lat, my_index.x, my_index.y);
    /* generate the indexes of the neighbors */
    fgBucketOffset(&my_index, &ne_index,  1,  1);
    fgBucketOffset(&my_index, &nw_index, -1,  1);
    fgBucketOffset(&my_index, &se_index,  1, -1);
    fgBucketOffset(&my_index, &sw_index, -1, -1);

    fgBucketOffset(&my_index, &north_index,  0,  1);
    fgBucketOffset(&my_index, &south_index,  0, -1);
    fgBucketOffset(&my_index, &east_index,  1,  0);
    fgBucketOffset(&my_index, &west_index, -1,  0);

    /* load the input data files */
    triload(basename, basepath);

    /* dump in WaveFront .obj format */
    dump_obj(basename);

    return(0);
}


/* $Log: tri2obj.c,v $
 * Revision 1.16  1998/05/23 15:20:41  curt
 * Output more digits after the decimal place.
 *
 * Revision 1.15  1998/05/02 01:54:39  curt
 * Converting to polar3d.h routines.
 *
 * Revision 1.14  1998/04/18 04:01:32  curt
 * Now use libMath rather than having local copies of math routines.
 *
 * Revision 1.13  1998/04/14 02:26:11  curt
 * Code reorganizations.  Added a Lib/ directory for more general libraries.
 *
 * Revision 1.12  1998/04/08 23:22:18  curt
 * Adopted Gnu automake/autoconf system.
 *
 * Revision 1.11  1998/03/03 16:01:00  curt
 * More c++ compile tweaks.
 *
 * Revision 1.10  1998/01/31 00:41:27  curt
 * Made a few changes converting floats to doubles.
 *
 * Revision 1.9  1998/01/27 18:37:04  curt
 * Lots of updates to get back in sync with changes made over in .../Src/
 *
 * Revision 1.8  1998/01/17 01:25:39  curt
 * Added support for shared normals.
 *
 * Revision 1.7  1998/01/12 02:42:00  curt
 * Average up to five triangles per vertex instead of three.
 *
 * Revision 1.6  1998/01/09 23:03:15  curt
 * Restructured to split 1deg x 1deg dem's into 64 subsections.
 *
 * Revision 1.5  1997/12/08 19:17:50  curt
 * Fixed a type in the normal generation code.
 *
 * Revision 1.4  1997/12/02 13:13:32  curt
 * Fixed problem with averaged vertex normals.
 *
 * Revision 1.3  1997/11/15 18:05:05  curt
 * minor tweaks ...
 *
 * Revision 1.2  1997/11/14 00:29:13  curt
 * Transform scenery coordinates at this point in pipeline when scenery is
 * being translated to .obj format, not when it is being loaded into the end
 * renderer.  Precalculate normals for each node as average of the normals
 * of each containing polygon so Garoude shading is now supportable.
 *
 * Revision 1.1  1997/10/29 23:05:15  curt
 * Initial revision.
 *
 */
