/* splittris.c -- read in a .ele/.node file pair generated by the
 *                triangle program and output a simple Wavefront .obj
 *                file for the north, south, east, and west edge
 *                verticies ... including the normals.
 *
 * Written by Curtis Olson, started January 1998.
 *
 * Copyright (C) 1997  Curtis L. Olson  - curt@infoplane.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: splittris.c,v 1.10 1998/05/02 01:54:37 curt Exp $
 * (Log is kept at end of this file) */


#include <math.h>
#include <stdio.h>
#include <stdlib.h>   /* for atoi() */
#include <string.h>
#include <sys/stat.h> /* for stat() */
#include <unistd.h>   /* for stat() */

#include "splittris.h"

#include <Include/fg_constants.h>
#include <Include/fg_types.h>
#include <Bucket/bucketutils.h>

#include <Math/fg_geodesy.h>
#include <Math/mat3.h>
#include <Math/polar3d.h>

int nodecount, tricount;
double xmin, xmax, ymin, ymax;

static double nodes_orig[MAX_NODES][3];
static int tris[MAX_TRIS][3];
/* static int new_tris[MAX_TRIS][3]; */

static fgCartesianPoint3d nodes_cart[MAX_NODES];

struct fgBUCKET ne_index, nw_index, sw_index, se_index;
struct fgBUCKET north_index, south_index, east_index, west_index;

/* convert a geodetic point lon(arcsec), lat(arcsec), elev(meter) to
 * a cartesian point */
fgCartesianPoint3d geod_to_cart(double geod[3]) {
    fgCartesianPoint3d cp;
    fgPolarPoint3d pp;
    double gc_lon, gc_lat, sl_radius;

    /* printf("A geodetic point is (%.2f, %.2f, %.2f)\n", 
	   geod[0], geod[1], geod[2]); */

    gc_lon = geod[0]*ARCSEC_TO_RAD;
    fgGeodToGeoc(geod[1]*ARCSEC_TO_RAD, geod[2], &sl_radius, &gc_lat);

    /* printf("A geocentric point is (%.2f, %.2f, %.2f)\n", gc_lon, 
	   gc_lat, sl_radius+geod[2]); */

    pp.lon = gc_lon;
    pp.lat = gc_lat;
    pp.radius = sl_radius+geod[2];
    cp = fgPolarToCart3d(pp);
    
    /* printf("A cart point is (%.8f, %.8f, %.8f)\n", cp.x, cp.y, cp.z); */

    return(cp);
}


/* given three points defining a triangle, calculate the normal */
void calc_normal(fgCartesianPoint3d p1, fgCartesianPoint3d p2, 
		 fgCartesianPoint3d p3, double normal[3])
{
    double v1[3], v2[3];
    double temp;

    v1[0] = p2.x - p1.x; v1[1] = p2.y - p1.y; v1[2] = p2.z - p1.z;
    v2[0] = p3.x - p1.x; v2[1] = p3.y - p1.y; v2[2] = p3.z - p1.z;

    MAT3cross_product(normal, v1, v2);
    MAT3_NORMALIZE_VEC(normal,temp);

/*  printf("  Normal = %.2f %.2f %.2f\n", normal[0], normal[1], normal[2]); */
}


/* return the file base name ( foo/bar/file.ext = file.ext ) */
void extract_file(char *in, char *base) {
    int len, i;

    len = strlen(in);

    i = len - 1;
    while ( (i >= 0) && (in[i] != '/') ) {
	i--;
    }

    in += (i + 1);
    strcpy(base, in);
}


/* return the file path name ( foo/bar/file.ext = foo/bar ) */
void extract_path(char *in, char *base) {
    int len, i;

    len = strlen(in);
    strcpy(base, in);

    i = len - 1;
    while ( (i >= 0) && (in[i] != '/') ) {
	i--;
    }

    base[i] = '\0';
}


/* return the index of all triangles containing the specified node */
void find_tris(int n, int *t1, int *t2, int *t3, int *t4, int *t5) {
    int i;

    *t1 = *t2 = *t3 = *t4 = *t5 = 0;

    i = 1;
    while ( i <= tricount ) {
        if ( (n == tris[i][0]) || (n == tris[i][1]) || (n == tris[i][2]) ) {
            if ( *t1 == 0 ) {
		*t1 = i;
            } else if ( *t2 == 0 ) {
		*t2 = i;
            } else if ( *t3 == 0 ) {
		*t3 = i;
            } else if ( *t4 == 0 ) {
		*t4 = i;
	    } else {
		*t5 = i;
	    }
        }
        i++;
    }
}


/* Initialize a new mesh structure */
void triload(char *basename) {
    char nodename[256], elename[256];
    FILE *node, *ele;
    int dim, junk1, junk2;
    int i;

    strcpy(nodename, basename);
    strcat(nodename, ".node");
    strcpy(elename, basename);
    strcat(elename, ".ele");

    printf("Loading node file:  %s ...\n", nodename);
    if ( (node = fopen(nodename, "r")) == NULL ) {
	printf("Cannot open file '%s'\n", nodename);
	exit(-1);
    }

    fscanf(node, "%d %d %d %d", &nodecount, &dim, &junk1, &junk2);

    if ( nodecount > MAX_NODES - 1 ) {
	printf("Error, too many nodes, need to increase array size\n");
	exit(-1);
    } else {
	printf("    Expecting %d nodes\n", nodecount);
    }

    for ( i = 1; i <= nodecount; i++ ) {
	fscanf(node, "%d %lf %lf %lf %d\n", &junk1, 
	       &nodes_orig[i][0], &nodes_orig[i][1], &nodes_orig[i][2], &junk2);
	/* printf("%d %.2f %.2f %.2f\n", junk1, n[0], n[1], n[2]); */
	nodes_cart[i] = geod_to_cart(nodes_orig[i]);
	/* printf("%d %.2f %.2f %.2f\n", 
	       junk1, nodes_cart[i].x, nodes_cart[i].y, nodes_cart[i].z); */

	if ( i == 1 ) {
	    xmin = xmax = nodes_orig[i][0];
	    ymin = ymax = nodes_orig[i][1];
	} else {
	    if ( nodes_orig[i][0] < xmin ) {
		xmin = nodes_orig[i][0];
	    }
	    if ( nodes_orig[i][0] > xmax ) {
		xmax = nodes_orig[i][0];
	    }
	    if ( nodes_orig[i][1] < ymin ) {
		ymin = nodes_orig[i][1];
	    }
	    if ( nodes_orig[i][1] > ymax ) {
		ymax = nodes_orig[i][1];
	    }
	}
    }

    fclose(node);

    printf("Loading element file:  %s ...\n", elename);
    if ( (ele = fopen(elename, "r")) == NULL ) {
	printf("Cannot open file '%s'\n", elename);
	exit(-1);
    }

    fscanf(ele, "%d %d %d", &tricount, &junk1, &junk2);

    if ( tricount > MAX_TRIS - 1 ) {
	printf("Error, too many elements, need to increase array size\n");
	exit(-1);
    } else {
	printf("    Expecting %d elements\n", tricount);
    }

    for ( i = 1; i <= tricount; i++ ) {
	fscanf(ele, "%d %d %d %d\n", &junk1, 
	       &tris[i][0], &tris[i][1], &tris[i][2]);
	/* printf("%d %d %d %d\n", junk1, tris[i][0], tris[i][1], tris[i][2]);*/
    }

    fclose(ele);
}


/* check if a file exists */
int file_exists(char *file) {
    struct stat stat_buf;
    int result;

    printf("checking %s ... ", file);

    result = stat(file, &stat_buf);

    if ( result != 0 ) {
	/* stat failed, no file */
	printf("not found.\n");
	return(0);
    } else {
	/* stat succeeded, file exists */
	printf("exists.\n");
	return(1);
    }
}


/* check to see if a shared object exists */
int shared_object_exists(char *basepath, char *ext) {
    char file[256], scene_path[256];
    long int index;

    if ( strcmp(ext, ".sw") == 0 ) {
	fgBucketGenBasePath(&west_index, scene_path);
	index = fgBucketGenIndex(&west_index);
	sprintf(file, "%s/%s/%ld.1.se", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
	fgBucketGenBasePath(&sw_index, scene_path);
	index = fgBucketGenIndex(&sw_index);
	sprintf(file, "%s/%s/%ld.1.ne", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
	fgBucketGenBasePath(&south_index, scene_path);
	index = fgBucketGenIndex(&south_index);
	sprintf(file, "%s/%s/%ld.1.nw", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
    }

    if ( strcmp(ext, ".se") == 0 ) {
	fgBucketGenBasePath(&east_index, scene_path);
	index = fgBucketGenIndex(&east_index);
	sprintf(file, "%s/%s/%ld.1.sw", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
	fgBucketGenBasePath(&se_index, scene_path);
	index = fgBucketGenIndex(&se_index);
	sprintf(file, "%s/%s/%ld.1.nw", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
	fgBucketGenBasePath(&south_index, scene_path);
	index = fgBucketGenIndex(&south_index);
	sprintf(file, "%s/%s/%ld.1.ne", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
    }

    if ( strcmp(ext, ".ne") == 0 ) {
	fgBucketGenBasePath(&east_index, scene_path);
	index = fgBucketGenIndex(&east_index);
	sprintf(file, "%s/%s/%ld.1.nw", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
	fgBucketGenBasePath(&ne_index, scene_path);
	index = fgBucketGenIndex(&ne_index);
	sprintf(file, "%s/%s/%ld.1.sw", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
	fgBucketGenBasePath(&north_index, scene_path);
	index = fgBucketGenIndex(&north_index);
	sprintf(file, "%s/%s/%ld.1.se", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
    }

    if ( strcmp(ext, ".nw") == 0 ) {
	fgBucketGenBasePath(&west_index, scene_path);
	index = fgBucketGenIndex(&west_index);
	sprintf(file, "%s/%s/%ld.1.ne", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
	fgBucketGenBasePath(&nw_index, scene_path);
	index = fgBucketGenIndex(&nw_index);
	sprintf(file, "%s/%s/%ld.1.se", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
	fgBucketGenBasePath(&north_index, scene_path);
	index = fgBucketGenIndex(&north_index);
	sprintf(file, "%s/%s/%ld.1.sw", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
    }

    if ( strcmp(ext, ".south") == 0 ) {
	fgBucketGenBasePath(&south_index, scene_path);
	index = fgBucketGenIndex(&south_index);
	sprintf(file, "%s/%s/%ld.1.north", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
    }

    if ( strcmp(ext, ".north") == 0 ) {
	fgBucketGenBasePath(&north_index, scene_path);
	index = fgBucketGenIndex(&north_index);
	sprintf(file, "%s/%s/%ld.1.south", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
    }

    if ( strcmp(ext, ".west") == 0 ) {
	fgBucketGenBasePath(&west_index, scene_path);
	index = fgBucketGenIndex(&west_index);
	sprintf(file, "%s/%s/%ld.1.east", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
    }

    if ( strcmp(ext, ".east") == 0 ) {
	fgBucketGenBasePath(&east_index, scene_path);
	index = fgBucketGenIndex(&east_index);
	sprintf(file, "%s/%s/%ld.1.west", basepath, scene_path, index);
	if ( file_exists(file) ) {
	    return(1);
	}
    }

    return(0);
}


/* my custom file opening routine ... don't open if a shared edge or
 * vertex alread exists */
FILE *my_open(char *basename, char *basepath, char *ext) {
    FILE *fp;
    char filename[256];

    /* create the output file name */
    strcpy(filename, basename);
    strcat(filename, ext);

    /* check if a shared object already exist from a different tile */

    if ( shared_object_exists(basepath, ext) ) {
	/* not an actual file open error, but we've already got the
         * shared edge, so we don't want to create another one */
	printf("not opening\n");
	return(NULL);
    } else {
	/* open the file */
	fp = fopen(filename, "w");
	printf("Opening %s\n", filename);
	return(fp);
    }
}


/* dump in WaveFront .obj format */
void dump_obj(char *basename, char *basepath) {
    double n1[3], n2[3], n3[3], n4[3], n5[3], norm[3], temp;
    FILE *fp, *sw, *se, *ne, *nw, *north, *south, *east, *west, *body;
    int i, t1, t2, t3, t4, t5, count;

    sw = my_open(basename, basepath, ".sw");
    se = my_open(basename, basepath, ".se");
    ne = my_open(basename, basepath, ".ne");
    nw = my_open(basename, basepath, ".nw");

    north = my_open(basename, basepath, ".north");
    south = my_open(basename, basepath, ".south");
    east = my_open(basename, basepath, ".east");
    west = my_open(basename, basepath, ".west");

    body = my_open(basename, basepath, ".body");

    printf("Dumping edges file basename:  %s ...\n", basename);

    /* dump vertices */
    printf("  writing vertices\n");
    for ( i = 1; i <= nodecount; i++ ) {

	if ( (fabs(nodes_orig[i][1] - ymin) < FG_EPSILON) && 
	     (fabs(nodes_orig[i][0] - xmin) < FG_EPSILON) ) {
	    fp = sw;
	} else if ( (fabs(nodes_orig[i][1] - ymin) < FG_EPSILON) &&
		    (fabs(nodes_orig[i][0] - xmax) < FG_EPSILON) ) {
	    fp = se;
	} else if ( (fabs(nodes_orig[i][1] - ymax) < FG_EPSILON) &&
		    (fabs(nodes_orig[i][0] - xmax) < FG_EPSILON)) {
	    fp = ne;
	} else if ( (fabs(nodes_orig[i][1] - ymax) < FG_EPSILON) &&
		    (fabs(nodes_orig[i][0] - xmin) < FG_EPSILON) ) {
	    fp = nw;
	} else if ( fabs(nodes_orig[i][0] - xmin) < FG_EPSILON ) {
	    fp = west;
	} else if ( fabs(nodes_orig[i][0] - xmax) < FG_EPSILON ) {
	    fp = east;
	} else if ( fabs(nodes_orig[i][1] - ymin) < FG_EPSILON ) {
	    fp = south;
	} else if ( fabs(nodes_orig[i][1] - ymax) < FG_EPSILON ) {
	    fp = north;
	} else {
	    fp = body;
	}

	if ( fp != NULL ) {
	    fprintf(fp, "gdn %.2f %.2f %.2f\n", 
		    nodes_orig[i][0], nodes_orig[i][1], nodes_orig[i][2]);
	}
    }

    printf("  calculating and writing normals\n");
    /* calculate and generate normals */
    for ( i = 1; i <= nodecount; i++ ) {
/* 	printf("Finding normal\n"); */

	find_tris(i, &t1, &t2, &t3, &t4, &t5);

	n1[0] = n1[1] = n1[2] = 0.0;
	n2[0] = n2[1] = n2[2] = 0.0;
	n3[0] = n3[1] = n3[2] = 0.0;
	n4[0] = n4[1] = n4[2] = 0.0;
	n5[0] = n5[1] = n5[2] = 0.0;

	count = 1;
	calc_normal(nodes_cart[tris[t1][0]], nodes_cart[tris[t1][1]], 
		    nodes_cart[tris[t1][2]], n1);

	if ( t2 > 0 ) {
	    calc_normal(nodes_cart[tris[t2][0]], nodes_cart[tris[t2][1]], 
			nodes_cart[tris[t2][2]], n2);
	    count = 2;
	}

	if ( t3 > 0 ) {
	    calc_normal(nodes_cart[tris[t3][0]], nodes_cart[tris[t3][1]],
			nodes_cart[tris[t3][2]], n3);
	    count = 3;
	}

	if ( t4 > 0 ) {
	    calc_normal(nodes_cart[tris[t4][0]], nodes_cart[tris[t4][1]],
			nodes_cart[tris[t4][2]], n4);
	    count = 4;
	}

	if ( t5 > 0 ) {
	    calc_normal(nodes_cart[tris[t5][0]], nodes_cart[tris[t5][1]],
			nodes_cart[tris[t5][2]], n5);
	    count = 5;
	}

/* 	printf("  norm[2] = %.2f %.2f %.2f\n", n1[2], n2[2], n3[2]); */

	norm[0] = ( n1[0] + n2[0] + n3[0] + n4[0] + n5[0] ) / (double)count;
	norm[1] = ( n1[1] + n2[1] + n3[1] + n4[1] + n5[1] ) / (double)count;
	norm[2] = ( n1[2] + n2[2] + n3[2] + n4[2] + n5[2] ) / (double)count;
	
/* 	printf("  count = %d\n", count); */
/* 	printf("  Ave. normal = %.4f %.4f %.4f\n", norm[0], norm[1], norm[2]);*/
	MAT3_NORMALIZE_VEC(norm, temp);
/* 	printf("  Normalized ave. normal = %.4f %.4f %.4f\n",  */
/* 	       norm[0], norm[1], norm[2]); */
	
	fp = NULL;

	if ( (fabs(nodes_orig[i][1] - ymin) < FG_EPSILON) && 
	     (fabs(nodes_orig[i][0] - xmin) < FG_EPSILON) ) {
	    fp = sw;
	} else if ( (fabs(nodes_orig[i][1] - ymin) < FG_EPSILON) &&
		    (fabs(nodes_orig[i][0] - xmax) < FG_EPSILON) ) {
	    fp = se;
	} else if ( (fabs(nodes_orig[i][1] - ymax) < FG_EPSILON) &&
		    (fabs(nodes_orig[i][0] - xmax) < FG_EPSILON)) {
	    fp = ne;
	} else if ( (fabs(nodes_orig[i][1] - ymax) < FG_EPSILON) &&
		    (fabs(nodes_orig[i][0] - xmin) < FG_EPSILON) ) {
	    fp = nw;
	} else if ( fabs(nodes_orig[i][0] - xmin) < FG_EPSILON ) {
	    fp = west;
	} else if ( fabs(nodes_orig[i][0] - xmax) < FG_EPSILON ) {
	    fp = east;
	} else if ( fabs(nodes_orig[i][1] - ymin) < FG_EPSILON ) {
	    fp = south;
	} else if ( fabs(nodes_orig[i][1] - ymax) < FG_EPSILON ) {
	    fp = north;
	}
	if ( fp != NULL ) {
	    fprintf(fp, "vn %.4f %.4f %.4f\n", norm[0], norm[1], norm[2]);
	}
    }

    if ( sw ) { fclose(sw); }
    if ( se ) { fclose(se); }
    if ( ne ) { fclose(ne); }
    if ( nw ) { fclose(nw); }

    if ( north ) { fclose(north); }
    if ( south ) { fclose(south); }
    if ( east ) { fclose(east); }
    if ( west ) { fclose(west); }

    if ( body ) { fclose(body); }
}


int main(int argc, char **argv) {
    char basename[256], basepath[256], temp[256];
    struct fgBUCKET p;
    long int index;
    int len;

    strcpy(basename, argv[1]);

    /* find the base path of the file */
    extract_path(basename, basepath);
    extract_path(basepath, basepath);
    extract_path(basepath, basepath);
    printf("%s\n", basepath);

    /* find the index of the current file */
    extract_file(basename, temp);
    len = strlen(temp);
    if ( len >= 2 ) {
	temp[len-2] = '\0';
    }
    index = atoi(temp);
    printf("%ld\n", index);
    fgBucketParseIndex(index, &p);

    printf("bucket = %d %d %d %d\n", p.lon, p.lat, p.x, p.y);
    /* generate the indexes of the neighbors */
    fgBucketOffset(&p, &ne_index,  1,  1);
    fgBucketOffset(&p, &nw_index, -1,  1);
    fgBucketOffset(&p, &se_index,  1, -1);
    fgBucketOffset(&p, &sw_index, -1, -1);

    fgBucketOffset(&p, &north_index,  0,  1);
    fgBucketOffset(&p, &south_index,  0, -1);
    fgBucketOffset(&p, &east_index,  1,  0);
    fgBucketOffset(&p, &west_index, -1,  0);

    /*
    printf("Corner indexes = %ld %ld %ld %ld\n", 
	   ne_index, nw_index, sw_index, se_index);
    printf("Edge indexes = %ld %ld %ld %ld\n",
	   north_index, south_index, east_index, west_index);
	   */

    /* load the input data files */
    triload(basename);

    /* dump in WaveFront .obj format */
    dump_obj(basename, basepath);

    return(0);
}


/* $Log: splittris.c,v $
 * Revision 1.10  1998/05/02 01:54:37  curt
 * Converting to polar3d.h routines.
 *
 * Revision 1.9  1998/04/18 04:01:20  curt
 * Now use libMath rather than having local copies of math routines.
 *
 * Revision 1.8  1998/04/14 02:26:08  curt
 * Code reorganizations.  Added a Lib/ directory for more general libraries.
 *
 * Revision 1.7  1998/04/08 23:21:13  curt
 * Adopted Gnu automake/autoconf system.
 *
 * Revision 1.6  1998/03/03 15:36:13  curt
 * Tweaks for compiling with g++
 *
 * Revision 1.5  1998/03/03 03:37:04  curt
 * Cumulative tweaks.
 *
 * Revision 1.4  1998/01/31 00:41:26  curt
 * Made a few changes converting floats to doubles.
 *
 * Revision 1.3  1998/01/27 18:37:04  curt
 * Lots of updates to get back in sync with changes made over in .../Src/
 *
 * Revision 1.2  1998/01/14 15:54:43  curt
 * Initial revision completed.
 *
 * Revision 1.1  1998/01/14 02:11:31  curt
 * Initial revision.
 *
 */
