#!/bin/sh
#
# Run a file conversion benchmark
#
# Requirements in order to run this benchmark:
#
#    GraphicsMagick
#    ImageMagick
#    Ghostscript
#    ZSH shell (to run command for consistent timing output)
#
# Execute like
#
# runbench.sh FORMAT [FORMAT] ...
#
# Commmands are executed using the zsh shell in order to obtain consistent
# timing output across systems.

# Image rendering density.  This determines the size of the test images.
# Good values are 150 300, 600, 1200, 2400 ...
DENSITY=600

# An extra operation to perform on the image
# Add -type Optimize if file output type should be optimized for size.
OPERATION='-type Optimize' # '-rotate 90'

# Configure first 'convert' program to test
CONVERT1="gm convert"


# Configure second 'convert' program to test
CONVERT2="convert"

# Name/Path to 'gm' program
GM=gm

# Path to golfer.ps (as delivered with Ghostscript)
GOLFER_PS=golfer.ps

# Path to tiger.ps (as delivered with Ghostscript)
TIGER_PS=tiger.ps

# Format to use for source images (TIFF recommended)
REF_FMT=tiff

# Time to allow system to settle before running next timing.
delay=1

MAGICK_LIMIT_PIXELS=420MP
export MAGICK_LIMIT_PIXELS

# Extend ImageMagick resource limits
MAGICK_MAP_LIMIT=3g
MAGICK_MEMORY_LIMIT=3g
MAGICK_AREA_LIMIT=20g
export MAGICK_MAP_LIMIT MAGICK_MEMORY_LIMIT MAGICK_AREA_LIMIT

DATE=`date`
CONVERT1_BASE=`echo $CONVERT1 | sed -e 's/ .*//'`
CONVERT2_BASE=`echo $CONVERT2 | sed -e 's/ .*//'`
VER1=`$CONVERT1_BASE -version | head -1`
VER2=`$CONVERT2_BASE -version | head -1`

echo "Date: $DATE"
echo "Resolution: ${DENSITY}DPI"
echo "Program 1 ($CONVERT1): $VER1"
echo "    vs"
echo "Program 2 ($CONVERT2): $VER2"
echo

#
# Create input files if they are missing.
#
if test ! -f golfer-${DENSITY}-bilevel-truecolor.${REF_FMT}
then
  echo "Creating golfer-${DENSITY}-bilevel-truecolor.${REF_FMT} ..."
  ${GM} convert -density ${DENSITY} -type bilevel ${GOLFER_PS} -type truecolor golfer-${DENSITY}-bilevel-truecolor.${REF_FMT}
fi

if test ! -f golfer-${DENSITY}-bilevel.${REF_FMT}
then
  echo "Creating golfer-${DENSITY}-bilevel.${REF_FMT} ..."
  ${GM} convert -density ${DENSITY} -type bilevel ${GOLFER_PS} golfer-${DENSITY}-bilevel.${REF_FMT}
fi

if test ! -f golfer-${DENSITY}-gray-truecolor.${REF_FMT}
then
  echo "Creating golfer-${DENSITY}-gray-truecolor.${REF_FMT} ..."
  ${GM} convert -density ${DENSITY} -type grayscale ${GOLFER_PS} -depth 8 -type truecolor golfer-${DENSITY}-gray-truecolor.${REF_FMT}
fi

if test ! -f golfer-${DENSITY}-gray.${REF_FMT}
then
  echo "Creating golfer-${DENSITY}-gray.${REF_FMT} ..."
  ${GM} convert -density ${DENSITY} -type grayscale ${GOLFER_PS} -depth 8 golfer-${DENSITY}-gray.${REF_FMT}
fi

if test ! -f golfer-${DENSITY}-gray16.${REF_FMT}
then
  echo "Creating golfer-${DENSITY}-gray16.${REF_FMT} ..."
  ${GM} convert -density ${DENSITY} -type grayscale ${GOLFER_PS} -depth 16 golfer-${DENSITY}-gray16.${REF_FMT}
fi

if test ! -f golfer-${DENSITY}-palette16.${REF_FMT}
then
  echo "Creating golfer-${DENSITY}-palette16.${REF_FMT} ..."
  ${GM} convert -density ${DENSITY} ${GOLFER_PS} -colors 16 -type palette golfer-${DENSITY}-palette16.${REF_FMT}
fi

if test ! -f golfer-${DENSITY}-pallet2.${REF_FMT}
then
  echo "Creating golfer-${DENSITY}-pallet2.${REF_FMT} ..."
  ${GM} convert -density ${DENSITY} -type bilevel ${GOLFER_PS} -colors 2 -type palette golfer-${DENSITY}-pallet2.${REF_FMT}
fi

if test ! -f tiger-${DENSITY}-palette256.${REF_FMT}
then
  echo "Creating tiger-${DENSITY}-palette256.${REF_FMT} ..."
  ${GM} convert -density ${DENSITY} ${TIGER_PS} -colors 256 -type palette tiger-${DENSITY}-palette256.${REF_FMT}
fi

if test ! -f tiger-${DENSITY}-rgb.${REF_FMT}
then
  echo "Creating tiger-${DENSITY}-rgb.${REF_FMT} ..."
  ${GM} convert -density ${DENSITY} ${TIGER_PS} -depth 8 -type truecolor tiger-${DENSITY}-rgb.${REF_FMT}
fi

if test ! -f tiger-${DENSITY}-rgb16.${REF_FMT}
then
  echo "Creating tiger-${DENSITY}-rgb16.${REF_FMT} ..."
  ${GM} convert -density ${DENSITY} ${TIGER_PS} -depth 16 -type truecolor tiger-${DENSITY}-rgb16.${REF_FMT}
fi

if test ! -f tiger-${DENSITY}-rgba.${REF_FMT}
then
  echo "Creating tiger-${DENSITY}-rgba.${REF_FMT} ..."
  ${GM} convert -density ${DENSITY} ${TIGER_PS} -depth 8 -type truecolormatte tiger-${DENSITY}-rgba.${REF_FMT}
fi

for format in "$@"
do
  for file in *${DENSITY}*.${REF_FMT}
  do
    outfile="crap.${format}"

    # Prime the cache and print input image info
    $GM identify +ping $file
    sleep $delay

    # Run first convert command
    command="${CONVERT1} ${file} ${OPERATION} ${outfile}"
    echo "$command"
    zsh -c "time $command"
    ls -lh "$outfile"
    rm -f "$outfile"
    sync
    sleep $delay
    #du -h "$outfile"

    # Run second convert command
    command="${CONVERT2} ${file} ${OPERATION} ${outfile}"
    echo "$command"
    zsh -c "time $command"
    #du -h "$outfile"
    ls -lh "$outfile"
    rm -f "$outfile"
    sync
    sleep $delay

    echo ""
  done
done 2>&1

