/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Pan - A Newsreader for X
 * Copyright (C) 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <string.h>

#include <pthread.h>

#include <libgnomeui/libgnomeui.h>

#include <pan/base/argset.h>
#include <pan/base/debug.h>
#include <pan/base/base-prefs.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/server.h>
#include <pan/base/serverlist.h>
#include <pan/base/status-item.h>

#include <pan/rules/rule.h>
#include <pan/rules/rule-ui.h>
#include <pan/rules/rule-edit-ui.h>
#include <pan/rules/rule-manager.h>

#include <pan/grouplist.h>
#include <pan/articlelist.h>
#include <pan/util.h>

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

typedef struct
{
	GtkWidget * rule_clist;
	GtkWidget * dialog;
	GPtrArray * rules;

	GtkWidget * edit_rule_button;
	GtkWidget * remove_rule_button;
	GtkWidget * apply_selected_rule_button;
	GtkWidget * apply_subscribed_rule_button;
}
RuleListDialog;

/*********************
**********************  Private Function Prototypes
*********************/

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PRIVATE
************/

static gint
get_selected_index (RuleListDialog * d)
{
	gint sel;
	GtkCList * clist;

	/* find the selected rule */
	pan_lock ();
	clist = GTK_CLIST(d->rule_clist);
	sel = -1;
	if (clist->selection != NULL)
		sel = GPOINTER_TO_INT(clist->selection->data);
	pan_unlock ();

	return sel;
}                                                                                                                               

static void
button_refresh (RuleListDialog * d)
{
	const gint sel = get_selected_index (d);
	const gboolean have_sel = sel != -1;

	pan_lock ();
	gtk_widget_set_sensitive (d->edit_rule_button, have_sel);
	gtk_widget_set_sensitive (d->remove_rule_button, have_sel);
	gtk_widget_set_sensitive (d->apply_selected_rule_button, have_sel);
	gtk_widget_set_sensitive (d->apply_subscribed_rule_button, have_sel);
	pan_unlock ();
}

static void
clist_refresh (RuleListDialog * d)
{
	gint i;
	const gint sel = get_selected_index (d);
	GtkCList * clist = GTK_CLIST(d->rule_clist);

	pan_lock ();

	gtk_clist_freeze (clist);
	gtk_clist_clear (clist);
	for (i=0; i<d->rules->len; ++i)
	{
		gint row;
		gchar * cols[2];
		Rule * r;
		double percent;
		
		r = RULE(g_ptr_array_index(d->rules,i));
		percent = r->tries==0 ? 0 : ((double)r->hits / (double)r->tries);
		cols[0] = r->name;
		cols[1] = g_strdup_printf ("%d%%", (int)(percent*100.0));

	       	row = gtk_clist_insert (clist, -1, cols);
		gtk_clist_set_row_data (clist, row, r);

		g_free (cols[1]);
	}
	gtk_clist_select_row (clist, sel, 0);
	gtk_clist_thaw (clist);

	pan_unlock ();
}

/*****
******
*****/

static gint
rule_dialog_close_cb (GnomeDialog * dialog, gpointer data)
{
	RuleListDialog * d = (RuleListDialog*) data;
	debug_enter ("rule_dialog_close_cb");

	/* save changed rules */
	rule_manager_set_rules (d->rules);

	/* cleanup rules */
	if (d->rules != NULL) {
		pan_g_ptr_array_foreach (d->rules, (GFunc)pan_object_unref, NULL);
		g_ptr_array_free (d->rules, TRUE);
		d->rules = NULL;
	}

	/* cleanup dialog */
	g_free (d);
	debug_exit ("rule_dialog_close_cb");
	return FALSE;
}

static void
rule_dialog_clicked_cb (GnomeDialog * dialog, gint index, gpointer data)
{
	gnome_dialog_close (dialog);
}


/**
***  Add
**/

static void
remove_button_clicked_cb (GtkButton * button, gpointer data)
{
	RuleListDialog * d = (RuleListDialog*) data;
	gint sel = get_selected_index (d);
	if (sel != -1)
	{
		Rule * f = RULE(g_ptr_array_index(d->rules, sel));
		g_ptr_array_remove_index (d->rules, sel);
		pan_object_unref (PAN_OBJECT(f));
		clist_refresh (d);
		button_refresh (d);
	}
}

static void
add_dialog_clicked_cb (GnomeDialog * dialog, gint index, gpointer data)
{
	if (index != RULE_EDIT_DIALOG_CLICKED_CANCEL)
	{
		guint i;
		GtkWidget * w = GTK_WIDGET(dialog);
		RuleListDialog * d = (RuleListDialog*) data;
		Rule * r_new = rule_edit_dialog_get_rule(w);
g_message ("new rule name: [%s]", r_new->name);

		/* see if we've already got this rule */
		for (i=0; i<d->rules->len; ++i) {
			Rule * r = RULE(g_ptr_array_index(d->rules,i));
			if (!pan_strcmp(r_new->name, r->name))
				break;
		}
g_message ("do we already have this? %d", i!=d->rules->len);

		/* either insert or update */
		if (i == d->rules->len) {
			g_ptr_array_add (d->rules, r_new);
		} else {
			Rule * f_old = RULE(g_ptr_array_index(d->rules,i));
			g_ptr_array_index (d->rules,i) = r_new;
			pan_object_unref (PAN_OBJECT(f_old));
		}

		/* rebuild the clist */
		clist_refresh (d);
	}

	if (index != RULE_EDIT_DIALOG_CLICKED_APPLY)
		gnome_dialog_close (dialog);
}
static void
add_button_clicked_cb (GtkButton * button, gpointer data)
{
	RuleListDialog * f = (RuleListDialog*) data;
	GtkWidget * dialog = rule_edit_dialog_create (NULL);
	gtk_signal_connect (GTK_OBJECT(dialog), "clicked", GTK_SIGNAL_FUNC(add_dialog_clicked_cb), f);
	gnome_dialog_set_parent (GNOME_DIALOG(dialog), GTK_WINDOW(f->dialog));
	gtk_widget_show_all (dialog);
}

/**
***  Edit
**/

static void
edit_dialog_clicked_cb (GnomeDialog * dialog, gint index, gpointer data)
{
	if (index != RULE_EDIT_DIALOG_CLICKED_CANCEL)
	{
		RuleListDialog * d = (RuleListDialog*) data;
		const gchar * e_name = (const gchar*) gtk_object_get_data (GTK_OBJECT(dialog), "edit_name");
		Rule * r_new = rule_edit_dialog_get_rule (GTK_WIDGET(dialog));
		gint i;

		/* find the index of the rule being edited */
		for (i=0; i<d->rules->len; ++i)
			if (!pan_strcmp(RULE(g_ptr_array_index(d->rules,i))->name, e_name))
				break;

		/* replace the edited rule with the new one */
		if (i!=d->rules->len) {
			gpointer old = g_ptr_array_index (d->rules, i);
			g_ptr_array_index (d->rules, i) = r_new;
			pan_object_unref (PAN_OBJECT(old));
			clist_refresh (d);
		}

		/* update the edit dialog */
		gtk_object_set_data_full (GTK_OBJECT(dialog), "edit_name", g_strdup(r_new->name), g_free);
	}

	if (index != RULE_EDIT_DIALOG_CLICKED_APPLY)
	{
		gnome_dialog_close (dialog);
	}
}
static void
edit_button_clicked_cb (GtkButton * button, gpointer data)
{
	RuleListDialog * d = (RuleListDialog*) data;
	gint sel = get_selected_index (d);
	if (sel != -1)
	{
		const Rule * r_old = RULE(g_ptr_array_index(d->rules,sel));
		GtkWidget * dialog = rule_edit_dialog_create (r_old);
		gtk_signal_connect (GTK_OBJECT(dialog), "clicked", GTK_SIGNAL_FUNC(edit_dialog_clicked_cb), d);
		gtk_object_set_data_full (GTK_OBJECT(dialog), "edit_name", g_strdup(r_old->name), g_free);
		gnome_dialog_set_parent (GNOME_DIALOG(dialog), GTK_WINDOW(d->dialog));
		gtk_widget_show_all (dialog);
	}
}

static gboolean
rule_clist_button_press_cb (GtkWidget * w, GdkEventButton * b, gpointer data)
{
	if (b->button==1 && b->type==GDK_2BUTTON_PRESS)
		edit_button_clicked_cb (NULL, data);
	return FALSE;
}

static void
list_selection_changed_cb (GtkCList          * clist,
                           gint                row,
                           gint                column,
                           GdkEventButton    * event,
                           gpointer            user_data)
{
	button_refresh ((RuleListDialog*)user_data);
}

/**
***  Apply
**/

typedef struct
{
	GPtrArray * groups;
	Rule * rule;
}
ApplyNow;

static gchar*
apply_now_describe (const StatusItem * item)
{
	return g_strdup (_("Applying Rule to Selected Groups"));
}

static void*
apply_now_thread (void * data)
{
	StatusItem * status;
	ApplyNow * now = (ApplyNow*) data;
	Rule * rule = now->rule;
	GPtrArray * groups = (GPtrArray *) now->groups;
	guint i;

	/* add a gui feedback tool */
	status = STATUS_ITEM(status_item_new(apply_now_describe));
	status_item_set_active (status, TRUE);
	status_item_emit_init_steps (status, groups->len);

	/* process each group in turn */
	for (i=0; i!=groups->len; ++i)
	{
		Group * g = GROUP(g_ptr_array_index(groups,i));
		GPtrArray * articles;
		Group * active;
		guint   matched;

		status_item_emit_status_va (status, _("Loading group `%s'"),
			group_get_readable_name(g));
		group_ref_articles (g, NULL);

		status_item_emit_status_va (status, _("Applying rule `%s' to Group `%s'"),
			rule->name, group_get_readable_name(g));
		articles = group_get_article_array (g);
		matched = rule_apply (rule, articles);
		g_ptr_array_free (articles, TRUE);

		status_item_emit_status_va (status, _("Saving group `%s'"),
				group_get_readable_name(g));
		group_unref_articles (g, NULL);

		active = articlelist_get_group();
		if (matched>0 && (active!=NULL) && !pan_strcmp(active->name, g->name))
		{
			status_item_emit_status_va (status, _("Refreshing group `%s'"),
				group_get_readable_name(g));
			articlelist_refresh ();
		}

		status_item_emit_next_step (status);
	}

	/* cleanup ui */
	status_item_set_active (status, FALSE);
	pan_object_unref (PAN_OBJECT(status));

	/* cleanup memory */
	g_ptr_array_free (groups, TRUE);
	g_free (data);
	return NULL;
}


static void
apply_rules_impl (RuleListDialog * d, GPtrArray * groups)
{
	Rule * rule = NULL;
	gint sel = get_selected_index (d);

	if (sel != -1)
		rule = RULE(g_ptr_array_index(d->rules, sel));

	if (rule != NULL)
	{
		ApplyNow * now;

		now = g_new (ApplyNow, 1);
		now->groups = pan_g_ptr_array_dup (groups);
		now->rule = rule;

		run_in_worker_thread (apply_now_thread, now);
	}
}

static void
apply_to_selected_cb (GtkButton * button, gpointer data)
{
	GPtrArray * groups = grouplist_get_selected_groups ();
	RuleListDialog * d = (RuleListDialog*) data;

	apply_rules_impl (d, groups);

	g_ptr_array_free (groups, TRUE);
}

static void
apply_to_subscribed_cb (GtkButton * button, gpointer data)
{
	Server * s;
	GPtrArray * groups;
	RuleListDialog * d = (RuleListDialog*) data;

	s = serverlist_get_active_server ();
	groups = server_get_groups (s, SERVER_GROUPS_SUBSCRIBED);
	apply_rules_impl (d, groups);

	g_ptr_array_free (groups, TRUE);
}

/************
*************  PROTECTED
************/

/************
*************  PUBLIC
************/

GtkWidget*
rule_dialog_new (void)
{
	GtkWidget * w;
	GtkWidget * hbox;
	GtkWidget * bbox;
	GtkTooltips * tips = gtk_tooltips_new ();
	RuleListDialog * d = g_new0 (RuleListDialog, 1);
	gchar * titles [2];
	debug_enter ("rule_dialog_new");

	/* load rules */
	d->rules = g_ptr_array_new ();
	rule_manager_get_rules (d->rules);

	/* dialog */
	w = d->dialog = gnome_dialog_new (_("Pan: Rules"), GNOME_STOCK_BUTTON_OK, NULL);
	gtk_window_set_policy (GTK_WINDOW(w), TRUE, TRUE, TRUE);
	gtk_signal_connect (GTK_OBJECT(w), "close", GTK_SIGNAL_FUNC(rule_dialog_close_cb), d);
	gtk_signal_connect (GTK_OBJECT(w), "clicked", GTK_SIGNAL_FUNC(rule_dialog_clicked_cb), d);

	/* workarea */
	hbox = gtk_hbox_new (FALSE, GNOME_PAD);
	gtk_box_pack_start (GTK_BOX(GNOME_DIALOG(w)->vbox), hbox, TRUE, TRUE, 0);

	/* clist */
	titles[0] = _("Rules");
	titles[1] = _("Hit Ratio");
	w = d->rule_clist= gtk_clist_new_with_titles (2, titles);
	gtk_clist_set_column_width (GTK_CLIST(w), 0, 300);
	gtk_clist_set_column_width (GTK_CLIST(w), 1, 100);
	gtk_widget_set_usize (w, 400, -2);
	gtk_signal_connect (GTK_OBJECT(w), "button_press_event",
	                    GTK_SIGNAL_FUNC(rule_clist_button_press_cb), d);
	gtk_signal_connect (GTK_OBJECT(w), "select_row",
	                    GTK_SIGNAL_FUNC(list_selection_changed_cb), d);
	gtk_signal_connect (GTK_OBJECT(w), "unselect_row",
	                    GTK_SIGNAL_FUNC(list_selection_changed_cb), d);
        w = gtk_scrolled_window_new (NULL, NULL);
	gtk_container_add (GTK_CONTAINER(w), d->rule_clist);
	gtk_box_pack_start (GTK_BOX (hbox), w, TRUE, TRUE, 0);
	gtk_widget_set_usize (w, -2, 300);

	/* button box */
	bbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX (hbox), bbox, FALSE, FALSE, 0);

	/* add button */
	w = gtk_button_new_with_label (_("Add New"));
	gtk_box_pack_start (GTK_BOX (bbox), w, FALSE, FALSE, 0);
	gtk_tooltips_set_tip (tips, w, _("Add a new rule."), NULL);
	gtk_signal_connect (GTK_OBJECT(w), "clicked", GTK_SIGNAL_FUNC(add_button_clicked_cb), d);

	/* edit button */
	w = gtk_button_new_with_label (_("Edit"));
	gtk_box_pack_start (GTK_BOX (bbox), w, FALSE, FALSE, 0);
	gtk_tooltips_set_tip (tips, w, _("Edit the selected rule."), NULL);
	gtk_signal_connect (GTK_OBJECT(w), "clicked", GTK_SIGNAL_FUNC(edit_button_clicked_cb), d);
	d->edit_rule_button = w;

	/* remove button */
	w = gtk_button_new_with_label (_("Remove"));
	gtk_box_pack_start (GTK_BOX (bbox), w, FALSE, FALSE, 0);
	gtk_tooltips_set_tip (tips, w, _("Remove the selected rule."), NULL);
	gtk_signal_connect (GTK_OBJECT(w), "clicked", GTK_SIGNAL_FUNC(remove_button_clicked_cb), d);
	d->remove_rule_button = w;

	/* apply to selected */
	w = gtk_button_new_with_label (_("Apply to\nSelected\nGroups"));
	gtk_box_pack_end (GTK_BOX (bbox), w, FALSE, FALSE, 0);
	gtk_tooltips_set_tip (tips, w, _("Apply the Selected Rule to the Selected Groups."), NULL);
	gtk_signal_connect (GTK_OBJECT(w), "clicked", GTK_SIGNAL_FUNC(apply_to_selected_cb), d);
	d->apply_selected_rule_button = w;

	/* apply to subscribed */
	w = gtk_button_new_with_label (_("Apply to\nSubscribed\nGroups"));
	gtk_box_pack_end (GTK_BOX (bbox), w, FALSE, FALSE, 0);
	gtk_tooltips_set_tip (tips, w, _("Apply the Selected Rule to all Subscribed Groups."), NULL);
	gtk_signal_connect (GTK_OBJECT(w), "clicked", GTK_SIGNAL_FUNC(apply_to_subscribed_cb), d);
	d->apply_subscribed_rule_button = w;

	clist_refresh (d);
	button_refresh (d);
	debug_exit ("rule_dialog_new");
	return d->dialog;
}
