/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Pan - A Newsreader for X
 * Copyright (C) 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>

#include <string.h>

#include <glib.h>

#include <pan/base/debug.h>
#include <pan/base/fnmatch.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/pan-i18n.h>

#include <pan/filters/filter-phrase.h>

const gchar * FILTER_PHRASE_CLASS_ID = "PanObject::Filter::FilterPhrase";

/************
*************  PROTECTED
************/

static Filter *
filter_phrase_dup (const Filter * f_old)
{
	Filter * f_new = filter_phrase_new ();
	FilterPhrase * fp_old = FILTER_PHRASE(f_old);
	FilterPhrase * fp_new = FILTER_PHRASE(f_new);
	filter_class_dup (f_old, f_new);
	filter_phrase_set (fp_new, fp_old->state, fp_old->phrase, fp_old->is_regex);
	return f_new;
}

static gchar*
filter_phrase_to_string (const Filter * filter)
{
	gchar * pch = NULL;
	const gboolean neg = filter->negate;
	const gchar * phrase = FILTER_PHRASE(filter)->phrase;
	const gboolean is_regex = FILTER_PHRASE(filter)->is_regex;
	const FilterPhraseState state = FILTER_PHRASE(filter)->state;

	if (state==FILTER_PHRASE_SUBJECT) {
		if (is_regex) {
			if (neg)
				pch = g_strdup_printf (_("Subject doesn't match the regex `%s'"), phrase);
			else
				pch = g_strdup_printf (_("Subject matches the regex `%s'"), phrase);
		}
		else {
			if (neg)
				pch = g_strdup_printf (_("Subject doesn't match the phrase `%s'"), phrase);
			else
				pch = g_strdup_printf (_("Subject matches the phrase `%s'"), phrase);
		}
	}
	else if (state==FILTER_PHRASE_AUTHOR) {
		if (is_regex) {
			if (neg)
				pch = g_strdup_printf (_("Author doesn't match regular expression `%s'"), phrase);
			else
				pch = g_strdup_printf (_("Author matches regular expression `%s'"), phrase);
		}
		else {
			if (neg)
				pch = g_strdup_printf (_("Author doesn't match phrase `%s'"), phrase);
			else
				pch = g_strdup_printf (_("Author matches phrase `%s'"), phrase);
		}
	}
	else if (state==FILTER_PHRASE_MESSAGE_ID) {
		if (is_regex) {
			if (neg)
				pch = g_strdup_printf (_("Message-ID doesn't match regular expression `%s'"), phrase);
			else
				pch = g_strdup_printf (_("Message-ID matches regular expression `%s'"), phrase);
		}
		else {
			if (neg)
				pch = g_strdup_printf (_("Message-ID doesn't match phrase `%s'"), phrase);
			else
				pch = g_strdup_printf (_("Message-ID matches phrase `%s'"), phrase);
		}
	}
	else pch = g_strdup_printf (_("Error"));

	return pch;
}

static void
filter_phrase_test_articles (Filter          * filter,
                             const Article  ** articles,
                             gint              article_qty,
                             gboolean        * passfail)
{
	gint i;
	FilterPhrase * f = FILTER_PHRASE(filter);

	for (i=0; i<article_qty; ++i)
	{
		gchar author[512];
		const gchar * checkme = NULL;

		switch (f->state) {
			case FILTER_PHRASE_SUBJECT:
				checkme = article_get_subject (articles[i]);
				break;
			case FILTER_PHRASE_MESSAGE_ID:
				checkme = article_get_message_id (articles[i]);
				break;
			case FILTER_PHRASE_AUTHOR:
				checkme = article_get_author_str (articles[i], author, sizeof(author));
				break;
		}

		passfail[i] = f->is_regex
			? !regexec (&f->phrase_regex,  checkme, 0, NULL, 0)
			: !fnmatch (f->phrase, checkme, PAN_CASEFOLD);
	}
}

static void
filter_phrase_destructor (PanObject * o)
{
	filter_phrase_set (FILTER_PHRASE(o), FILTER_PHRASE_SUBJECT, NULL, 0);
	filter_destructor (o);
}

static void
filter_phrase_constructor (FilterPhrase * f)
{
	debug_enter ("filter_phase_constructor");

	filter_constructor ((Filter*)f,
	                    filter_phrase_destructor,
	                    filter_phrase_test_articles,
	                    filter_phrase_to_string,
	                    filter_phrase_dup,
	                    FILTER_PHRASE_CLASS_ID);

	f->state = FILTER_PHRASE_SUBJECT;
	f->is_regex = FALSE;
	f->phrase = NULL;

	debug_exit ("filter_phase_constructor");
}

/************
*************  PUBLIC
************/

Filter*
filter_phrase_new (void)
{
	FilterPhrase * f;
	debug_enter ("filter_phrase_new");

	f = g_new0 (FilterPhrase, 1);
	filter_phrase_constructor (f);

	debug_exit ("filter_phrase_new");
	return FILTER(f);
}

void
filter_phrase_set (FilterPhrase        * filter,
                   FilterPhraseState     state,
                   const gchar         * phrase,
                   gboolean              is_regex)
{
	debug_enter ("filter_phrase_set");
	g_return_if_fail (filter!=NULL);

	/* clear out old phrase */
	replace_gstr (&filter->phrase, NULL);
	if (filter->is_regex)
		regfree (&filter->phrase_regex);

	/* repopulate the filter */
	filter->state = state;
	filter->is_regex = is_regex;
	if (is_nonempty_string (phrase))
	{
		if (filter->is_regex)
		{
			filter->phrase = g_strdup (phrase);
			regcomp (&filter->phrase_regex, filter->phrase, REG_EXTENDED|REG_ICASE);
		}
		else
		{
			filter->phrase = strchr(phrase,'*')!=NULL
				? g_strdup (phrase)
				: g_strdup_printf ("*%s*", phrase);
		}
	}

	debug_exit ("filter_phrase_set");
}
