/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <ctype.h>

#include <gnome.h>

#include <pan/base/debug.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>

#include <pan/articlelist.h>
#include <pan/globals.h>
#include <pan/grouplist.h>
#include <pan/gui.h>
#include <pan/gui-paned.h>
#include <pan/prefs.h>
#include <pan/text.h>
#include <pan/util.h>

static GtkWidget*
pack_widgets (GtkWidget * w1, GtkWidget * w2, GtkOrientation orient)
{
	GtkWidget * w;
	gboolean vertical = orient == GTK_ORIENTATION_VERTICAL;

	if (w1!=NULL && w2!=NULL) {
		gint pos;
		w = vertical ? gtk_vpaned_new() : gtk_hpaned_new ();
		gtk_paned_pack1 (GTK_PANED(w), w1, FALSE, TRUE);
		gtk_paned_pack2 (GTK_PANED(w), w2, TRUE, FALSE);
		pos = vertical
			?  gnome_config_get_int ("/Pan/Geometry/vpaned=302")
			:  gnome_config_get_int ("/Pan/Geometry/hpaned=265");
		gtk_paned_set_position (GTK_PANED(w), pos);
	} else if (w1!=NULL) {
		w = w1;
	} else if (w2!=NULL) {
		w = w2;
	} else {
		w = NULL;
	}

	return w;
}


/*---[ gui_paned_construct ]------------------------------------------
 * build the paned GUI layout
 *--------------------------------------------------------------------*/
void
gui_paned_construct (void)
{
	GtkWidget * w [3];
	GtkWidget * work;
	const gchar * pch;
	int i;
	int vpos;
	int hpos;
	debug_enter ("gui_paned_construct");

	pan_lock ();

	/* clear out the workarea */
	gtk_widget_unrealize (GTK_WIDGET(contents_vbox));
	gtk_widget_hide_all (GTK_WIDGET(contents_vbox));
	if (groups_vbox->parent != NULL)
		gtk_container_remove (GTK_CONTAINER(groups_vbox->parent), groups_vbox);
	if (articlelist_ctree->parent != NULL)
		gtk_container_remove (GTK_CONTAINER(articlelist_ctree->parent), articlelist_ctree);
	if (text_box->parent != NULL)
		gtk_container_remove (GTK_CONTAINER(text_box->parent), text_box);
	gtk_widget_unrealize (groups_vbox);
	gtk_widget_unrealize (articlelist_ctree);
	gtk_widget_unrealize (text_box);

	/* make a new UI */
	if (1) {
		GList * children = gtk_container_children (GTK_CONTAINER(contents_vbox));
		GList * l;
		for (l=children; l!=NULL; l=l->next)
			gtk_container_remove (GTK_CONTAINER(contents_vbox), GTK_WIDGET(l->data));
		g_list_free (l);
	}

	gtk_widget_show (groups_vbox);
	gtk_widget_show (articlelist_ctree);
	gtk_widget_show (text_box);

	/**
	***  The layout_str is a four-character string where the first
	***  character is in range [1..5] representing the five layouts
	***  Pan supports.  The remaining three characters show which
	***  widgets are placed first, second, and third, respectively.
	***  The grouplist is represented by 'g', the thread view is
	***  represented by 't', and the article is represented by 'a'.
	**/

	pch = layout_str + 1;
	for (i=0; i<=2; ++i, ++pch) {
		if (tolower(*pch)=='g')
			w[i] = show_group_pane ? groups_vbox : NULL;
		else if (tolower(*pch)=='t')
			w[i] = show_header_pane ? articlelist_ctree : NULL;
		else
			w[i] = show_article_pane ? text_box : NULL;
	}

	vpos = gnome_config_get_int_with_default ("/Pan/Geometry/vpaned=302", NULL);
	hpos = gnome_config_get_int_with_default ("/Pan/Geometry/hpaned=265", NULL);

	/* layout part 2: build the array */
	switch (*layout_str)
	{
		case '1':
		{
			GtkWidget * h = pack_widgets (w[0], w[1], GTK_ORIENTATION_HORIZONTAL);
			work = pack_widgets (h, w[2], GTK_ORIENTATION_VERTICAL);
			break;
		}
		case '2':
		{
			GtkWidget * h = pack_widgets (w[1], w[2], GTK_ORIENTATION_HORIZONTAL);
			work = pack_widgets (w[0], h, GTK_ORIENTATION_VERTICAL);
			break;
		}
		case '3':
		{
			GtkWidget * v = pack_widgets (w[0], w[1], GTK_ORIENTATION_VERTICAL);
			work = pack_widgets (v, w[2], GTK_ORIENTATION_HORIZONTAL);
			break;
		}
		case '5':
		{
			GtkWidget * v = pack_widgets (w[0], w[1], GTK_ORIENTATION_VERTICAL);
			work = pack_widgets (v, w[2], GTK_ORIENTATION_VERTICAL);
			break;
		}
		case '4':
		default:
		{
			GtkWidget * v = pack_widgets (w[1], w[2], GTK_ORIENTATION_VERTICAL);
			work = pack_widgets (w[0], v, GTK_ORIENTATION_HORIZONTAL);
			break;
		}
	}

	if (work != NULL)
		gtk_box_pack_start (GTK_BOX(contents_vbox), work, TRUE, TRUE, 0);

	Pan.viewmode = GUI_PANED;
	gui_restore_column_widths (Pan.group_clist, "group");
	gui_restore_column_widths (Pan.article_ctree, "thread_pane");
	gtk_widget_show_all (GTK_WIDGET(contents_vbox));

	pan_unlock ();
	debug_exit ("gui_paned_construct");
}


/*---[ gui_paned_get_current_pane ]-----------------------------------
 * get the current pane, returns an integer reflecting which pane
 * the user is 'in' ... almost identical to the gtk_notebook_get_page
 * functions
 *--------------------------------------------------------------------*/
int
gui_paned_get_current_pane (void)
{
	if (GTK_WIDGET_HAS_FOCUS (Pan.group_clist))
		return 0;
	if (GTK_WIDGET_HAS_FOCUS (Pan.article_ctree))
		return 1;
	if (GTK_WIDGET_HAS_FOCUS (Pan.text))
		return 2;
	
	return -1;
}


/*--------------------------------------------------------------------
 * ENTER key pressed while in the paned mode, process accordingly
 *--------------------------------------------------------------------*/
void
gui_paned_enter_clicked_nolock (void)
{
	switch (gui_paned_get_current_pane())
	{
		case 0: /* group list */
		{
			Group * g = grouplist_get_selected_group ();
			if (g != NULL)
				articlelist_set_group_nolock (g);
			break;
		}
		case 1: /* article tree */
		{
			Article* a = articlelist_get_selected_article_nolock ();
			if (a != NULL)
				text_set_from_article (a);
			break;
		}
		case 2: /* message body */
		{
			break;
		}
		default:
		{
			pan_warn_if_reached ();
			break;
		}
	}
}
