/* Mergeant
 *
 * Copyright (C) 1999 - 2006 Vivien Malerba
 * Copyright (C) 2002 - 2003 Rodrigo Moya
 *
 * Authors:
 *       Vivien Malerba <malerba@gnome-db.org>
 *       Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include <glib/gspawn.h>
#include <gtk/gtkstatusbar.h>
#include <gtk/gtktable.h>
#include <glib/gi18n-lib.h>
#include <libgnomeui/gnome-about.h>
#include <libgnomedb/libgnomedb.h>
#include <libgnomedb/gnome-db-login-dialog.h>
#include <libgnomedb/gnome-db-dbms-update-viewer.h>
#include "workspace.h"
#include "workspace-window.h"
#include <libgda/gda-threader.h>
#include "utils.h"

#define PROGRESS_BAR_TIMEOUT 20

typedef struct {
	GtkWidget      *window;
	GtkWidget      *workspace;
	GtkWidget      *status_bar;

	/* the connection to the database */
	GdaDict        *dict;
	
	GSList         *consoles; /* list of GtkDialog for Sql consoles */

	GdaTransaction *pending_transaction;
} WorkspaceWindowPrivate;

static GList         *opened_workspaces = NULL;
static GtkWidget     *dsn_dialog = NULL;

static void perform_dbms_update (WorkspaceWindowPrivate *priv);
static void stop_dbms_update_cb (GtkWidget *dialog, gint response, GdaDict *dict);

static void
destroy_private_data_cb (WorkspaceWindowPrivate *priv)
{
	if (priv) {
		opened_workspaces = g_list_remove (opened_workspaces, priv);

		while (priv->consoles) 
			gtk_widget_destroy (GTK_WIDGET (priv->consoles->data));

		if (priv->dict) {
			gda_dict_save (priv->dict, NULL); /* save before destruction */
			g_object_unref (priv->dict);
			priv->dict = NULL;
		}

		if (priv->pending_transaction) {
			g_object_unref (priv->pending_transaction);
			priv->pending_transaction = NULL;
		}

		g_free (priv);

		if (!opened_workspaces)
			gnome_db_main_quit ();
	}
}

static void
on_file_new_workspace (GtkAction *action, WorkspaceWindowPrivate *priv)
{
	create_new_workspace_window (NULL);
}

static void
on_file_copy_workspace (GtkAction *action, WorkspaceWindowPrivate *priv)
{
	create_new_workspace_window (priv->dict);
}

static void
on_file_close (GtkAction *action, WorkspaceWindowPrivate *priv)
{
	GtkWidget *window = priv->window;
	
	gtk_widget_destroy (window);
}

static void
on_file_save (GtkAction *action, WorkspaceWindowPrivate *priv)
{
	GError *error = NULL;

	/* saving of GdaDict */
	if (!gda_dict_save (priv->dict, &error)) {
		g_warning ("Could not save: %s\n", error->message);
		g_error_free (error);
	}
}

static void
on_file_exit (GtkAction *action, WorkspaceWindowPrivate *priv)
{
	while (opened_workspaces) {
		priv = opened_workspaces->data;

		on_file_close (NULL, priv);
	}
}

static void
on_database_sync (GtkAction *action, WorkspaceWindowPrivate *priv)
{
	perform_dbms_update (priv);
	gda_dict_save (priv->dict, NULL);
}

static void
on_database_begin (GtkAction *action, WorkspaceWindowPrivate *priv)
{
}

static void
on_database_commit (GtkAction *action, WorkspaceWindowPrivate *priv)
{
}

static void
on_database_rollback (GtkAction *action, WorkspaceWindowPrivate *priv)
{
}


static void
on_database_new (GtkAction *action, WorkspaceWindowPrivate *priv)
{
	GtkWidget *dlg;
	gchar *str;

	str = g_strdup_printf ("%s\n<small>%s</small>",
			       _("Database creation requires that first a database provider be selected."),
			       _("The 'SQLite' provider allows you to create a database stored in a single file\n"
				 "and does not require a database server to be set up."));
	dlg = mergeant_server_op_create_dialog (GTK_WINDOW (priv->window), GDA_SERVER_OPERATION_CREATE_DB, 
						gda_dict_get_connection (priv->dict), NULL,
						_("Create a database"), str, NULL, NULL);
	g_free (str);
	gtk_widget_show (dlg);
}

static void
on_database_del (GtkAction *action, WorkspaceWindowPrivate *priv)
{
	GtkWidget *dlg;

	dlg = mergeant_server_op_create_dialog (GTK_WINDOW (priv->window), GDA_SERVER_OPERATION_DROP_DB, 
						gda_dict_get_connection (priv->dict), NULL,
						_("Delete a database"),
						_("Database destruction requires that first a database provider be selected."),
						NULL, NULL);
	gtk_widget_show (dlg);
}

static void
on_tools_data_sources (GtkAction *action, WorkspaceWindowPrivate *priv)
{
	char *argv[2];
	GError *error = NULL;

	/* run gnome-database-properties dictig tool */
	argv[0] = (char *) "gnome-database-properties";
        argv[1] = NULL;
                                                                                  
	if (!g_spawn_async (NULL, argv, NULL, G_SPAWN_SEARCH_PATH,
			    NULL, NULL, NULL, &error)) {
		GtkWidget *dlg;
		dlg = gtk_message_dialog_new_with_markup (NULL, GTK_DIALOG_MODAL,
							  GTK_MESSAGE_ERROR,
							  GTK_BUTTONS_CLOSE, 
							  "<b>%s:</b>\n\n%s",
							  _("Could not find the 'gnome-database-properties' program"),
							  error && error->message ? error->message : _("No detail"));
		if (error)
			g_error_free (error);
		gtk_dialog_run (GTK_DIALOG (dlg));
		gtk_widget_destroy (dlg);
	}
}


static void console_destroyed_cb (GtkWidget *widget, WorkspaceWindowPrivate *priv);
static void
on_sql_console (GtkAction *action, void *data)
{
	GtkWidget *dlg, *sw, *console;
	WorkspaceWindowPrivate *priv;
	gchar *title;
	GdaConnection *cnc;

	priv = g_object_get_data (G_OBJECT (data), "Mergeant_WorkspaceWindowPrivate");

	/* dialog */
	dlg = gtk_dialog_new_with_buttons (_("SQL console"), NULL, 0,
					   GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, NULL);
	gtk_dialog_set_has_separator (GTK_DIALOG (dlg), FALSE);
	
	cnc = gda_dict_get_connection (priv->dict);
	if (gda_connection_get_username (cnc))
                title = g_strdup_printf ("%s@%s - Mergeant", gda_connection_get_username (cnc),
                                         gda_connection_get_dsn (cnc));
        else
                title = g_strdup_printf ("%s - Mergeant", gda_connection_get_dsn (cnc));
	
	gtk_window_set_title (GTK_WINDOW (dlg), title);
	g_free (title);
	
	/* console widget */
	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (sw), GTK_SHADOW_IN);	
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox), sw, TRUE, TRUE, 0);
	gtk_widget_show (sw);
	
	console = gnome_db_sql_console_new (priv->dict, _("SQL Console (type \\? + ENTER for help)\n"));
	gtk_container_add (GTK_CONTAINER (sw), console);
	gtk_widget_show (console);

	gtk_widget_set_size_request (dlg, 450, 500);
	gtk_widget_show (dlg);

	/* management */
	priv->consoles = g_slist_prepend (priv->consoles, dlg);
	g_signal_connect_swapped (dlg, "response", 
				  G_CALLBACK (gtk_widget_destroy), dlg);
	g_signal_connect (G_OBJECT (dlg), "destroy",
			  G_CALLBACK (console_destroyed_cb), priv);
}

static void
console_destroyed_cb (GtkWidget *widget, WorkspaceWindowPrivate *priv)
{
	priv->consoles = g_slist_remove (priv->consoles, widget);
}

static void
on_help_about (GtkAction *action, void *data)
{
	GtkWidget *dialog = NULL;                                                                                    
	GdkPixbuf *icon;
	const gchar *authors[] = {
		"Vivien Malerba <malerba@gnome-db.org>",
		"Fernando Martins <fmartins@hetnet.nl>",
		"Rodrigo Moya <rodrigo@gnome-db.org>",
		"Carlos Perello Marin <carlos@gnome-db.org>",
		"Gonzalo Paniagua Javier <gonzalo@gnome-db.org>",
		NULL
	};
                                                                                    
	const gchar *documenters[] = {
		"Vivien Malerba <malerba@gnome-db.org>",
		NULL
	};
	const gchar *translator_credits =
		"Ge'ez Frontier Foundation <locales@geez.org> Amharic translations\n" \
		"Mətin ?mirov <metin@karegen.com> Azerbaijani translations\n" \
		"Aleix Badia i Bosch <abadia@ica.es> Catalan translations\n" \
		"Miloslav Trmac <mitr@volny.cz> & Michal Bukovjan <bukm@centrum.cz> Czech translations\n" \
		"Ole Laursen <olau@hardworking.dk> Danish translations\n" \
		"Gerhard Dieringer <DieringG@eba-haus.de> & Christian Neumair <chris@gnome-de.org> German translations\n" \
		"Kostas Papadimas <pkst@gnome.org> Greek translations\n" \
		"Alexander Winston <alexander.winston@comcast.net> English/Canada translations\n" \
		"David Lodge <dave@cirt.net> English/British translations\n" \
		"Francisco Javier F. Serrador <serrador@cvs.gnome.org> & Pablo del Campo <pablodc@bigfoot.com> Spanish translations\n" \
		"Roozbeh Pournader <roozbeh@sharif.edu> Persian translations\n" \
		"Christophe Merlet (RedFox) <redfox@redfoxcenter.org> French translation\n" \
		"Alastair McKinstry <mckinstry@computer.org> Irish (gaeilge) translations\n" \
		"Croatian language team <lokalizacija@linux.hr> Croatiann translations\n" \
		"Mauro Colorio <linuxbox@interfree.it> & Alessio Dessì <alkex@inwind.it> Italian translations\n" \
		"FSF-India <locale@gnu.org.in> Malayalam translations\n" \
		"Norazah Abd Aziz  <azahaziz@yahoo.com> Malay translations\n" \
		"Jan-Willem Harmanny <jwharmanny@hotmail.com> Dutch translations\n" \
		"Kjartan Maraas <kmaraas@online.no> Norwegian translation\n" \
		"GNOME PL Team <translators@gnome.pl> Polish translation\n" \
		"Afonso Celso Medina <medina@maua.br> Português/Brasil translation\n" \
		"Duarte Loreto <happyguy_pt@hotmail.com> Portuguese translations\n" \
		"Valek Filippov <frob@df.ru> Russian translations\n" \
		"Martin Lacko <lacko@host.sk> Slovak translations\n" \
		"Goran Rakić <gox@devbase.net> Serbian translations\n" \
		"Горан Ракић <gox@devbase.net> Serbian translations\n" \
		"Christian Rose <menthos@menthos.com> Swedish translations\n" \
		"Ali Pakkan <apakkan@hotmail.com> Turk translations\n" \
		"Yuriy Syrota <yuri@renome.rovno.ua> Ukrainian translations\n" \
		"Trinh Minh Thanh <tmthanh@linuxmail.org> Vietnamese translation\n" \
		"Fan Zhang <a17841@bigpond.com> Simplified Chinese translations\n"
		; 
                                                                                    
	icon = gdk_pixbuf_new_from_file (MERGEANT_PIXMAPDIR "/mergeant.png", NULL);
		
	dialog = gtk_about_dialog_new ();
	gtk_about_dialog_set_name (GTK_ABOUT_DIALOG (dialog), PACKAGE);
	gtk_about_dialog_set_version (GTK_ABOUT_DIALOG (dialog), PACKAGE_VERSION);
	gtk_about_dialog_set_copyright (GTK_ABOUT_DIALOG (dialog), "(C) 1998-2005 GNOME Foundation");
	gtk_about_dialog_set_comments (GTK_ABOUT_DIALOG (dialog), _("Database services for the GNOME Desktop"));
	gtk_about_dialog_set_license (GTK_ABOUT_DIALOG (dialog), "GNU General Public License");
	gtk_about_dialog_set_website (GTK_ABOUT_DIALOG (dialog), "http://www.gnome-db.org");
	gtk_about_dialog_set_authors (GTK_ABOUT_DIALOG (dialog), authors);
	gtk_about_dialog_set_documenters (GTK_ABOUT_DIALOG (dialog), documenters);
	gtk_about_dialog_set_translator_credits (GTK_ABOUT_DIALOG (dialog), translator_credits);
	gtk_about_dialog_set_logo (GTK_ABOUT_DIALOG (dialog), icon);
	g_signal_connect (G_OBJECT (dialog), "close",
			  G_CALLBACK (gtk_widget_destroyed), &dialog);
	gtk_window_set_transient_for (GTK_WINDOW (dialog), GTK_WINDOW (data));
	gtk_widget_show (dialog);
}

static GtkActionEntry ui_actions[] = {
	{ "File", NULL, "_File", NULL, "File", NULL },
	{ "FileNewWorkspace", GTK_STOCK_NEW, "_New workspace", NULL, "Open a new workspace", G_CALLBACK (on_file_new_workspace) },
	{ "FileCopyWorkspace", GTK_STOCK_COPY, "_Copy of this Workspace", NULL, "Open a new workspace with the same connection parameters", G_CALLBACK (on_file_copy_workspace) },
	{ "FileClose", GTK_STOCK_CLOSE, "_Close", NULL, "Close this window", G_CALLBACK (on_file_close) },
	{ "FileSave", GTK_STOCK_SAVE, "_Save now", NULL, "Save metadata", G_CALLBACK (on_file_save) },
	{ "FileExit", GTK_STOCK_QUIT, "_Quit", NULL, "Exit the application", G_CALLBACK (on_file_exit) },
	{ "Database", NULL, "_Database", NULL, "Database", NULL },
	{ "DatabaseSync", GTK_STOCK_REFRESH, "_Sync", NULL, "Refresh metadata from database", G_CALLBACK (on_database_sync) },
	{ "DatabaseBeginTransaction", NULL, "_Begin", NULL, "Start a new transaction", G_CALLBACK (on_database_begin) },
	{ "DatabaseCommitTransaction", NULL, "_Commit", NULL, "Commit current transaction", G_CALLBACK (on_database_commit) },
	{ "DatabaseRollbackTransaction", NULL, "_Rollback", NULL, "Discard current transaction", G_CALLBACK (on_database_rollback) },
	{ "DatabaseCreate", GTK_STOCK_NEW, "_Create a database", NULL, "Create a new database", G_CALLBACK (on_database_new) },
	{ "DatabaseDelete", GTK_STOCK_DELETE, "_Delete a database", NULL, "Delete an existing database", G_CALLBACK (on_database_del) },
	{ "ToolsDataSources", GTK_STOCK_PREFERENCES, "_Datasources", NULL, "Manage the list of declared datasources", G_CALLBACK (on_tools_data_sources) },
	{ "SqlConsole", NULL, "_Console SQL", NULL, "Open an SQL console", G_CALLBACK (on_sql_console) },
	{ "About", NULL, "_About", NULL, "About", NULL },
	{ "HelpAbout", GTK_STOCK_ABOUT, "_About", NULL, "Show information about Mergeant", G_CALLBACK (on_help_about) }
};

static const gchar *ui_actions_info =
	"<ui>"
        "  <menubar name='MenuBar'>"
        "    <menu name='File' action='File'>"
        "      <menuitem name='FileNewWorkspace' action= 'FileNewWorkspace'/>"
        "      <menuitem name='FileCopyWorkspace' action= 'FileCopyWorkspace'/>"
        "      <separator/>"
        "      <menuitem name='FileClose' action= 'FileClose'/>"
        "      <menuitem name='FileSave' action= 'FileSave'/>"
        "      <menuitem name='FileExit' action= 'FileExit'/>"
        "    </menu>"
        "    <menu name='Database' action='Database'>"
        "      <menuitem name='ToolsDataSources' action= 'ToolsDataSources'/>"
        "      <menuitem name='DatabaseCreate' action= 'DatabaseCreate'/>"
        "      <menuitem name='DatabaseDelete' action= 'DatabaseDelete'/>"
        "      <separator/>"
        "      <menuitem name='DatabaseSync' action= 'DatabaseSync'/>"
        "      <separator/>"
        "      <menuitem name='DatabaseBeginTransaction' action= 'DatabaseBeginTransaction'/>"
        "      <menuitem name='DatabaseCommitTransaction' action= 'DatabaseCommitTransaction'/>"
        "      <menuitem name='DatabaseRollbackTransaction' action= 'DatabaseRollbackTransaction'/>"
        "      <separator/>"
        "      <menuitem name='SqlConsole' action='SqlConsole'/>"
        "    </menu>"
	"    <placeholder name='PageExtension'/>"
        "    <menu name='About' action='About'>"
        "      <menuitem name='HelpAbout' action= 'HelpAbout'/>"
        "    </menu>"
        "  </menubar>"
        "  <toolbar  name='ToolBar'>"
        "    <toolitem action='ToolsDataSources'/>"
	"    <separator/>"
        "  </toolbar>"
        "</ui>";


static void
stop_dbms_update_cb (GtkWidget *dialog, gint response, GdaDict *dict)
{
	gda_dict_stop_update_dbms_data (dict);
}

static void
perform_dbms_update (WorkspaceWindowPrivate *priv)
{
	GtkWidget *dialog, *view;
	GError *error;

	dialog = gtk_dialog_new_with_buttons (_("Metadata synchronisation"),
					      NULL,
					      GTK_DIALOG_MODAL | GTK_DIALOG_NO_SEPARATOR,
					      GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, NULL);
	g_signal_connect (G_OBJECT (dialog), "response", G_CALLBACK (stop_dbms_update_cb), priv->dict);
	
	view = gnome_db_dbms_update_viewer_new (priv->dict);
	gtk_widget_show (view);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), view, TRUE, TRUE, 6);
	gtk_widget_show (dialog);
	
	error = NULL;
	if (!gda_dict_update_dbms_data (priv->dict, 0, NULL, &error)) {
		if (error->code != GDA_DICT_META_DATA_UPDATE_USER_STOPPED) {
			GtkWidget *msg;

                        msg = gtk_message_dialog_new (GTK_WINDOW (dialog),
                                                      GTK_DIALOG_DESTROY_WITH_PARENT,
                                                      GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
                                                      _("Error updating Server metadata:\n%s\n"), error->message);
                        gtk_dialog_run (GTK_DIALOG (msg));
                        gtk_widget_destroy (msg);
		}
		g_error_free (error);
	}	
	
	gtk_widget_destroy (dialog);
}

/*
 * If no other workspace exists, then we need to quit the application
 */
static void
quit_if_no_other_workspace (void)
{
	if (!opened_workspaces)
		gtk_main_quit ();
}

static void create_new_workspace_window_no_dict (WorkspaceOptions *options);
static void create_new_workspace_window_with_dict (GdaDict *dict, gboolean force_dbms_update, WorkspaceOptions *options);

/**
 * create_new_workspace_window
 * @dict: a #GdaDict object, or %NULL
 * @ui:
 *
 * Creates a new workspace window. If @dict is not, %NULL
 * then it is reused for the new workspace.
 *
 * Returns:
 */
void
create_new_workspace_window (GdaDict *dict)
{
	if (dict) {
		if (! GDA_IS_DICT (dict)) {
			g_warning ("dict is not a GdaDict object, cancelling workspace window creation");
			quit_if_no_other_workspace ();
			return;
		}
		
		create_new_workspace_window_with_dict (dict, FALSE, NULL);
	}
	else
		create_new_workspace_window_no_dict (NULL);
}

/**
 * create_new_workspace_window_spec
 * @options: 
 * @ui:
 *
 * Same as create_new_workspace_window but with specific options
 */
void
create_new_workspace_window_spec (WorkspaceOptions *options)
{
	create_new_workspace_window_no_dict (options);
}

static void 
create_new_workspace_window_with_dict (GdaDict *dict, gboolean force_dbms_update, WorkspaceOptions *options)
{
	WorkspaceWindowPrivate *priv;
	GtkWidget *table;
	gchar *title;
	GtkActionGroup *actions;
        GtkUIManager *ui;
	GtkWidget *toolbar, *menubar;
	GdaConnection *cnc;

	/* here we MUST have a valid dictionary */
	g_assert (dict && GDA_IS_DICT (dict));
	g_object_ref (dict);

	/* create the private structure */
	priv = g_new0 (WorkspaceWindowPrivate, 1);
	priv->dict = dict;
	opened_workspaces = g_list_prepend (opened_workspaces, priv);

	gda_dict_extend_with_functions (priv->dict);

	if (force_dbms_update) {
		perform_dbms_update (priv);
		gda_dict_save (dict, NULL);
	}

	/* create the window */
	priv->window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title (GTK_WINDOW (priv->window), "Mergeant");
	g_object_set_data_full (G_OBJECT (priv->window), "Mergeant_WorkspaceWindowPrivate",
				priv, (GDestroyNotify) destroy_private_data_cb);
	gtk_window_set_icon_from_file (GTK_WINDOW (priv->window), MERGEANT_PIXMAPDIR "/mergeant.png", NULL);

	cnc = gda_dict_get_connection (priv->dict);
        if (gda_connection_get_username (cnc))
                title = g_strdup_printf ("%s@%s - Mergeant", gda_connection_get_username (cnc),
                                         gda_connection_get_dsn (cnc));
        else
                title = g_strdup_printf ("%s - Mergeant", gda_connection_get_dsn (cnc));
	
	gtk_window_set_title (GTK_WINDOW (priv->window), title);
	g_free (title);
	gtk_widget_set_size_request (priv->window, 900, 700);
	table = gtk_table_new (4, 1, FALSE);

	/* add the menu bar/toolbar */
	actions = gtk_action_group_new ("Actions");
	gtk_action_group_add_actions (actions, ui_actions, G_N_ELEMENTS (ui_actions), priv);

	ui = gtk_ui_manager_new ();
        gtk_ui_manager_insert_action_group (ui, actions, 0);
        gtk_ui_manager_add_ui_from_string (ui, ui_actions_info, -1, NULL);

	menubar = gtk_ui_manager_get_widget (ui, "/MenuBar");
	gtk_table_attach (GTK_TABLE (table), menubar, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show (menubar);

	toolbar = gtk_ui_manager_get_widget (ui, "/ToolBar");
	gtk_table_attach (GTK_TABLE (table), toolbar, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 0, 0);
	gtk_widget_show (toolbar);

	/* create the workspace */
	priv->workspace = workspace_new (priv->dict, ui);
	gtk_widget_show (priv->workspace);
	gtk_table_attach (GTK_TABLE (table), priv->workspace, 0, 1, 2, 3,
			  GTK_SHRINK | GTK_FILL | GTK_EXPAND,
			  GTK_SHRINK | GTK_FILL | GTK_EXPAND, 6, 6);

	priv->status_bar = gtk_statusbar_new ();
	gtk_widget_show (priv->status_bar);
	gtk_table_attach (GTK_TABLE (table), priv->status_bar, 0, 1, 3, 4, GTK_FILL, GTK_FILL, 0, 0);
	gtk_container_add (GTK_CONTAINER (priv->window), table);
	gtk_widget_show (table);

	/* display the window */
	gtk_widget_show (priv->window);

	/* apply options */
	if (options) {
		if (options->start_page) {
			switch (* options->start_page) {
			case 'q':
			case 'Q':
				workspace_show_page (WORKSPACE (priv->workspace), WORKSPACE_QUERIES_PAGE);
				break;
			case 'd':
			case 'D':
				workspace_show_page (WORKSPACE (priv->workspace), WORKSPACE_TYPES_PAGE);
				break;
			case 'r':
			case 'R':
				workspace_show_page (WORKSPACE (priv->workspace), WORKSPACE_RELATIONS_PAGE);
				break;
			case 't':
			case 'T':
			default:
				workspace_show_page (WORKSPACE (priv->workspace), WORKSPACE_TABLES_PAGE);
				break;
			}
		}
	}
}

/* structure holding data used during the threads usage */
typedef struct {
	GdaThreader   *threader;
	GdaDict       *dict;
	gchar         *filename;
	GError       **error;

	gchar         *dsn;
	gchar         *username;
	gchar         *password;
	GdaConnection *cnc;

	gboolean       func_retval;
	gboolean       force_dbms_update;

	GtkWidget     *label;
	gint           timeout_id;
	GtkWidget     *bar; /* pulse bar */
} ThreadData;

static void
thread_data_free (ThreadData *thdata) 
{
	if (thdata->threader)
		g_object_unref (thdata->threader);
	if (thdata->dict)
		g_object_unref (thdata->dict);
	g_free (thdata->filename);
	if (thdata->error) {
		if (*(thdata->error))
			g_error_free (*(thdata->error));
		g_free (thdata->error);
	}
	g_free (thdata->dsn);
	g_free (thdata->username);
	g_free (thdata->password);
	if (thdata->cnc)
		g_object_unref (thdata->cnc);
	g_free (thdata);
}

/*
 * Make the progress bar pulse
 */
static gboolean
update_progress_bar (GtkProgressBar *bar)
{
	gtk_progress_bar_pulse (bar);
	return TRUE;
}

/* 
 * function executed in another thread, to open the connection
 */
static void
thread_open_connect (ThreadData *thdata)
{
	GdaConnection *cnc;
	static GdaClient *gda_client = NULL;	

	if (!gda_client)
		gda_client = gda_client_new ();
	cnc = gda_dict_get_connection (thdata->dict);
	g_assert (!cnc);
	cnc = gda_client_open_connection (gda_client, thdata->dsn, thdata->username, thdata->password, 
					  0, thdata->error);
	thdata->cnc = cnc;
}

/* 
 * called in main thread by an idle loop when thread_open_connect() has finished
 */
static void
thread_open_connect_finished (GdaThreader *thread, guint job_id, ThreadData *thdata)
{
	g_source_remove (thdata->timeout_id);
	thdata->timeout_id = 0;

	if (!thdata->cnc) {
		GError *error = *(thdata->error);
		/* gdk_threads_enter (); */

		GtkWidget *msg;

		msg = gtk_message_dialog_new (NULL,
					      GTK_DIALOG_DESTROY_WITH_PARENT,
					      GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
					      _("Error Opening connection:\n%s\n"), 
					      error && error->message ? error->message : _("No detail"));
		gtk_dialog_run (GTK_DIALOG (msg));
		gtk_widget_destroy (msg);

		/* gdk_threads_leave (); */
		if (error)
			g_error_free (error);
		quit_if_no_other_workspace ();
	}
	else {
		WorkspaceOptions *options;

		gda_dict_set_connection (thdata->dict, thdata->cnc);
		g_object_unref (thdata->cnc);
		thdata->cnc = NULL;

		options = g_object_get_data (G_OBJECT (dsn_dialog), "options");
		/* gdk_threads_enter (); */
		create_new_workspace_window_with_dict (thdata->dict, thdata->force_dbms_update, options);
		/* gdk_threads_leave (); */
	}
	
	thread_data_free (thdata);
	gtk_widget_hide (dsn_dialog);
}

/*
 * Try to open the connection, run from the main thread.
 */
static void
open_connection (ThreadData *thdata)
{
	/* try to open the connection */
	gchar *str;
	
	str = g_strdup_printf ("<b>%s:</b>\n%s", _("Opening connection"),
			       _("The connection to the data source is now being opened."));
	gtk_label_set_markup (GTK_LABEL (thdata->label), str);
	g_free (str);
	
	thdata->timeout_id = g_timeout_add (PROGRESS_BAR_TIMEOUT, (GSourceFunc) update_progress_bar, thdata->bar);
	
	gda_threader_start_thread (thdata->threader, (GThreadFunc) thread_open_connect, thdata, 
				   (GdaThreaderFunc) thread_open_connect_finished, NULL, NULL);
}

/* 
 * function executed in another thread, to load a dictionary
 */
static void
thread_load_dict (ThreadData *thdata)
{
	thdata->func_retval = gda_dict_load_xml_file (thdata->dict, thdata->filename, thdata->error);
}

static void
load_dict_error_choose_cb (GtkDialog *dialog, gint response, ThreadData *thdata)
{
	g_signal_handlers_disconnect_by_func (G_OBJECT (dialog), G_CALLBACK (load_dict_error_choose_cb), thdata);
	g_signal_handlers_disconnect_by_func (G_OBJECT (dialog), G_CALLBACK (load_dict_error_choose_cb), thdata);

	gtk_widget_destroy (GTK_WIDGET (dialog));
	if (response == GTK_RESPONSE_NO) {
		thread_data_free (thdata);
		quit_if_no_other_workspace ();
		return;
	}
	else
		open_connection (thdata);	
}

/* 
 * called in main thread by an idle loop when thread_load_dict() has finished
 */
static void
thread_load_dict_finished (GdaThreader *thread, guint job_id, ThreadData *thdata)
{
	g_source_remove (thdata->timeout_id);
	thdata->timeout_id = 0;
	
	if (! thdata->func_retval) {
		GError *error = *(thdata->error);
		thdata->force_dbms_update = TRUE;

		if ((error->domain != GDA_DICT_ERROR) || 
		    (error->code != GDA_DICT_LOAD_FILE_NOT_EXIST_ERROR)) {
			GtkWidget *dlg;
			gchar *msg;
			
			gtk_widget_hide (dsn_dialog);
			msg = g_markup_printf_escaped (_("<b><big>Failed loading data dictionary</big></b>\n"
							 "<small>The previously saved data dictionary for this connection "
							 "can't be loaded because it contains errors:\n%s</small>\n\n"
							 "Do you want to force a synchronization with the database "
							 "dictionary now?\n\n"
							 "<small><u>Warning:</u> if you answer 'Yes', all the queries, etc. "
							 "defined will be lost; answering 'No' allows you try to fix the "
							 "error</small>"), error->message);
			dlg = gtk_message_dialog_new_with_markup (NULL, GTK_DIALOG_MODAL,
								  GTK_MESSAGE_QUESTION,
								  GTK_BUTTONS_YES_NO, " ");
			gtk_message_dialog_set_markup (GTK_MESSAGE_DIALOG (dlg), msg);
			g_free (msg);

			g_signal_connect (G_OBJECT (dlg), "destroy", G_CALLBACK
					  (load_dict_error_choose_cb), thdata);
			g_signal_connect (G_OBJECT (dlg), "response", G_CALLBACK
					  (load_dict_error_choose_cb), thdata);
			gtk_window_set_modal (GTK_WINDOW (dlg), TRUE);
			gtk_widget_show (dlg);
		}
		else 
			open_connection (thdata);	

		g_error_free (error);
		*(thdata->error) = NULL;
	}
	else
		open_connection (thdata);
}


static void 
create_new_workspace_window_no_dict_choose_cb (GtkDialog *dialog, gint response, gpointer data)
{
	WorkspaceWindowPrivate *priv;
	GdaDict *workdict = NULL;
	const gchar *dsn;
	GList *list = opened_workspaces;
	gchar *username, *password;
	GtkWidget *nb;

	if (response != GTK_RESPONSE_OK) {
		/* User hasn't selected a data source */
		gtk_widget_hide (GTK_WIDGET (dialog));
		quit_if_no_other_workspace ();
		return;
	}

	/* User has selected a data source */
	username = (gchar *) gnome_db_login_dialog_get_username (GNOME_DB_LOGIN_DIALOG (dialog));
	if (username)
		username = g_strdup (username);
	password = (gchar *) gnome_db_login_dialog_get_password (GNOME_DB_LOGIN_DIALOG (dialog));
	if (password)
		password = g_strdup (password);
		
	/* try to find another workspace window with the same DSN */
	dsn = gnome_db_login_dialog_get_dsn (GNOME_DB_LOGIN_DIALOG (dialog));
	while (list && !workdict) {
		gchar *dsn2;
		priv = (WorkspaceWindowPrivate *) list->data;
		dsn2 = (gchar *) gda_connection_get_dsn (gda_dict_get_connection (priv->dict));
		if (!strcmp (dsn, dsn2)) 
			workdict = priv->dict;
		g_free (dsn2);
		
		list = g_list_next (list);
	}

	/* changing the login dialog's page */
	nb = g_object_get_data (G_OBJECT (dsn_dialog), "main_part");
	gtk_notebook_set_current_page (GTK_NOTEBOOK (nb), 1);
	gtk_widget_set_sensitive (GTK_DIALOG (dsn_dialog)->action_area, FALSE);

	/* use workdict if not NULL, or create a new one */
	if (workdict) {
		create_new_workspace_window_with_dict (workdict, FALSE, NULL);
		gtk_widget_hide (dsn_dialog);
	}
	else {
		gchar *filename = NULL;
		ThreadData *thdata;
		GdaThreader *threader;
		GtkWidget *wid;		
		gchar *str;

		workdict = (GdaDict *) gda_dict_new ();
		
		filename = gda_dict_compute_xml_filename (workdict, dsn, NULL, NULL);
		gda_dict_set_xml_filename (workdict, filename);

		/* structure passed during the multi-thread */
		thdata = g_new0 (ThreadData, 1);
		thdata->dict = workdict;
		thdata->dsn = g_strdup (dsn);
		thdata->username = username;
		username = NULL;
		thdata->password = password;
		password = NULL;
		thdata->func_retval = FALSE;
		thdata->filename = filename;
		thdata->error = g_new0 (GError *, 1);

		/* set the label */
		wid = g_object_get_data (G_OBJECT (dsn_dialog), "info_label");
		thdata->label = wid;
		str = g_strdup_printf ("<b>%s:</b>\n%s", _("Loading dictionary"),
				       _("The dictionary is a file where data from a previous "
					 "session is stored. If this is the first session for the "
					 "requested data source, then there is no dictionary."));
		gtk_label_set_markup (GTK_LABEL (wid), str);
		g_free (str);
		gtk_widget_show (wid);
		
		/* show the progress pulse bar */
		wid = g_object_get_data (G_OBJECT (dsn_dialog), "info_bar");
		thdata->bar = wid;
		gtk_widget_show (wid);
		
		thdata->timeout_id = g_timeout_add (PROGRESS_BAR_TIMEOUT, (GSourceFunc) update_progress_bar, wid);

		/* try loading dictionary */
		threader = GDA_THREADER (gda_threader_new ());
		thdata->threader = threader;
		gda_threader_start_thread (threader, (GThreadFunc) thread_load_dict, thdata, 
					   (GdaThreaderFunc) thread_load_dict_finished, NULL, NULL);
	}

	g_free (username);
	g_free (password);
}

static void 
create_new_workspace_window_no_dict_destroy_cb (GtkDialog *dialog, gpointer data)
{
	gtk_widget_hide (GTK_WIDGET (dialog));
	quit_if_no_other_workspace ();
}

static void 
create_new_workspace_window_no_dict (WorkspaceOptions *options)
{
	GtkWidget *nb;

	if (!dsn_dialog) {
		GtkWidget *wid, *hbox, *vbox;

		/* dialog to choose the connection */
		dsn_dialog = gnome_db_login_dialog_new (_("Connect"));
		g_signal_connect (G_OBJECT (dsn_dialog), "destroy", G_CALLBACK
				  (create_new_workspace_window_no_dict_destroy_cb), NULL);
		g_signal_connect (G_OBJECT (dsn_dialog), "response", G_CALLBACK
				  (create_new_workspace_window_no_dict_choose_cb), NULL);
		gtk_window_set_modal (GTK_WINDOW (dsn_dialog), TRUE);

		nb = g_object_get_data (G_OBJECT (dsn_dialog), "main_part");

		vbox = gtk_vbox_new (FALSE, 0);
		gtk_notebook_append_page (GTK_NOTEBOOK (nb), vbox, NULL);
		gtk_widget_show (vbox);
			
		/* create label */
		wid = gtk_label_new ("");
		gtk_misc_set_alignment (GTK_MISC (wid), 0, 0);
		gtk_label_set_line_wrap (GTK_LABEL (wid), TRUE);
		g_object_set_data (G_OBJECT (dsn_dialog), "info_label", wid);
		gtk_box_pack_start (GTK_BOX (vbox), wid, FALSE, TRUE, 0);
		gtk_widget_show (wid);
			
		/* create progress bar */
		hbox = gtk_hbox_new (FALSE, 10);
		gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 10);
		gtk_widget_show (hbox);

		wid = gtk_progress_bar_new ();
		gtk_progress_bar_set_pulse_step (GTK_PROGRESS_BAR (wid), 0.02);
		g_object_set_data (G_OBJECT (dsn_dialog), "info_bar", wid);
		gtk_box_pack_start (GTK_BOX (hbox), wid, FALSE, TRUE, 0);
		gtk_widget_show (wid);
	}
	else
		nb = g_object_get_data (G_OBJECT (dsn_dialog), "main_part");

	gtk_widget_show (dsn_dialog);
	gtk_notebook_set_current_page (GTK_NOTEBOOK (nb), 0);
	gtk_widget_set_sensitive (GTK_DIALOG (dsn_dialog)->action_area, TRUE);

	g_object_set_data (G_OBJECT (dsn_dialog), "options", options);
	if (options) {
		GnomeDbLogin *login;
		
		login = GNOME_DB_LOGIN (gnome_db_login_dialog_get_login_widget (GNOME_DB_LOGIN_DIALOG (dsn_dialog)));
		gnome_db_login_set_dsn (login, options->dsn);
		if (options->user)
			gnome_db_login_set_username (login, options->user);
		if (options->pass)
			gnome_db_login_set_password (login, options->pass);
		gtk_dialog_response (GTK_DIALOG (dsn_dialog), GTK_RESPONSE_OK);
	}
}
