/* GDA library
 * Copyright (C) 1998-2002 The GNOME Foundation.
 *
 * AUTHORS:
 *      Michael Lausch <michael@lausch.at>
 *	Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#if !defined(__gda_client_h__)
#  define __gda_client_h__

#include <libgda/gda-connection.h>

G_BEGIN_DECLS

#define GDA_TYPE_CLIENT            (gda_client_get_type())
#define GDA_CLIENT(obj)            (G_TYPE_CHECK_INSTANCE_CAST (obj, GDA_TYPE_CLIENT, GdaClient))
#define GDA_CLIENT_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST (klass, GDA_TYPE_CLIENT, GdaClientClass))
#define GDA_IS_CLIENT(obj)         (G_TYPE_CHECK_INSTANCE_TYPE(obj, GDA_TYPE_CLIENT))
#define GDA_IS_CLIENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE((klass), GDA_TYPE_CLIENT))

typedef enum {
	GDA_CLIENT_EVENT_INVALID,

	/* events usually notified by the library itself, and which the providers
	   should notify on very special cases (like a transaction being started
	   or committed via a BEGIN/COMMIT command directly sent to the
	   execute_command method on the provider */
	GDA_CLIENT_EVENT_ERROR,                 /* params: "error" */
	GDA_CLIENT_EVENT_CONNECTION_OPENED,     /* params: */
	GDA_CLIENT_EVENT_CONNECTION_CLOSED,     /* params: */
	GDA_CLIENT_EVENT_TRANSACTION_STARTED,   /* params: "transaction" */
	GDA_CLIENT_EVENT_TRANSACTION_COMMITTED, /* params: "transaction" */
	GDA_CLIENT_EVENT_TRANSACTION_CANCELLED, /* params: "transaction" */
} GdaClientEvent;

typedef struct _GdaClientClass   GdaClientClass;
typedef struct _GdaClientPrivate GdaClientPrivate;

struct _GdaClient {
	GObject object;
	GdaClientPrivate *priv;
};

struct _GdaClientClass {
	GObjectClass parent_class;

	/* signals */
	void (* event_notification) (GdaClient *client,
				     GdaConnection *cnc,
				     GdaClientEvent event,
				     GdaParameterList *params);
};

GType          gda_client_get_type (void);
GdaClient     *gda_client_new (void);

GdaConnection *gda_client_open_connection (GdaClient *client,
					   const gchar *dsn,
					   const gchar *username,
					   const gchar *password,
					   GdaConnectionOptions options);
GdaConnection *gda_client_open_connection_from_string (GdaClient *client,
						       const gchar *provider_id,
						       const gchar *cnc_string,
						       GdaConnectionOptions options);
const GList   *gda_client_get_connection_list (GdaClient *client);
GdaConnection *gda_client_find_connection (GdaClient *client,
					   const gchar *dsn,
					   const gchar *username,
					   const gchar *password);
void           gda_client_close_all_connections (GdaClient *client);

void           gda_client_notify_event (GdaClient *client, GdaConnection *cnc,
					GdaClientEvent event, GdaParameterList *params);
void           gda_client_notify_error_event (GdaClient *client, GdaConnection *cnc, GdaError *error);
void           gda_client_notify_connection_opened_event (GdaClient *client, GdaConnection *cnc);
void           gda_client_notify_transaction_started_event (GdaClient *client,
							    GdaConnection *cnc,
							    GdaTransaction *xaction);
void           gda_client_notify_transaction_committed_event (GdaClient *client,
							      GdaConnection *cnc,
							      GdaTransaction *xaction);
void           gda_client_notify_transaction_cancelled_event (GdaClient *client,
							      GdaConnection *cnc,
							      GdaTransaction *xaction);

/*
 * Connection stack functions
 */

gboolean       gda_client_begin_transaction (GdaClient *client, GdaTransaction *xaction);
gboolean       gda_client_commit_transaction (GdaClient *client, GdaTransaction *xaction);
gboolean       gda_client_rollback_transaction (GdaClient *client, GdaTransaction *xaction);

G_END_DECLS

#endif
