// -*- C++ -*-

/* 
 * pieplotstate.cc 
 *
 * Copyright (C) 1999 Frank Koormann, Bernhard Reiter & Jan-Oliver Wagner
 *
 * Developed by Frank Koormann <fkoorman@usf.uos.de>,
 * Bernhard Reiter <breiter@usf.uos.de> and
 * Jan-Oliver Wagner <jwagner@usf.uos.de>.
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "pieplotstate.h"

#include "scalardata.h"

PiePlotState::PiePlotState(DataStore * ds)
  : data_(0), store_(ds), width_(100.0), height_(100.0),
    background_(255,255,255) 
{
  g_return_if_fail(store_ != 0);
}

PiePlotState::~PiePlotState()
{
  release_data();
}

const string& 
PiePlotState::name() const
{
  return name_;
}

void 
PiePlotState::set_name(const string & name)
{
  name_ = name;
  state_model.name_changed(this, name);
}

bool 
PiePlotState::name_from_data() const
{
  return set_name_from_data_;
}

void 
PiePlotState::set_name_from_data(bool setting)
{
  set_name_from_data_ = setting;
  if (setting)
    update_name_from_data();
}

void 
PiePlotState::update_name_from_data()
{
  if (!set_name_from_data_)
    return;

  if (data_ == 0)
    {
      name_ = _("No data to chart");
    } else {
      const string & dname = data_ ? data_->name() : _("(no data)");

      guint bufsize = dname.size() + 50;
      char* buf = new char[bufsize];

      g_snprintf(buf, bufsize, _("Pie Chart of `%s'"), dname.c_str());
 
      set_name(buf);

      delete [] buf;
    }
}

void 
PiePlotState::grab_data(Data* d)
{
  if (d != 0)
    {
      data_ = d->cast_to_scalar();
    } else {
      data_ = 0;
    }

  if (data_ != 0)
    {
      data_->ref();

      data_->data_model.add_view(*this);
    }      
}

void 
PiePlotState::release_data()
{
  if (data_ != 0)
    {
      data_->data_model.remove_view(*this);
      if (data_->unref() == 0) delete data_;
      data_ = 0;
    }
}

void 
PiePlotState::set_data(Data* d)
{
  if (data_ == (ScalarData*)d) return;

  release_data();

  grab_data(d);

  rebuild_slices();

  update_name_from_data();

  state_model.data_changed(this,data_);
}

ScalarData* 
PiePlotState::data()
{
  return data_;
}

void 
PiePlotState::rebuild_slices()
{
  if (data_ == 0)
    {
      slices_.clear();
      state_model.slices_changed(this);
      return;
    }

  gsize N = data_->size();
  if (N == 0)
    {
      slices_.clear();
      state_model.slices_changed(this);
      return;
    }

  g_assert (N > 0);

  const double * scalars = data_->scalars();
  g_return_if_fail(scalars != 0);

  // calculate sum
  double sum = 0.0;
  for (gsize i = 0; i < N; i++)
    {
      sum += scalars[i];
    }
  g_assert (sum > 0.0);

  g_assert (width_ > 0.0);
  g_assert (height_ > 0.0);

  double centerX = width_ / 2;
  double centerY = height_ / 2;
  double radius = MIN(width_/2,height_/2) / 2;	// FIXME: FIXED radius for the moment

  double angle = 0.0;

  gsize j = 0;
  while (j < N)
    {
      g_assert(j <= slices_.size());

      if (j == slices_.size())
        {
          slices_.push_back(Slice(this));
          g_assert(j < slices_.size());
        }

      slices_[j].set_index(j);
      slices_[j].set_posX(centerX);
      slices_[j].set_posY(centerY);
      slices_[j].set_radius(radius);
      slices_[j].set_start(angle);

      double distance = (scalars[j] / sum) * 360.0;
      slices_[j].set_width(distance);

      ++j;
      angle += distance;
    }

  // Clean up extra pies
  if (slices_.size() > j)
    {
      vector<Slice>::iterator erasefrom = slices_.begin() + j;
      slices_.erase(erasefrom, slices_.end());
    }

  g_assert(j == slices_.size());

  state_model.slices_changed(this);
}

void 
PiePlotState::set_width(double w)
{
  if (w <= PlotUtil::EPSILON) return; // simply refuse to do this
  width_ = w;

  recalc_layout();
}

void 
PiePlotState::set_height(double h)
{
  if (h <= PlotUtil::EPSILON) return; // simply refuse to do this
  height_ = h;

  recalc_layout();
}

const Rectangle & 
PiePlotState::plot_rect() const
{
  return plot_rect_;
}

void 
PiePlotState::recalc_layout()
{
  plot_rect_.set(0.0, 0.0, width_, height_);

  rebuild_slices();

  state_model.size_changed(this, width_, height_);
}

void 
PiePlotState::size_request(double* w, double* h)
{
  g_return_if_fail(w != 0);
  g_return_if_fail(h != 0);
  // For now, always request the current size
  *w = width_;
  *h = height_;
}
