// -*- C++ -*-

/* 
 * gfont.cc
 *
 * Copyright (C) 1998 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@emccta.com> and
 * Havoc Pennington <hp@emccta.com>.
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gfont.h"

#include "libgnomeprint/gnome-font.h"

#include <math.h>

// Eventually we might have two kinds of internal, one based on t1lib
// and one using gnome-print. Or we might pick one. Anyway, we're
// keeping this info nice and private.
class GFontInternal {
public:
  GFontInternal();
  ~GFontInternal();

  double string_width(const gchar* str);
  double font_height();
  GdkFont* gdk_font(double scale);

private:  
  GnomeFont* font_;
  GnomeDisplayFont* cached_display_font_;

};

GFont::GFont()
  : internal_(0)
{
  internal_ = new GFontInternal;
}

GFont::~GFont()
{
  delete internal_;
}

double 
GFont::string_width(const gchar* str)
{
  return internal_->string_width(str);
}

double 
GFont::font_height()
{
  return internal_->font_height();
}

GdkFont* 
GFont::gdk_font(double scale)
{
  return internal_->gdk_font(scale);
}

///////////

GFontInternal::GFontInternal()
  : font_(0), cached_display_font_(0)
{
  // FIXME, obviously this won't always be hardcoded.
  font_ = gnome_font_new_closest("Times", GNOME_FONT_BOOK, FALSE, 16);
  
  if (font_ == 0)
    {
      font_ = gnome_font_new_closest("Helvetica", GNOME_FONT_BOOK, FALSE, 16);
    }

  if (font_ == 0)
    {
      font_ = gnome_font_new_closest("fixed", GNOME_FONT_BOOK, FALSE, 16);
    }

  if (font_ == 0)
    {
      gnome_error_dialog(_("Couldn't get a default font from the GNOME print library.\n"
                           "This indicates a misconfigured GNOME print that can't find any fonts.\n"
                           "Guppi will use an X font instead, but it will be slow, and layout and printing may not work properly."));
    }
#ifdef GNOME_ENABLE_DEBUG
  else
    {
      g_debug("Default font: %s", font_->fontmap_entry->font_name);
    }
#endif
}

GFontInternal::~GFontInternal()
{
  if (font_ != 0)
    gtk_object_unref(GTK_OBJECT(font_));
}

double 
GFontInternal::string_width(const gchar* str)
{
  if (font_ != 0)
    return gnome_font_get_width_string(font_, str);
  else 
    return 0.0;
}

double
GFontInternal::font_height()
{
  if (font_ != 0)
    return font_->size; // I guess; I'm not sure this is right.
  else 
    return 0.0;
}

GdkFont* 
GFontInternal::gdk_font(double scale)
{
  if (font_ == 0)
    return 0;

  // FIXME This does the wrong thing wrt the scale; GnomeFont 
  //  is where scaling is actually kept.

  if (cached_display_font_ == 0 || 
      (fabs(cached_display_font_->scale - scale) < 1e-10))
    {
      // Note that we don't own the memory for cached_display_font_, 
      // it's in an internal gnome-print hash.
      cached_display_font_ = gnome_font_get_display_font(font_);
    }
  
  if (cached_display_font_ != 0)
    {
      return cached_display_font_->gdk_font;
    }
  else 
    {
      return 0;
    }
}
