/*
 *  Copyright (C) 1999-2001 Bruno Pires Marinho
 *  Copyright (C) 2000-2001 Matthew Tuck
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include "gtm.h"
#include "file-data.h"
#include "file-list.h"
#include "dialogs.h"
#include "download-dirs.h"
#include "dialog-common-elements.h"

/* Defines the characteristics of the adjustments in the dialog box */
#define NUMBER_LOWER_DEFAULT      1.0
#define NUMBER_UPPER_DEFAULT      1.0
#define NUMBER_MIN                1.0
#define NUMBER_MAX                9999.0
#define NUMBER_STEP_INCREMENT     1.0
#define NUMBER_PAGE_INCREMENT     10.0

#define PAD_FACTOR_DEFAULT        1.0
#define PAD_FACTOR_MIN            1.0
#define PAD_FACTOR_MAX            20.0
#define PAD_FACTOR_STEP_INCREMENT 1.0
#define PAD_FACTOR_PAGE_INCREMENT 5.0

#define PAGE_SIZE                 1.0

/* Structure that keeps the widgets with the information of the numeric
 * download */
typedef struct 
{
    GtkWidget       *dlg;        /* GNOME dialog */
    GtkWidget       *pre_url;    /* Entry */
    GtkWidget       *post_url;   /* Entry */
    GtkWidget       *min_num;    /* Spin Button */
    GtkWidget       *max_num;    /* Spin Button */
    GtkWidget       *pad_factor; /* Spin Button */
    DlPolicyWidgets dl_policy;
} NumDownData;

/* Structure that holds information about the URLs to generate */
typedef struct
{
    guint16 min_num;
    guint16 max_num;
    guint8  pad_factor;
    gchar   *pre_url;
    gchar   *post_url;
} URLData;

/* Converts information from the dialog and puts into a structure */
static void
get_url_data (URLData *url_data, NumDownData *num_down_data)
{
    g_return_if_fail (num_down_data != NULL);

    url_data->min_num = (guint16)
        gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (
            num_down_data->min_num) );
    url_data->max_num = (guint16)
        gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (
            num_down_data->max_num) );
    url_data->pad_factor = (guint8)
        gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (
            num_down_data->pad_factor) );
    url_data->pre_url =
        gtk_entry_get_text (GTK_ENTRY (num_down_data->pre_url));
    url_data->post_url =
        gtk_entry_get_text (GTK_ENTRY (num_down_data->post_url));
}

/* Create a specific URL */
static gchar *
form_url (URLData *url_data, guint16 num)
{
    gchar *url;
    gchar *format_str;

    g_return_val_if_fail (url_data != NULL, NULL);
 
    /* FIXME: Help! Does anyone know a better way to do this? */
    format_str = g_strdup_printf ("%s%d%s",
                                  "%s%.", url_data->pad_factor, "d%s");
    url = g_strdup_printf (format_str,
                           url_data->pre_url, num, url_data->post_url);
    g_free (format_str);

    return url;
}

/* Checks that URLs are not directories */
static gboolean
check_urls (URLData *url_data, GtkWidget *dlg)
{
    guint16 num;
    gchar *url;

    g_return_val_if_fail (url_data != NULL, FALSE);
    g_return_val_if_fail (dlg != NULL, FALSE);

    for (num = url_data->min_num; num <= url_data->max_num; num++) {
        url = form_url (url_data, num);

        /* We don't support downloading directories yet */
        if (!download_not_dir_check (url, dlg)) {
            g_free (url);
            return FALSE;
        }

        g_free (url);
    }

    return TRUE;
}

/* Handles adding a specific URL to the list */
static gboolean
handle_url (URLData *url_data, GtkWidget *dlg, guint16 num,
            DlPolicyData *dl_policy)
{
    gchar *url;
    FileData *file_data;

    g_return_val_if_fail (url_data != NULL, FALSE);
    g_return_val_if_fail (dlg != NULL, FALSE);
    g_return_val_if_fail (dl_policy != NULL, FALSE);
    g_return_val_if_fail (dl_policy->download_dir != NULL, FALSE);

    url = form_url (url_data, num);

    /* Create the file data with the information */
    file_data = dl_policy_file_data_create (url, dl_policy);
    
    /* Add the new download file to the file list (if not a duplicate) */
    /* FIXME: Use bulk add feature instead when completed */
    if (!file_list_add (GTK_CLIST (dl_file_list), file_data)) {
        /* URL already on the file list */
        GtkWidget *error;
        gchar *error_msg;
        
        error_msg = g_strconcat (_ ("URL:\n"), url, "\n",
                                 _ ("already in the list"), NULL);
        error = gnome_error_dialog_parented (error_msg, GTK_WINDOW (dlg));
        file_data_free (file_data);
        g_free (error_msg);
	return FALSE;
    }

    return TRUE;
}

/* Check URLs are valid */
static gboolean
valid_urls_check (URLData *url_data, GtkWidget *dlg)
{
    GtkWidget *error;

    g_return_val_if_fail (url_data != NULL, FALSE);
    g_return_val_if_fail (dlg != NULL, FALSE);

    if (!file_data_url_valid (url_data->pre_url)) {
        error = gnome_error_dialog_parented (_ ("Invalid URLs."),
                                             GTK_WINDOW (dlg));
        gnome_dialog_run_and_close (GNOME_DIALOG (error));
	return FALSE;
    } else {
        return TRUE;
    }
}

/* Call back function used when the OK button gets pressed */
static void
ok_num_down (GtkWidget *widget, NumDownData *num_down_data)
{
    guint16 num;
    URLData url_data;
    DlPolicyData dl_policy;
    GtkWidget *error;

    g_return_if_fail (widget != NULL);
    g_return_if_fail (num_down_data != NULL);

    /* Get the download policy info from the widgets */
    get_dl_policy_data (&dl_policy, &(num_down_data->dl_policy));

    /* Get the URL info from the widgets */
    get_url_data (&url_data, num_down_data);

    /* Check min num <= max num */
    if (url_data.max_num < url_data.min_num) {
        error = gnome_error_dialog_parented (_ ("Minimum number is greater "
                                             "than maximum number." ),
                                            GTK_WINDOW (num_down_data->dlg));
        gnome_dialog_run_and_close (GNOME_DIALOG (error));
	return;
    }

    /* Check URLs are valid */
    if (!valid_urls_check (&url_data, num_down_data->dlg))
        return;

    /* Check all the URLs before adding any */
    if (!check_urls (&url_data, num_down_data->dlg))
        return;

    /* Add the URLs */
    for (num = url_data.min_num; num <= url_data.max_num ; num++) {
        if (!handle_url (&url_data, num_down_data->dlg, num, &dl_policy))
	    return;
    }
	  
    /* If dialog changes default download dir update the def dl dir */
    change_default_download_dir (dl_policy.download_dir);

    /* Destroy the dialog box and free unused that */
    gtk_widget_destroy (num_down_data->dlg);
    g_free (num_down_data);
}

/* Call back function used when the cancel button gets pressed */
static void
cancel_num_down (GtkWidget *widget, NumDownData *num_down_data)
{
    g_return_if_fail (widget != NULL);
    g_return_if_fail (num_down_data != NULL);

    /* Destroy the dialog box and free unused that */
    gtk_widget_destroy (num_down_data->dlg);
    g_free (num_down_data);
}

/* Suggests a directory for a specific URL */
static gchar *
suggest_dir (URLData *url_data, gint num)
{
    gchar *url;
    gchar *dir;

    g_return_val_if_fail (url_data != NULL, NULL);

    url = form_url (url_data, num);

    dir = NULL;
    download_dirs_suggest_dir (url, &dir);

    g_free (url);

    return dir;
}

/* If possible, calculate the directory by applying pattern matching
   to the URL */
static void
match_cb (GtkWidget *widget, NumDownData *num_down_data)
{
    gint     num;
    URLData  url_data;
    gchar    *dir;
    gchar    *this_dir;
    gboolean success;
    GtkWidget *error;

    g_return_if_fail (widget != NULL);
    g_return_if_fail (num_down_data != NULL);

    get_url_data (&url_data, num_down_data);

    /* Check URLs are valid */
    if (!valid_urls_check (&url_data, num_down_data->dlg))
        return;

    dir = suggest_dir (&url_data, url_data.min_num);

    for (num = (url_data.min_num + 1); num <= url_data.max_num; num++) {
        this_dir = suggest_dir (&url_data, num);

	success = ((this_dir == NULL) == (dir == NULL));

	if (success && this_dir != NULL && dir != NULL )
	    success = (g_strcasecmp (this_dir, dir) == 0);

        if (this_dir != NULL)
            g_free (this_dir);

	if (!success) {
            if (dir != NULL)
                g_free (dir);

            error = gnome_error_dialog_parented (_ ("Can't suggest directory "
                                                 "as it would not be the same "
                                                 "for all URLs."),
                                              GTK_WINDOW (num_down_data->dlg));
            gnome_dialog_run_and_close (GNOME_DIALOG (error));
            return;
	}
    }

    dir_suggested_check (dir, &(num_down_data->dl_policy),
                         num_down_data->dlg);
}

/* Call back function used when a URL is dropped onto the dialog */
static void
dialog_drag_data_received (GtkWidget *widget, GdkDragContext *context,
                           gint x, gint y, 
                           GtkSelectionData *selection_data,
                           guint info, guint time,
                           NumDownData *num_down_data)
{
    g_return_if_fail (widget != NULL);
    g_return_if_fail (selection_data->data != NULL);
    gtk_entry_set_text (GTK_ENTRY (num_down_data->pre_url),
                        selection_data->data);
}

/* Fill the contents of the dialog */
static void
dialog_numdown_contents (NumDownData *num_down_data)
{
    GtkWidget *label, *hbox1, *hbox2;
    GtkObject *adjustment;

    g_return_if_fail (num_down_data != NULL);

    /* Pre-URL */
    label = gtk_label_new (_ ("Part of URL before number:"));
    gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (num_down_data->dlg)->vbox), 
                        label, FALSE, FALSE, 0);
    num_down_data->pre_url = gtk_entry_new ();
    gtk_widget_set_usize (num_down_data->pre_url, 300, -2);
    gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (num_down_data->dlg)->vbox), 
                        num_down_data->pre_url, FALSE, FALSE, 0);
    gtk_widget_grab_focus (num_down_data->pre_url);

    /* Numbers From/To */
    label = gtk_label_new (_ ("Numbers:"));
    gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (num_down_data->dlg)->vbox), 
                        label, FALSE, FALSE, 0);

    hbox1 = gtk_hbox_new (TRUE, 0);

    hbox2 = gtk_hbox_new (FALSE, 0);
    label = gtk_label_new (_ ("From: "));
    gtk_box_pack_start (GTK_BOX (hbox2), label, FALSE, FALSE, 0);

    adjustment = gtk_adjustment_new (NUMBER_LOWER_DEFAULT, NUMBER_MIN,
                                     NUMBER_MAX, NUMBER_STEP_INCREMENT,
                                     NUMBER_PAGE_INCREMENT, PAGE_SIZE);
    num_down_data->min_num =
        gtk_spin_button_new (GTK_ADJUSTMENT (adjustment), 0.5, 0);
    gtk_box_pack_start (GTK_BOX (hbox2), num_down_data->min_num, FALSE, FALSE,
                        0);
    gtk_box_pack_start (GTK_BOX (hbox1), hbox2, FALSE, FALSE, 0);

    hbox2 = gtk_hbox_new (FALSE, 0);
    label = gtk_label_new (_ ("To: "));
    gtk_box_pack_start (GTK_BOX (hbox2), label, FALSE, FALSE, 0);

    adjustment = gtk_adjustment_new (NUMBER_UPPER_DEFAULT, NUMBER_MIN,
                                     NUMBER_MAX, NUMBER_STEP_INCREMENT,
                                     NUMBER_PAGE_INCREMENT, PAGE_SIZE);
    num_down_data->max_num =
        gtk_spin_button_new (GTK_ADJUSTMENT (adjustment), 0.5, 0);
    gtk_box_pack_start (GTK_BOX (hbox2), num_down_data->max_num, FALSE, FALSE,
                        0);
    gtk_box_pack_start (GTK_BOX (hbox1), hbox2, FALSE, FALSE, 0);

    gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (num_down_data->dlg)->vbox),
                        hbox1, FALSE, FALSE, 0);
    
    /* Pad Factor */
    hbox1 = gtk_hbox_new (FALSE, 0);

    label = gtk_label_new (_ ("Number of digits to pad leading zeroes to:"));
    gtk_box_pack_start (GTK_BOX (hbox1), label, FALSE, FALSE, 0);

    adjustment = gtk_adjustment_new (PAD_FACTOR_DEFAULT, PAD_FACTOR_MIN,
                                     PAD_FACTOR_MAX, PAD_FACTOR_STEP_INCREMENT,
                                     PAD_FACTOR_PAGE_INCREMENT, PAGE_SIZE);
    num_down_data->pad_factor =
        gtk_spin_button_new (GTK_ADJUSTMENT (adjustment), 0.0, 0);
    gtk_box_pack_start (GTK_BOX (hbox1), num_down_data->pad_factor, FALSE,
                        FALSE, 0);

    gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (num_down_data->dlg)->vbox), 
                        hbox1, FALSE, FALSE, 0);

    /* Post-URL */
    label = gtk_label_new (_ ("Part of URL after number:"));
    gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (num_down_data->dlg)->vbox), 
                        label, FALSE, FALSE, 0);
    num_down_data->post_url = gtk_entry_new ();
    gtk_widget_set_usize (num_down_data->post_url, 300, -2);
    gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (num_down_data->dlg)->vbox), 
                        num_down_data->post_url, FALSE, FALSE, 0);

    /* Download Policy */
    add_download_policy_widget (num_down_data->dlg,
                                &(num_down_data->dl_policy),
				gtm_pref.def_download_dir,
                                GTK_SIGNAL_FUNC(match_cb),
                                num_down_data);
}

/* Dialog used to add a numeric download */
void
dialog_numdown (void)
{
    NumDownData *num_down_data = g_new (NumDownData, 1);

    /* Create dialog box */
    num_down_data->dlg = 
	gnome_dialog_new (_ ("Numeric Download"),
			  GNOME_STOCK_BUTTON_OK, GNOME_STOCK_BUTTON_CANCEL,
			  NULL);
    gnome_dialog_set_parent (GNOME_DIALOG (num_down_data->dlg),
			     GTK_WINDOW (gtm_app));

    /* Fill the contents */
    dialog_numdown_contents (num_down_data);

    /* Set up OK/Cancel callbacks, defaults, etc */
    configure_ok_cancel_dialog (GNOME_DIALOG (num_down_data->dlg),
                                GTK_SIGNAL_FUNC (ok_num_down),
                                GTK_SIGNAL_FUNC (cancel_num_down),
				(gpointer) num_down_data);

    /* Set drag 'n drop stuff */
    gtk_drag_dest_set (num_down_data->dlg,
                       GTK_DEST_DEFAULT_MOTION | GTK_DEST_DEFAULT_DROP,
                       target_table (), target_table_count (),
                       GDK_ACTION_COPY | GDK_ACTION_MOVE);

    /* Set the signal handlers for the content */
    gtk_signal_connect (GTK_OBJECT (num_down_data->dlg), "drag_data_received",
                        GTK_SIGNAL_FUNC (dialog_drag_data_received),
                        (gpointer) num_down_data);

    /* Show the dialog box and every thing it contains */
    gtk_widget_show_all (num_down_data->dlg);
}
