/* The Cairo CSS Drawing Library.
 * Copyright (C) 2008 Robert Staudinger
 *
 * Functions parse_hex() and  hex() were derived from pango-1.21.3,
 * Copyright (C) 2000  Red Hat Software, License:  LGPL 2 or later.
 *
 * This  library is free  software; you can  redistribute it and/or
 * modify it  under  the terms  of the  GNU Lesser  General  Public
 * License  as published  by the Free  Software  Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed  in the hope that it will be useful,
 * but  WITHOUT ANY WARRANTY; without even  the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License  along  with  this library;  if not,  write to  the Free
 * Software Foundation, Inc., 51  Franklin St, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 */

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include "ccd-color.h"

static const struct {
	char const		*name;
	const ccd_color_t	color;
} _color_map[] = {
  { "aliceblue",		{ 0xf0/255., 0xf8/255., 0xff/255. } },
  { "antiquewhite",		{ 0xfa/255., 0xeb/255., 0xd7/255. } },
  { "aqua",			{ 0x00/255., 0xff/255., 0xff/255. } },
  { "aquamarine",		{ 0x7f/255., 0xff/255., 0xd4/255. } },
  { "azure",			{ 0xf0/255., 0xff/255., 0xff/255. } },
  { "beige",			{ 0xf5/255., 0xf5/255., 0xdc/255. } },
  { "bisque",			{ 0xff/255., 0xe4/255., 0xc4/255. } },
  { "black",			{ 0x00/255., 0x00/255., 0x00/255. } },
  { "blanchedalmond",		{ 0xff/255., 0xeb/255., 0xcd/255. } },
  { "blue",			{ 0x00/255., 0x00/255., 0xff/255. } },
  { "blueviolet",		{ 0x8a/255., 0x2b/255., 0xe2/255. } },
  { "brown",			{ 0xa5/255., 0x2a/255., 0x2a/255. } },
  { "burlywood",		{ 0xde/255., 0xb8/255., 0x87/255. } },
  { "cadetblue",		{ 0x5f/255., 0x9e/255., 0xa0/255. } },
  { "chartreuse",		{ 0x7f/255., 0xff/255., 0x00/255. } },
  { "chocolate",		{ 0xd2/255., 0x69/255., 0x1e/255. } },
  { "coral",			{ 0xff/255., 0x7f/255., 0x50/255. } },
  { "cornflowerblue",		{ 0x64/255., 0x95/255., 0xed/255. } },
  { "cornsilk",			{ 0xff/255., 0xf8/255., 0xdc/255. } },
  { "crimson",			{ 0xdc/255., 0x14/255., 0x3c/255. } },
  { "cyan",			{ 0x00/255., 0xff/255., 0xff/255. } },
  { "darkblue",			{ 0x00/255., 0x00/255., 0x8b/255. } },
  { "darkcyan",			{ 0x00/255., 0x8b/255., 0x8b/255. } },
  { "darkgoldenrod",		{ 0xb8/255., 0x86/255., 0x0b/255. } },
  { "darkgray",			{ 0xa9/255., 0xa9/255., 0xa9/255. } },
  { "darkgrey",			{ 0xa9/255., 0xa9/255., 0xa9/255. } },
  { "darkgreen",		{ 0x00/255., 0x64/255., 0x00/255. } },
  { "darkkhaki",		{ 0xbd/255., 0xb7/255., 0x6b/255. } },
  { "darkmagenta",		{ 0x8b/255., 0x00/255., 0x8b/255. } },
  { "darkolivegreen",		{ 0x55/255., 0x6b/255., 0x2f/255. } },
  { "darkorange",		{ 0xff/255., 0x8c/255., 0x00/255. } },
  { "darkorchid",		{ 0x99/255., 0x32/255., 0xcc/255. } },
  { "darkred",			{ 0x8b/255., 0x00/255., 0x00/255. } },
  { "darksalmon",		{ 0xe9/255., 0x96/255., 0x7a/255. } },
  { "darkseagreen",		{ 0x8f/255., 0xbc/255., 0x8f/255. } },
  { "darkslateblue",		{ 0x48/255., 0x3d/255., 0x8b/255. } },
  { "darkslategray",		{ 0x2f/255., 0x4f/255., 0x4f/255. } },
  { "darkslategrey",		{ 0x2f/255., 0x4f/255., 0x4f/255. } },
  { "darkturquoise",		{ 0x00/255., 0xce/255., 0xd1/255. } },
  { "darkviolet",		{ 0x94/255., 0x00/255., 0xd3/255. } },
  { "deeppink",			{ 0xff/255., 0x14/255., 0x93/255. } },
  { "deepskyblue",		{ 0x00/255., 0xbf/255., 0xff/255. } },
  { "dimgray",			{ 0x69/255., 0x69/255., 0x69/255. } },
  { "dimgrey",			{ 0x69/255., 0x69/255., 0x69/255. } },
  { "dodgerblue",		{ 0x1e/255., 0x90/255., 0xff/255. } },
  { "firebrick",		{ 0xb2/255., 0x22/255., 0x22/255. } },
  { "floralwhite",		{ 0xff/255., 0xfa/255., 0xf0/255. } },
  { "forestgreen",		{ 0x22/255., 0x8b/255., 0x22/255. } },
  { "fuchsia",			{ 0xff/255., 0x00/255., 0xff/255. } },
  { "gainsboro",		{ 0xdc/255., 0xdc/255., 0xdc/255. } },
  { "ghostwhite",		{ 0xf8/255., 0xf8/255., 0xff/255. } },
  { "gold",			{ 0xff/255., 0xd7/255., 0x00/255. } },
  { "goldenrod",		{ 0xda/255., 0xa5/255., 0x20/255. } },
  { "gray",			{ 0x80/255., 0x80/255., 0x80/255. } },
  { "grey",			{ 0x80/255., 0x80/255., 0x80/255. } },
  { "green",			{ 0x00/255., 0x80/255., 0x00/255. } },
  { "greenyellow",		{ 0xad/255., 0xff/255., 0x2f/255. } },
  { "honeydew",			{ 0xf0/255., 0xff/255., 0xf0/255. } },
  { "hotpink",			{ 0xff/255., 0x69/255., 0xb4/255. } },
  { "indianred",		{ 0xcd/255., 0x5c/255., 0x5c/255. } },
  { "indigo",			{ 0x4b/255., 0x00/255., 0x82/255. } },
  { "ivory",			{ 0xff/255., 0xff/255., 0xf0/255. } },
  { "khaki",			{ 0xf0/255., 0xe6/255., 0x8c/255. } },
  { "lavender",			{ 0xe6/255., 0xe6/255., 0xfa/255. } },
  { "lavenderblush",		{ 0xff/255., 0xf0/255., 0xf5/255. } },
  { "lawngreen",		{ 0x7c/255., 0xfc/255., 0x00/255. } },
  { "lemonchiffon",		{ 0xff/255., 0xfa/255., 0xcd/255. } },
  { "lightblue",		{ 0xad/255., 0xd8/255., 0xe6/255. } },
  { "lightcoral",		{ 0xf0/255., 0x80/255., 0x80/255. } },
  { "lightcyan",		{ 0xe0/255., 0xff/255., 0xff/255. } },
  { "lightgoldenrodyellow",	{ 0xfa/255., 0xfa/255., 0xd2/255. } },
  { "lightgray",		{ 0xd3/255., 0xd3/255., 0xd3/255. } },
  { "lightgrey",		{ 0xd3/255., 0xd3/255., 0xd3/255. } },
  { "lightgreen",		{ 0x90/255., 0xee/255., 0x90/255. } },
  { "lightpink",		{ 0xff/255., 0xb6/255., 0xc1/255. } },
  { "lightsalmon",		{ 0xff/255., 0xa0/255., 0x7a/255. } },
  { "lightseagreen",		{ 0x20/255., 0xb2/255., 0xaa/255. } },
  { "lightskyblue",		{ 0x87/255., 0xce/255., 0xfa/255. } },
  { "lightslategray",		{ 0x77/255., 0x88/255., 0x99/255. } },
  { "lightslategrey",		{ 0x77/255., 0x88/255., 0x99/255. } },
  { "lightsteelblue",		{ 0xb0/255., 0xc4/255., 0xde/255. } },
  { "lightyellow",		{ 0xff/255., 0xff/255., 0xe0/255. } },
  { "lime",			{ 0x00/255., 0xff/255., 0x00/255. } },
  { "limegreen",		{ 0x32/255., 0xcd/255., 0x32/255. } },
  { "linen",			{ 0xfa/255., 0xf0/255., 0xe6/255. } },
  { "magenta",			{ 0xff/255., 0x00/255., 0xff/255. } },
  { "maroon",			{ 0x80/255., 0x00/255., 0x00/255. } },
  { "mediumaquamarine",		{ 0x66/255., 0xcd/255., 0xaa/255. } },
  { "mediumblue",		{ 0x00/255., 0x00/255., 0xcd/255. } },
  { "mediumorchid",		{ 0xba/255., 0x55/255., 0xd3/255. } },
  { "mediumpurple",		{ 0x93/255., 0x70/255., 0xd8/255. } },
  { "mediumseagreen",		{ 0x3c/255., 0xb3/255., 0x71/255. } },
  { "mediumslateblue",		{ 0x7b/255., 0x68/255., 0xee/255. } },
  { "mediumspringgreen",	{ 0x00/255., 0xfa/255., 0x9a/255. } },
  { "mediumturquoise",		{ 0x48/255., 0xd1/255., 0xcc/255. } },
  { "mediumvioletred",		{ 0xc7/255., 0x15/255., 0x85/255. } },
  { "midnightblue",		{ 0x19/255., 0x19/255., 0x70/255. } },
  { "mintcream",		{ 0xf5/255., 0xff/255., 0xfa/255. } },
  { "mistyrose",		{ 0xff/255., 0xe4/255., 0xe1/255. } },
  { "moccasin",			{ 0xff/255., 0xe4/255., 0xb5/255. } },
  { "navajowhite",		{ 0xff/255., 0xde/255., 0xad/255. } },
  { "navy",			{ 0x00/255., 0x00/255., 0x80/255. } },
  { "oldlace",			{ 0xfd/255., 0xf5/255., 0xe6/255. } },
  { "olive",			{ 0x80/255., 0x80/255., 0x00/255. } },
  { "olivedrab",		{ 0x6b/255., 0x8e/255., 0x23/255. } },
  { "orange",			{ 0xff/255., 0xa5/255., 0x00/255. } },
  { "orangered",		{ 0xff/255., 0x45/255., 0x00/255. } },
  { "orchid",			{ 0xda/255., 0x70/255., 0xd6/255. } },
  { "palegoldenrod",		{ 0xee/255., 0xe8/255., 0xaa/255. } },
  { "palegreen",		{ 0x98/255., 0xfb/255., 0x98/255. } },
  { "paleturquoise",		{ 0xaf/255., 0xee/255., 0xee/255. } },
  { "palevioletred",		{ 0xd8/255., 0x70/255., 0x93/255. } },
  { "papayawhip",		{ 0xff/255., 0xef/255., 0xd5/255. } },
  { "peachpuff",		{ 0xff/255., 0xda/255., 0xb9/255. } },
  { "peru",			{ 0xcd/255., 0x85/255., 0x3f/255. } },
  { "pink",			{ 0xff/255., 0xc0/255., 0xcb/255. } },
  { "plum",			{ 0xdd/255., 0xa0/255., 0xdd/255. } },
  { "powderblue",		{ 0xb0/255., 0xe0/255., 0xe6/255. } },
  { "purple",			{ 0x80/255., 0x00/255., 0x80/255. } },
  { "red",			{ 0xff/255., 0x00/255., 0x00/255. } },
  { "rosybrown",		{ 0xbc/255., 0x8f/255., 0x8f/255. } },
  { "royalblue",		{ 0x41/255., 0x69/255., 0xe1/255. } },
  { "saddlebrown",		{ 0x8b/255., 0x45/255., 0x13/255. } },
  { "salmon",			{ 0xfa/255., 0x80/255., 0x72/255. } },
  { "sandybrown",		{ 0xf4/255., 0xa4/255., 0x60/255. } },
  { "seagreen",			{ 0x2e/255., 0x8b/255., 0x57/255. } },
  { "seashell",			{ 0xff/255., 0xf5/255., 0xee/255. } },
  { "sienna",			{ 0xa0/255., 0x52/255., 0x2d/255. } },
  { "silver",			{ 0xc0/255., 0xc0/255., 0xc0/255. } },
  { "skyblue",			{ 0x87/255., 0xce/255., 0xeb/255. } },
  { "slateblue",		{ 0x6a/255., 0x5a/255., 0xcd/255. } },
  { "slategray",		{ 0x70/255., 0x80/255., 0x90/255. } },
  { "slategrey",		{ 0x70/255., 0x80/255., 0x90/255. } },
  { "snow",			{ 0xff/255., 0xfa/255., 0xfa/255. } },
  { "springgreen",		{ 0x00/255., 0xff/255., 0x7f/255. } },
  { "steelblue",		{ 0x46/255., 0x82/255., 0xb4/255. } },
  { "tan",			{ 0xd2/255., 0xb4/255., 0x8c/255. } },
  { "teal",			{ 0x00/255., 0x80/255., 0x80/255. } },
  { "thistle",			{ 0xd8/255., 0xbf/255., 0xd8/255. } },
  { "tomato",			{ 0xff/255., 0x63/255., 0x47/255. } },
  { "turquoise",		{ 0x40/255., 0xe0/255., 0xd0/255. } },
  { "violet",			{ 0xee/255., 0x82/255., 0xee/255. } },
  { "wheat",			{ 0xf5/255., 0xde/255., 0xb3/255. } },
  { "white",			{ 0xff/255., 0xff/255., 0xff/255. } },
  { "whitesmoke",		{ 0xf5/255., 0xf5/255., 0xf5/255. } },
  { "yellow",			{ 0xff/255., 0xff/255., 0x00/255. } },
  { "yellowgreen",		{ 0x9a/255., 0xcd/255., 0x32/255. } },
};

static bool
parse_name (ccd_color_t	*self,
	    char const	*css_color_name)
{
	g_return_val_if_fail (css_color_name && self, false);

	for (unsigned int i = 0; i < G_N_ELEMENTS (_color_map); i++) {
		if (0 == g_ascii_strcasecmp (_color_map[i].name, css_color_name)) {
			self->red = _color_map[i].color.red;
			self->green = _color_map[i].color.green;
			self->blue = _color_map[i].color.blue;
			return true;
		}
	}

	return false;
}

static gboolean
hex (char const		*color,
     int		 len,
     unsigned int	*c)
{
	const char *end;

	*c = 0;
	for (end = color + len; color != end; color++) {
		if (g_ascii_isxdigit (*color)) {
			*c = (*c << 4) | g_ascii_xdigit_value (*color);
		} else {
			return false;
		}
	}

	return true;
}

static bool
parse_hex (ccd_color_t	*self,
		     char const		*color)
{
	size_t		len;
	unsigned int	r, g, b;

	g_return_val_if_fail (color, false);

	len = strlen (color);
	if (len % 3 || len < 3 || len > 12)
		return false;

	len /= 3;

	if (!hex (color, len, &r) ||
	    !hex (color + len, len, &g) ||
	    !hex (color + len * 2, len, &b))
		return false;

	if (self) {
		int bits = len * 4;
		r <<= 16 - bits;
		g <<= 16 - bits;
		b <<= 16 - bits;
		while (bits < 16) {
			r |= (r >> bits);
			g |= (g >> bits);
			b |= (b >> bits);
			bits *= 2;
		}
		self->red   = r / 65535.;
		self->green = g / 65535.;
		self->blue  = b / 65535.;
	}

	return true;
}

ccd_property_spec_t
ccd_color_parse (ccd_color_t	 *self,
		 CRTerm	const	**value)
{
	ccd_property_spec_t	 type;
	char const		*str;
	bool			 ret;

	g_return_val_if_fail (self, CCD_PROPERTY_SPEC_UNSET);

	if (!*value) {
		return CCD_PROPERTY_SPEC_UNSET;
	}

	switch ((*value)->type) {
	case TERM_IDENT:
		type = ccd_property_parse_spec (value);
		if (type != CCD_PROPERTY_SPEC_SET) {
			return type;
		}
		str = cr_string_peek_raw_str ((*value)->content.str);
		ret = parse_name (self, str);
		if (ret) {
			*value = (*value)->next;
			return CCD_PROPERTY_SPEC_SET;
		}
		return CCD_PROPERTY_SPEC_UNSET;
	case TERM_HASH:
		str = cr_string_peek_raw_str ((*value)->content.str);
		ret = parse_hex (self, str);
		if (ret) {
			*value = (*value)->next;
			return CCD_PROPERTY_SPEC_SET;
		}
		return CCD_PROPERTY_SPEC_UNSET;
	case TERM_RGB:
		self->red = (*value)->content.rgb->red;
		self->green = (*value)->content.rgb->green;
		self->blue = (*value)->content.rgb->blue;
		return CCD_PROPERTY_SPEC_SET;
	/* fall thru for all other enum values to prevent compiler warnings */
	case TERM_NO_TYPE:
	case TERM_NUMBER:
	case TERM_FUNCTION:
	case TERM_STRING:
	case TERM_URI:
	case TERM_UNICODERANGE:
	default:
		return CCD_PROPERTY_SPEC_UNSET;
	}
}

void
ccd_color_dump (ccd_color_t const *self)
{
	printf ("rgb(%.3f,%.3f,%.3f)", self->red, self->green, self->blue);
}

