/* vim: set sw=8: -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * dialog-analysis-tool-wilcoxon-mann-whitney.c:
 *
 * Authors:
  *  Andreas J. Guelzow  <aguelzow@pyrshep.ca>
 *
 * (C) Copyright 2010 by Andreas J. Guelzow  <aguelzow@pyrshep.ca>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gnumeric-config.h>
#include <glib/gi18n-lib.h>
#include <gnumeric.h>
#include "dialogs.h"
#include "analysis-wilcoxon-mann-whitney.h"
#include "analysis-tools.h"

#include <workbook.h>
#include <workbook-control.h>
#include <wbc-gtk.h>
#include <workbook-view.h>
#include <gui-util.h>
#include <parse-util.h>
#include <gnm-format.h>
#include <tool-dialogs.h>
#include <dao-gui-utils.h>
#include <sheet.h>
#include <expr.h>
#include <number-match.h>
#include <ranges.h>
#include <selection.h>
#include <value.h>
#include <commands.h>
#include "help.h"

#include <widgets/gnm-dao.h>
#include <widgets/gnumeric-expr-entry.h>

#include <glade/glade.h>
#include <string.h>
#include <gtk/gtk.h>

#define WILCOXON_MANN_WHITNEY_KEY "analysistools-principal-components-dialog"

static char const * const grouped_by_group[] = {
	"grouped_by_row",
	"grouped_by_col",
	"grouped_by_area",
	NULL
};

static void
wilcoxon_mann_whitney_tool_update_sensitivity_cb (G_GNUC_UNUSED GtkWidget *dummy,
						 GenericToolState *state)
{
        GnmValue *input_range;

	/* Checking Input Range */
        input_range = gnm_expr_entry_parse_as_value (
		GNM_EXPR_ENTRY (state->input_entry), state->sheet);
	if (input_range == NULL) {
		gtk_label_set_text (GTK_LABEL (state->warning),
				    _("The input range is invalid."));
		gtk_widget_set_sensitive (state->ok_button, FALSE);
		return;
	} else {
		GnmRangeRef const *rr = value_get_rangeref (input_range);
		guint len = 0;
		GnmRange r;

		if (rr != NULL) {
			group_by_t group_by 
				= gnumeric_glade_group_value 
				(state->gui, grouped_by_group);
			range_init_rangeref (&r, rr);
			len = (group_by == GROUPED_BY_ROW)
				? range_height (&r) : range_width (&r);
		}

		value_release (input_range);
		if (len != 2) {
			gtk_label_set_text (GTK_LABEL (state->warning),
					    _("The input range should consists of 2 groups."));
			gtk_widget_set_sensitive (state->ok_button, FALSE);
			return;
		}
	}

	/* Checking Output Page */
	if (!gnm_dao_is_ready (GNM_DAO (state->gdao))) {
		gtk_label_set_text (GTK_LABEL (state->warning),
				    _("The output specification "
				      "is invalid."));
		gtk_widget_set_sensitive (state->ok_button, FALSE);
		return;
	}

	gtk_label_set_text (GTK_LABEL (state->warning), "");
	gtk_widget_set_sensitive (state->ok_button, TRUE);

	return;
}

/**
 * wilcoxon_mann_whitney_tool_ok_clicked_cb:
 * @button:
 * @state:
 *
 **/
static void
wilcoxon_mann_whitney_tool_ok_clicked_cb (G_GNUC_UNUSED GtkWidget *button,
			GenericToolState *state)
{
	data_analysis_output_t  *dao;
	analysis_tools_data_generic_t  *data;

        char   *text;
	GtkWidget *w;

	if (state->warning_dialog != NULL)
		gtk_widget_destroy (state->warning_dialog);

	data = g_new0 (analysis_tools_data_generic_t, 1);
	dao  = parse_output (state, NULL);

	data->input = gnm_expr_entry_parse_as_list (
		GNM_EXPR_ENTRY (state->input_entry), state->sheet);
	data->group_by = gnumeric_glade_group_value (state->gui, grouped_by_group);

	w = glade_xml_get_widget (state->gui, "labels_button");
        data->labels = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w));

	if (cmd_analysis_tool (WORKBOOK_CONTROL (state->wbcg), state->sheet,
			       dao, data, analysis_tool_wilcoxon_mann_whitney_engine)) {
		text = g_strdup_printf (
			_("An unexpected error has occurred: %d."), data->err);
		error_in_entry ((GenericToolState *) state,
				GTK_WIDGET (state->input_entry), text);
		g_free (text);
		range_list_destroy (data->input);
		g_free (dao);
		g_free (data);
	} else
		gtk_widget_destroy (state->dialog);
	return;
}



/**
 * dialog_wilcoxon_mann_whitney_tool:
 * @wbcg:
 * @sheet:
 *
 * Show the dialog (guru).
 *
 **/
int
dialog_wilcoxon_m_w_tool (WBCGtk *wbcg, Sheet *sheet)
{
        GenericToolState *state;
	char const * plugins[] = { "Gnumeric_fnstat", 
				   "Gnumeric_fnmath",
				   "Gnumeric_fnlookup",
				   NULL};

	if ((wbcg == NULL) ||
	    gnm_check_for_plugins_missing (plugins, wbcg_toplevel (wbcg)))
		return 1;

	/* Only pop up one copy per workbook */
	if (gnumeric_dialog_raise_if_exists (wbcg, WILCOXON_MANN_WHITNEY_KEY))
		return 0;

	state = g_new0 (GenericToolState, 1);

	if (dialog_tool_init (state, wbcg, sheet,
			      GNUMERIC_HELP_LINK_WILCOXON_MANN_WHITNEY,
			      "wilcoxon-mann-whitney.glade", "WilcoxonMannWhitney",
			      _("Could not create the Wilcoxon-Mann-Whitney Analysis Tool dialog."),
			      WILCOXON_MANN_WHITNEY_KEY,
			      G_CALLBACK (wilcoxon_mann_whitney_tool_ok_clicked_cb), NULL,
			      G_CALLBACK (wilcoxon_mann_whitney_tool_update_sensitivity_cb),
			      GNM_EE_SINGLE_RANGE))
		return 0;

	g_signal_connect_after 
		(G_OBJECT (glade_xml_get_widget
			   (state->gui,
			    "grouped_by_row")), "toggled",
		 G_CALLBACK (wilcoxon_mann_whitney_tool_update_sensitivity_cb),
		 state);

	gnm_dao_set_put (GNM_DAO (state->gdao), TRUE, TRUE);
	wilcoxon_mann_whitney_tool_update_sensitivity_cb (NULL, state);
	tool_load_selection ((GenericToolState *)state, TRUE);

        return 0;
}
