/*
 * Main header file of the LP_SOLVE toolkit.
 *
 * Original by Jeroen Dirks, 21-2-95
 * Maintained by Michel Berkelaar
 *
 * Glib & Gnumeric bindings by Jukka-Pekka Iivonen, 2002
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License, version 2, as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#ifndef __SOLVER_LPKIT_H__
#define __SOLVER_LPKIT_H__

#include <glib.h>
#include "numbers.h"
#include "solver.h"
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <glib.h>


#define DEFNUMINV        50
#define INITIAL_MAT_SIZE 10000

/* lag_solve extra status values */
#define FEAS_FOUND   	6
#define NO_FEAS_FOUND 	7
#define BREAK_BB	8

#define FIRST_NI	0
#define RAND_NI		1


/* FIXME: use Gnumeric's round instead */
#ifdef CHECK
#define my_round(val, eps) { \
	gnm_float absv; \
        absv = ((val) < 0 ? -(val) : (val)); \
        if(absv < (eps)) \
          val = 0; \
	if(Warn_count < MAX_WARN_COUNT) \
	  { \
	    if(absv > 0.5 * (eps) && absv < 2 * (eps)) \
	      { \
		Warn_count++; \
		fprintf(stderr, \
			"Warning Value close to epsilon V: %e E: %e\n", \
			(double)absv, (double)(eps)); \
		if(Warn_count == MAX_WARN_COUNT) \
		  { \
		    fprintf(stderr, \
			    "*** Surpressing further rounding warnings\n"); \
		  } \
	      } \
	  } \
}

#else
#define my_round(val,eps) if (((val) < 0 ? -(val) : (val)) < (eps)) val = 0;
#endif


#define DEF_INFINITE  1e24    /* limit for dynamic range */
#define DEF_EPSB      5.01e-7 /* for rounding RHS values to 0 determine	
				 infeasibility basis */
#define DEF_EPSEL     1e-8    /* for rounding other values (vectors) to 0 */
#define DEF_EPSD      1e-6    /* for rounding reduced costs to zero */
#define DEF_EPSILON   1e-3    /* to determine if a float value is integer */
 
#define PREJ          1e-3    /* pivot reject (try others first) */

#define ETA_START_SIZE 10000 /* start size of array Eta. Realloced if needed */
#define NAMELEN 25
#define MAXSTRL (NAMELEN-1)


typedef char nstring[NAMELEN];

typedef struct _matrec
{
        int        row_nr;
        gnm_float value;
} matrec;

typedef struct _column
{
        int            row;
        gnm_float     value;
        struct _column *next ;
} column;

typedef struct _constraint_name
{
        char  name[NAMELEN];
        int   row;
        struct _constraint_name *next;
} constraint_name;

typedef struct _bound
{
        gnm_float upbo;
        gnm_float lowbo;
} bound;

typedef struct _tmp_store_struct
{
        nstring       name;
        int           row;
        gnm_float    value;
        gnm_float    rhs_value;
        short         relat;
} tmp_store_struct;

typedef struct _rside /* contains relational operator and rhs value */
{
        gnm_float     value;
        struct _rside  *next;
        short          relat;
} rside;


/* fields indicated with ## may be modified directly */
/* pointers will have their array size in the comments */

typedef struct _lprec
{
        nstring    lp_name;	      /* the name of the lp */

        gboolean   verbose;           /* ## Verbose flag */
        gboolean   print_duals;       /* ## PrintDuals flag for PrintSolution
				       */
        gboolean   print_sol;         /* ## used in lp_solve */
        gboolean   debug;             /* ## Print B&B information */
        gboolean   print_at_invert;   /* ## Print information at every
				       * reinversion */
        gboolean   trace;             /* ## Print information on pivot
				       * selection */
        gboolean   anti_degen;	      /* ## Do perturbations */

        int	   rows;              /* Nr of constraint rows in the problem
				       */
        int        rows_alloc;        /* The allocated memory for Rows sized
				       * data */
        int        columns;           /* The number of columns (= variables) */
        int        columns_alloc;  
        int        sum;               /* The size of the variables + the 
				       * slacks */
        int        sum_alloc;

        gboolean   names_used;        /* Flag to indicate if names for rows and
				       * columns are used */
        nstring    *row_name;	      /* rows_alloc+1 */
        nstring    *col_name;	      /* columns_alloc+1 */

        /* Row[0] of the sparce matrix is the objective function */

        int        non_zeros;         /* The number of elements in the sparce
				       * matrix*/
        int        mat_alloc;         /* The allocated size for matrix sized 
				       * structures */
        matrec     *mat;              /* mat_alloc :The sparse matrix */
        int        *col_end;          /* columns_alloc+1 :Cend[i] is the index
				       * of the first element after column i.
				       * column[i] is stored in elements 
				       * col_end[i-1] to col_end[i]-1 */
        int        *col_no;           /* mat_alloc :From Row 1 on, col_no
				       * contains the column nr. of the
				       * nonzero elements, row by row */
        gboolean   row_end_valid;     /* true if row_end & col_no are valid */
        int        *row_end;          /* rows_alloc+1 :row_end[i] is the index
				       * of the first element in Colno after
				       * row i */
        gnm_float *orig_rh;          /* rows_alloc+1 :The RHS after scaling &
				       * sign changing, but before `Bound
				       * transformation' */
        gnm_float *rh;		      /* rows_alloc+1 :As orig_rh, but after
				       * Bound transformation */
        gnm_float *rhs;	      /* rows_alloc+1 :The RHS of the current
				       * simplex tableau */
        char       *must_be_int;      /* sum_alloc+1 :TRUE if variable must be 
				       * Integer */
        gnm_float *orig_upbo;        /* sum_alloc+1 :Bound before
				       * transformations */
        gnm_float *orig_lowbo;	      /*  "       "                   */
        gnm_float *upbo;             /*  " " :Upper bound after 
				       * transformation & B&B work */
        gnm_float *lowbo;            /*  "       "  :Lower bound after 
				       * transformation & B&B work */

        gboolean   basis_valid;       /* TRUE is the basis is still valid */
        int        *bas;              /* rows_alloc+1 :The basis column list */
        char       *basis;            /* sum_alloc+1 : basis[i] is TRUE if
				       * the column is in the basis */
        char       *lower;            /*  "       "  :TRUE is the variable is
				       * at its lower bound (or in the basis),
				       * it is FALSE if the variable is at
				       * its upper bound */

        gboolean   eta_valid;         /* TRUE if current Eta structures are
				       * valid */
        int        eta_alloc;         /* The allocated memory for Eta */
        int        eta_size;          /* The number of Eta columns */
        int        num_inv;           /* The number of real pivots */
        int        max_num_inv;       /* ## The number of real pivots between 
				       * reinversions */
        gnm_float *eta_value;        /* eta_alloc :The Structure containing
				       * the values of Eta */
        int        *eta_row_nr;       /*  "     "  :The Structure containing
				       * the Row indexes of Eta */
        int        *eta_col_end;      /* rows_alloc + MaxNumInv :
				       * eta_col_end[i] is the start index of
				       * the next Eta column */

        short	   bb_rule;	      /* what rule for selecting B&B variables
				       */

        gboolean   break_at_int;      /* TRUE if stop at first integer better
				       * than break_value */
        gnm_float break_value;        

        gnm_float obj_bound;         /* ## Objective function bound for
				       * speedup of B&B */
        int        iter;              /* The number of iterations in the
				       * simplex solver (LP) */
        int        total_iter;        /* The total number of iterations (B&B)
				       * (ILP) */
        int        max_total_iter;    /* The maximum total number of iterations (B&B)
				       * (ILP) */
        int        max_time;          /* The maximum time (sec.) before timeout. */
        gnm_float start_time;        /* The starting time of the solving. */
        int        max_level;         /* The Deepest B&B level of the last
				       * solution */
        int	   total_nodes;       /* total number of nodes processed in
				       * b&b */
        gnm_float *solution;         /* sum_alloc+1 :The Solution of the
				       * last LP, 
				       * 0 = The Optimal Value, 
				       * 1..rows The Slacks, 
				       * rows+1..sum The Variables */
        gnm_float *best_solution;    /*  "       "  :The Best 'Integer'
				       * Solution */
        gnm_float *duals;            /* rows_alloc+1 :The dual variables of
				       * the last LP */
  
        gboolean   maximise;          /* TRUE if the goal is to maximise the 
				       * objective function */
        gboolean   floor_first;       /* TRUE if B&B does floor bound first */
        char       *ch_sign;          /* rows_alloc+1 :TRUE if the Row in the
				       * matrix has changed sign 
				       * (a`x > b, x>=0) is translated to 
				       * s + -a`x = -b with x>=0, s>=0) */ 

        gboolean   scaling_used;      /* TRUE if scaling is used */
        gboolean   columns_scaled;    /* TRUE is the columns are scaled too,
				       * Only use if all variables are
				       * non-integer */
        gnm_float *scale;            /* sum_alloc+1:0..Rows the scaling of
				       * the Rows, Rows+1..Sum the scaling
				       * of the columns */

        int	   nr_lagrange;       /* Nr. of Langrangian relaxation
				       * constraints */
        gnm_float **lag_row;	      /* NumLagrange, columns+1:Pointer to
				       * pointer of rows */
        gnm_float *lag_rhs;	      /* NumLagrange :Pointer to pointer of
				       * Rhs */
        gnm_float *lambda;	      /* NumLagrange :Lambda Values */

        SolverConstraintType
	           *lag_con_type;     /* NumLagrange :TRUE if constraint type
				       * SolverEQ */
        gnm_float lag_bound;	      /* the lagrangian lower bound */

        gboolean   valid;	      /* Has this lp pased the 'test' */
        gnm_float infinite;          /* ## numerical stuff */
        gnm_float epsilon;           /* ## */
        gnm_float epsb;              /* ## */
        gnm_float epsd;              /* ## */
        gnm_float epsel;             /* ## */
} lprec;



/* function interface for the user */

lprec      *lp_solve_make_lp (int rows, int columns);
/* create and initialise a lprec structure
   defaults:
   Empty (Rows * Columns) matrix,
   Minimise the objective function
   constraints all type <=
   Upperbounds all Infinite
   no integer variables
   floor first in B&B
   no scaling
   default basis */

void       lp_solve_delete_lp (lprec *lp);
/* Remove problem from memory */

void lp_solve_set_max_iter (lprec *lp, int max);
/* Set the maximum number of iterations. */

void lp_solve_set_max_time (lprec *lp, int max, gnm_float st);
/* Set the maximum time. */

void       lp_solve_set_mat (lprec *lp, int row, int column, gnm_float value);
/* fill in element (Row,Column) of the matrix
   Row in [0..Rows] and Column in [1..Columns] */

void       lp_solve_set_obj_fn (lprec *lp, int col, gnm_float value);
/* set the objective function of the matrix */

void       lp_solve_set_upbo (lprec *lp, int column, gnm_float value);
/* Set the upperbound of a variable */

void       lp_solve_set_lowbo (lprec *lp, int column, gnm_float value);
/* Set the lowerbound of a variable */

void       lp_solve_set_int (lprec *lp, int column, gboolean must_be_int);
/* Set the type of variable, if must_be_int = TRUE then the variable must
 * be integer */

void       lp_solve_set_rh (lprec *lp, int row, gnm_float value);
/* Set the right hand side of a constraint row */

void       lp_solve_set_maxim (lprec *lp);
/* maximise the objective function */

void       lp_solve_set_minim (lprec *lp);
/* minimise the objective function */

void       lp_solve_set_constr_type (lprec *lp, int row,
				     SolverConstraintType con_type);
/* Set the type of constraint in row Row (SolverLE, SolverGE, SolverEQ) */

void       lp_solve_auto_scale (lprec *lp);
/* Automatic scaling of the problem */

SolverStatus lp_solve_solve (lprec *lp);
/* Solve the problem */

void       lp_solve_print_lp (lprec *lp);
/* Print the current problem, only usefull in very small (test) problems. 
 * Shows the effect of scaling */

void       lp_solve_print_solution (lprec *lp);
/* Print the solution to stdout */

gnm_float lp_solve_get_solution (lprec *lp, int column);
/* returns the value of a variable at the given column */

gnm_float lp_solve_get_value_of_obj_fn (lprec *lp);
/* returns the value of the objective function. */

gnm_float lp_solve_get_dual (lprec *lp, int row);
/* returns the value of a dual. */

/* functions used internaly by the lp toolkit */
void       lp_solve_unscale_columns (lprec *lp);
void       btran (lprec *lp, gnm_float *row);
void       invert (lprec *lp);
void       presolve (lprec *lp);

#endif

