/*
 * application.c: Manage the data common to all workbooks
 *
 * Author:
 *     Jody Goldberg <jgoldberg@home.com>
 *
 */
#include <config.h>
#include "application.h"
#include "clipboard.h"
#include "selection.h"
#include "sheet.h"
#include "workbook.h"
#include "workbook-view.h"

typedef struct
{
	/* Clipboard */
	Sheet		*clipboard_sheet;
	CellRegion	*clipboard_copied_contents;
	Range		 clipboard_cut_range;

	/* Display resolution */
	float horizontal_dpi, vertical_dpi;

	/* History for file menu */
	GList           *history_list;

	gboolean         edit_auto_complete;
} GnumericApplication;

static GnumericApplication app;

/**
 * application_init:
 *
 * Initialize the application specific data structures.
 */
void
application_init (void)
{
	app.clipboard_copied_contents = NULL;
	app.clipboard_sheet = NULL;

	/* FIXME : 96 as the default will scale Helvetica-9 to Helvetica-12
	 * which is not too ugly.  Ideally we could get the correct values here but,
	 *
	 * XFree86-3) lies.  It defaults to 75x75 dpi and only allows the user to 
	 *            specify one resolution, which it uses for both axis.
	 * XFree86-4) Makes a better guess, but still seems to use the same
	 *            resolution for both directions.
	 *
	 * XL seems to assume that everything is 96 dpi.
	 *
	 * I'll leave it as is for now, and revisit the solution when we shake
	 * out the flaws in the display code.
	 */
	gnome_config_push_prefix ("Gnumeric/Screen_Resolution/"); 
	app.horizontal_dpi = gnome_config_get_float ("Horizontal_dpi=96");
	app.vertical_dpi = gnome_config_get_float ("Vertical_dpi=96");
	gnome_config_pop_prefix ();

	gnome_config_push_prefix ("Gnumeric/Editing/");
	app.edit_auto_complete = gnome_config_get_bool ("AutoComplete=true");
	gnome_config_pop_prefix ();
}

/**
 * application_clipboard_clear:
 *
 * Clear and free the contents of the clipboard if it is
 * not empty.
 */
void
application_clipboard_clear (void)
{
	if (app.clipboard_copied_contents) {
		clipboard_release (app.clipboard_copied_contents);
		app.clipboard_copied_contents = NULL;
	}
	if (app.clipboard_sheet != NULL) {
		Sheet *sheet = app.clipboard_sheet;

		sheet_selection_unant (sheet);
		workbook_view_set_paste_special_state (sheet->workbook, FALSE);
		app.clipboard_sheet = NULL;

		/* Release the selection */
		gtk_selection_owner_set (NULL,
					 GDK_SELECTION_PRIMARY,
					 GDK_CURRENT_TIME);
	}
}

void
application_clipboard_unant (void)
{
	if (app.clipboard_sheet != NULL)
		sheet_selection_unant (app.clipboard_sheet);
}

static gboolean
application_set_selected_sheet (Sheet *sheet)
{
	g_return_val_if_fail (sheet != NULL, FALSE);

	/* Short circuit if we already have the selection */
	if (app.clipboard_sheet == sheet) {
		if (app.clipboard_copied_contents) {
			clipboard_release (app.clipboard_copied_contents);
			app.clipboard_copied_contents = NULL;
		}
		sheet_selection_unant (sheet);
		return TRUE;
	}

	application_clipboard_clear ();

	if (gtk_selection_owner_set (sheet->workbook->toplevel,
				     GDK_SELECTION_PRIMARY,
				     GDK_CURRENT_TIME)) {
		app.clipboard_sheet = sheet;
		return TRUE;
	}

	g_warning ("Unable to set selection ?");

	return FALSE;
}

/**
 * application_clipboard_copy:
 *
 * @sheet : The source sheet for the copy.
 * @area  : A single rectangular range to be copied.
 *
 * Clear and free the contents of the clipboard and COPY the designated region
 * into the clipboard.
 */
void
application_clipboard_copy (Sheet *sheet, Range const *area)
{
	g_return_if_fail (sheet != NULL);
	g_return_if_fail (area != NULL);

	if (application_set_selected_sheet (sheet) ) {
		app.clipboard_cut_range = *area;
		app.clipboard_copied_contents = 
			clipboard_copy_range (sheet, area);

		workbook_view_set_paste_special_state (sheet->workbook, TRUE);

		sheet_selection_ant (sheet);
	}
}

/**
 * application_clipboard_cut:
 *
 * @sheet : The source sheet for the copy.
 * @area  : A single rectangular range to be cut.
 *
 * Clear and free the contents of the clipboard and save the sheet and area
 * to be cut.  DO NOT ACTUALLY CUT!  Paste will move the region if this was a
 * cut operation.
 */
void
application_clipboard_cut (Sheet *sheet, Range const *area)
{
	g_return_if_fail (sheet != NULL);
	g_return_if_fail (area != NULL);

	if (application_set_selected_sheet (sheet) ) {
		app.clipboard_cut_range = *area;

		/* No paste special for copies */
		workbook_view_set_paste_special_state (sheet->workbook, FALSE);

		sheet_selection_ant (sheet);
	}
}

gboolean
application_clipboard_is_empty (void)
{
	return app.clipboard_sheet == NULL;
}

Sheet *
application_clipboard_sheet_get (void)
{
	return app.clipboard_sheet;
}

CellRegion *
application_clipboard_contents_get (void)
{
	return app.clipboard_copied_contents;
}

Range const *
application_clipboard_area_get (void)
{
	/*
	 * Only return the range if the sheet has been set.
	 * The range will still contain data even after
	 * the clipboard has been cleared so we need to be extra
	 * safe and only return a range if there is a valid selection
	 */
	if (app.clipboard_sheet != NULL)
		return &app.clipboard_cut_range;
	return NULL;
}

struct wb_name_closure
{
	Workbook *wb;
	char const * name;
};
static gboolean
cb_workbook_name (Workbook * wb, gpointer closure)
{
	struct wb_name_closure *dat = closure;
	if (0 == strcmp (wb->filename, dat->name)) {
		dat->wb = wb;
		return FALSE;
	}
	return TRUE;
}

Workbook *
application_workbook_get_by_name (char const * const name)
{
	struct wb_name_closure close;
	close.wb = NULL;
	close.name = name;
	workbook_foreach (&cb_workbook_name, &close);

	return close.wb;
}

struct wb_index_closure
{
	Workbook *wb;
	int index;	/* 1 based */
};
static gboolean
cb_workbook_index (Workbook * wb, gpointer closure)
{
	struct wb_index_closure *dat = closure;
	return (--(dat->index) != 0);
}

Workbook *
application_workbook_get_by_index (int i)
{
	struct wb_index_closure close;
	close.wb = NULL;
	close.index = i;
	workbook_foreach (&cb_workbook_index, &close);

	return close.wb;
}

float
application_display_dpi_get (gboolean const horizontal)
{
    return horizontal ? app.horizontal_dpi : app.vertical_dpi;
}

void
application_display_dpi_set (gboolean const horizontal, float const val)
{
    if (horizontal)
	    app.horizontal_dpi  = val;
    else
	    app.vertical_dpi = val;
}

/**
 * application_history_get_list:
 * 
 *  This function returns a pointer to the history list,
 * creating it if neccessary.
 * 
 * Return value: the list./
 **/
GList*
application_history_get_list (void)
{
        gchar *filename, *key;
        gint max_entries, i;
	gboolean do_set = FALSE;

	/* If the list is already populated, return it. */
	if (app.history_list)
		return app.history_list;

        gnome_config_push_prefix ("/Gnumeric/History/");

        /* Get maximum number of history entries.  Write default value to 
	 * config file if no entry exists. */
        max_entries = gnome_config_get_int_with_default ("MaxFiles=4", &do_set);
	if (do_set)
		gnome_config_set_int ("MaxFiles", 4);

	/* Read the history filenames from the config file */
        for (i = 0; i < max_entries; i++) {
		key = g_strdup_printf ("File%d", i);
 	        filename = gnome_config_get_string (key);
 	        if (filename == NULL) {
                       /* Ran out of filenames. */
                       g_free (key);
                       break;
		}
		app.history_list = g_list_append (app.history_list, filename);
               	g_free (key);
       	}
       	gnome_config_pop_prefix ();

	return app.history_list;
}

/**
 * application_history_update_list:
 * @filename: 
 * 
 * This function updates the history list.  The return value is a 
 * pointer to the filename that was removed, if the list was already full
 * or NULL if no items were removed.
 * 
 * Return value: 
 **/
gchar *
application_history_update_list (gchar *filename)
{
	gchar *name, *old_name = NULL;
	GList *l = NULL;
	GList *new_list = NULL;
	gint max_entries, count = 0;
	gboolean do_set = FALSE;
	gboolean found = FALSE;

	g_return_val_if_fail (filename != NULL, NULL);

	/* Get maximum list length from config */
	gnome_config_push_prefix ("Gnumeric/History/");
	max_entries = gnome_config_get_int_with_default ("MaxFiles=4", &do_set);
	if (do_set)
		gnome_config_set_int ("MaxFiles", max_entries);
	gnome_config_pop_prefix ();

	/* Check if this filename already exists in the list */
	for (l = app.history_list; l && (count < max_entries); l = l->next) {

		if (!found && (!strcmp ((gchar *)l->data, filename) || 
			       (count == max_entries - 1))) {
			/* This is either the last item in the list, or a
			 * duplicate of the requested entry. */
			old_name = (gchar *)l->data;
			found = TRUE;
		} else  /* Add this item to the new list */
			new_list = g_list_append (new_list, l->data);

		count++;
	}

	/* Insert the new filename to the new list and free up the old list */
	name = g_strdup (filename);
	new_list = g_list_prepend (new_list, name);
	g_list_free (app.history_list);
	app.history_list = new_list;

	return old_name;
}

/* Remove the last item from the history list and return it. */
gchar *
application_history_list_shrink (void)
{
	gchar *name;
	GList *l;

	if (app.history_list == NULL)
		return NULL;

	l = g_list_last (app.history_list);
	name = (gchar *)l->data;
	app.history_list = g_list_remove (app.history_list, name);

	return name;
}

/* Write contents of the history list to the configuration file. */
void
application_history_write_config (void)
{
	gchar *key; 
	GList *l;
	gint max_entries, i = 0;

	if (app.history_list == NULL) return;

	max_entries = gnome_config_get_int ("Gnumeric/History/MaxFiles=4");
	gnome_config_clean_section ("Gnumeric/History");
	gnome_config_push_prefix("Gnumeric/History/");
	gnome_config_set_int ("MaxFiles", max_entries);

	for (l = app.history_list; l; l = l->next) {
		key = g_strdup_printf ("File%d", i++);
		gnome_config_set_string (key, (gchar *)l->data);
		g_free (l->data);
		g_free (key);
	}

	gnome_config_sync ();
	gnome_config_pop_prefix ();
	g_list_free (app.history_list);
	app.history_list = NULL;
}

gboolean
application_use_auto_complete_get (void)
{
	return app.edit_auto_complete;
}

void
application_use_auto_complete_set (gboolean use_auto_complete)
{
	app.edit_auto_complete = use_auto_complete;
}
