/*
 * ranges.c: various functions for common operations on cell ranges.
 *
 * Author:
 *   Miguel de Icaza (miguel@gnu.org).
 *   Michael Meeks   (mmeeks@gnu.org).
 *
 */
#include <config.h>
#include <stdlib.h>
#include <stdio.h>
#include <glib.h>
#include <string.h>
#include <string.h>
#include <ctype.h>
#include "numbers.h"
#include "symbol.h"
#include "expr.h"
#include "parse-util.h"
#include "gnumeric.h"
#include "ranges.h"
#include "value.h"

#undef RANGE_DEBUG

Range *
range_init (Range *r, int start_col, int start_row,
	    int end_col, int end_row)
{
	g_return_val_if_fail (r != NULL, r);

	r->start.col = start_col;
	r->start.row = start_row;
	r->end.col   = end_col;
	r->end.row   = end_row;

	return r;
}

/**
 * range_parse:
 * @sheet: the sheet where the cell range is evaluated
 * @range: a range specification (ex: "A1", "A1:C3").
 *
 * Returns a (Value *) of type VALUE_CELLRANGE if the @range was
 * succesfully parsed or NULL on failure.
 */
Value *
range_parse (Sheet *sheet, const char *range, gboolean strict)
{
	CellRef a, b;
	Value *v;
	int n_read = 0;
	
	g_return_val_if_fail (range != NULL, FALSE);

	a.col_relative = 0;
	b.col_relative = 0;
	a.row_relative = 0;
	b.row_relative = 0;

	if (!parse_cell_name (range, &a.col, &a.row, strict, &n_read))
		return FALSE;

	a.sheet = sheet;

	if (range [n_read] == ':'){
		if (!parse_cell_name (&range [n_read+1], &b.col, &b.row, strict, NULL))
			return FALSE;

		b.sheet = sheet;
	} else
		b = a;

	/*
	 * We can dummy out the calling cell because we know that both
	 * refs use the same mode.  This will handle inversions.
	 */
	v = value_new_cellrange (&a, &b, 0, 0);
	
	return v;
}

/* Pulled from dialog-analysis-tools.c
 * Should be merged with range_parse
 */
int
parse_range (char *text, int *start_col, int *start_row,
	     int *end_col, int *end_row)
{
        char buf[256];
        char *p;

	strcpy(buf, text);
	p = strchr(buf, ':');
	if (p == NULL)
	        return 0;
	*p = '\0';
	if (!parse_cell_name (buf, start_col, start_row, TRUE, NULL))
	        return 0;
	if (!parse_cell_name (p+1, end_col, end_row, TRUE, NULL))
	        return 0;
	return 1;
}

/**
 * range_list_destroy:
 * @ranges: a list of value ranges to destroy. 
 *
 * Destroys a list of ranges returned from parse_cell_range_list
 */
void
range_list_destroy (GSList *ranges)
{
	GSList *l;

	for (l = ranges; l; l = l->next){
		Value *value = l->data;
		
		value_release (value);
	}
	g_slist_free (ranges);
}


/**
 * range_list_parse:
 * @sheet: Sheet where the range specification is relatively parsed to
 * @range_spec: a range or list of ranges to parse (ex: "A1", "A1:B1,C2,D2:D4")
 * @strict: whether we should be strict during the parsing or allow for trailing garbage
 *
 * Parses a list of ranges, relative to the @sheet and returns a list with the
 * results.
 *
 * Returns a GSList containing Values of type VALUE_CELLRANGE, or NULL on failure
 */
GSList *
range_list_parse (Sheet *sheet, const char *range_spec, gboolean strict)
{
	
	char *copy, *range_copy, *r;
	GSList *ranges = NULL;
	
	g_return_val_if_fail (sheet != NULL, NULL);
	g_return_val_if_fail (IS_SHEET (sheet), NULL);
	g_return_val_if_fail (range_spec != NULL, NULL);
	
	range_copy = copy = g_strdup (range_spec);

	while ((r = strtok (range_copy, ",")) != NULL){
		Value *v;

		v = range_parse (sheet, r, strict);
		if (!v){
			range_list_destroy (ranges);
			g_free (copy);
			return NULL;
		}

		ranges = g_slist_prepend (ranges, v);
		range_copy = NULL;
	}
	
	g_free (copy);

	return ranges;
}

/**
 * range_list_foreach_full:
 * 
 * foreach cell in the range, make sure it exists, and invoke the routine
 * @callback on the resulting cell, passing @data to it
 *
 */
void
range_list_foreach_full (GSList *ranges, void (*callback)(Cell *cell, void *data),
			 void *data, gboolean create_empty)
{
	GSList *l;

	{
		static int message_shown;

		if (!message_shown){
			g_warning ("This routine should also iterate"
				   "through the sheets in the ranges");
			message_shown = TRUE;
		}
	}

	for (l = ranges; l; l = l->next){
		Value *value = l->data;
		CellRef a, b;
		int col, row;
		
		g_assert (value->type == VALUE_CELLRANGE);

		/*
		 * FIXME : Are these ranges normalized ?
		 *         Are they absolute ?
		 */
		a = value->v_range.cell.a;
		b = value->v_range.cell.b;

		for (col = a.col; col <= b.col; col++)
			for (row = a.row; row < b.row; row++){
				Cell *cell;

				if (create_empty)
					cell = sheet_cell_fetch (a.sheet, col, row);
				else
					cell = sheet_cell_get (a.sheet, col, row);
				if (cell)
					(*callback)(cell, data);
			}
	}
}

void
range_list_foreach_all (GSList *ranges,
			void (*callback)(Cell *cell, void *data),
			void *data)
{
	range_list_foreach_full (ranges, callback, data, TRUE);
}

void
range_list_foreach (GSList *ranges, void (*callback)(Cell *cell, void *data),
		    void *data)
{
	range_list_foreach_full (ranges, callback, data, FALSE);
}

/**
 * range_list_foreach_full:
 * @sheet:     The current sheet.
 * @ranges:    The list of ranges.
 * @callback:  The user function
 * @user_data: Passed to callback intact.
 * 
 * Iterates over the ranges calling the callback with the
 * range, sheet and user data set
 **/
void
range_list_foreach_area (Sheet *sheet, GSList *ranges,
			 void (*callback)(Sheet       *sheet,
					  const Range *range,
					  gpointer     user_data),
			 gpointer user_data)
{
	GSList *l;

	g_return_if_fail (sheet != NULL);

	for (l = ranges; l; l = l->next) {
		Value *value = l->data;
		Sheet *s;
		Range   range;
		
		g_assert (value->type == VALUE_CELLRANGE);

		/*
		 * FIXME : Are these ranges normalized ?
		 *         Are they absolute ?
		 */
		range.start.col = value->v_range.cell.a.col;
		range.start.row = value->v_range.cell.a.row;
		range.end.col   = value->v_range.cell.b.col;
		range.end.row   = value->v_range.cell.b.row;

		s = sheet;
		if (value->v_range.cell.b.sheet)
			s = value->v_range.cell.b.sheet;
		if (value->v_range.cell.a.sheet)
			s = value->v_range.cell.a.sheet;
		callback (s, &range, user_data);
	}
}

/**
 * range_adjacent:
 * @a: First range
 * @b: Second range
 * 
 * Detects whether a range of similar size is adjacent
 * to the other range. Similarity is determined by having
 * a shared side of equal length. NB. this will clearly
 * give odd results for overlapping regions.
 * 
 * Return value: if they share a side of equal length
 **/
gboolean
range_adjacent (Range const *a, Range const *b)
{
	int adx, bdx, ady, bdy;
       
	g_return_val_if_fail (a != NULL, FALSE);
	g_return_val_if_fail (b != NULL, FALSE);
	
	adx = a->end.col - a->start.col;
	bdx = b->end.col - b->start.col;
	ady = a->end.row - a->start.row;
	bdy = b->end.row - b->start.row;

	if ((a->start.col == b->start.col) &&
	    (a->end.col   == b->end.col)) {
		if (a->end.row + 1 == b->start.row ||
		    b->end.row + 1 == a->start.row)
			return TRUE;
		else
			return FALSE;
	} else if ((a->start.row == b->start.row) &&
	    (a->end.row   == b->end.row)) {
		if (a->end.col + 1 == b->start.col ||
		    b->end.col + 1 == a->start.col)
			return TRUE;
		else
			return FALSE;
	}
	return FALSE;
}

/**
 * range_merge:
 * @a: Range a.
 * @b: Range b.
 * 
 * This routine coalesces two adjacent regions, eg.
 * (A1, B1) would return A1:B1 or (A1:B2, C1:D2)) would
 * give A1:D2. NB. it is imperative that the regions are
 * actualy adjacent or unexpected results will ensue.
 *
 * Fully commutative.
 * 
 * Return value: the merged range.
 **/
Range
range_merge (Range const *a, Range const *b)
{
	Range ans;

	ans.start.col = 0;
	ans.start.row = 0;
	ans.end.col   = 0;
	ans.end.row   = 0;

	g_return_val_if_fail (a != NULL, ans);
	g_return_val_if_fail (b != NULL, ans);

/*      Useful perhaps but kills performance */
/*	g_return_val_if_fail (range_adjacent (a, b), ans); */
	
	if (a->start.row < b->start.row) {
		ans.start.row = a->start.row;
		ans.end.row   = b->end.row;
	} else {
		ans.start.row = b->start.row;
		ans.end.row   = a->end.row;
	}

	if (a->start.col < b->start.col) {
		ans.start.col = a->start.col;
		ans.end.col   = b->end.col;
	} else {
		ans.start.col = b->start.col;
		ans.end.col   = a->end.col;
	}

	return ans;
}

const char *
range_name (Range const *src)
{
	static char buffer [(2 + 4 * sizeof (long)) * 2 + 1];
	char *name;
	
	if (src->start.col != src->end.col ||
	    src->start.row != src->end.row) {
		name = g_strdup (col_name (src->start.col));
		sprintf (buffer, "%s%d:%s%d", name, src->start.row + 1,
			 col_name (src->end.col), src->end.row + 1);
		g_free (name);
	} else {
		sprintf (buffer, "%s%d",
			 col_name (src->start.col),
			 src->start.row + 1);
	}
	
	return buffer;
}

void
range_dump (Range const *src)
{
	/*
	 * keep these as 2 print statements, because
	 * col_name uses a static buffer
	 */
	fprintf (stderr, "%s%d",
		col_name (src->start.col),
		src->start.row + 1);

	if (src->start.col != src->end.col ||
	    src->start.row != src->end.row)
		fprintf (stderr, ":%s%d\n",
			col_name (src->end.col),
			src->end.row + 1);
	else
		fputc ('\n', stderr);
}

static void
ranges_dump (GList *l, char *txt)
{
	fprintf (stderr, "%s", txt);
	for (; l; l = l->next)
		range_dump (l->data);
	fprintf (stderr, "\n");
}

/**
 * range_contained:
 * @a: 
 * @b: 
 * 
 * Is @a totaly contained by @b
 * 
 * Return value: 
 **/
gboolean
range_contained (Range const *a, Range const *b)
{
	if (a->start.row < b->start.row)
		return FALSE;

	if (a->end.row > b->end.row)
		return FALSE;

	if (a->start.col < b->start.col)
		return FALSE;
       
	if (a->end.col > b->end.col)
		return FALSE;

	return TRUE;
}

/**
 * range_split_ranges:
 * @hard: The region that is split against
 * @soft: The region that is split
 * @copy_fn: the function used to create the copies or NULL.
 * 
 * Splits soft into several chunks, and returns the still
 * overlapping remainder of soft as the first list item
 * ( the central region in the pathalogical case ).
 *
 * Return value: A list of fragments.
 **/
GList *
range_split_ranges (const Range *hard, const Range *soft,
		    RangeCopyFn copy_fn)
{
	/*
	 * There are lots of cases so think carefully.
	 *
	 * Original Methodology ( approximately )
	 *	a) Get a vertex: is it contained ?
	 *	b) Yes: split so it isn't
	 *	c) Continue for all verticees.
	 *
	 * NB. We prefer to have long columns at the expense
	 *     of long rows.
	 */
	GList *split  = NULL;
	Range *middle, *sp;
	gboolean split_left  = FALSE;
	gboolean split_right = FALSE;

	g_return_val_if_fail (range_overlap (hard, soft), NULL);

	if (copy_fn)
		middle = copy_fn (soft);
	else {
		middle = g_new (Range, 1);
		*middle = *soft;
	}

	/* Split off left entirely */
	if (hard->start.col > soft->start.col) {
		if (copy_fn)
			sp = copy_fn (middle);
		else
			sp = g_new (Range, 1);
		sp->start.col = soft->start.col;
		sp->start.row = soft->start.row;
		sp->end.col   = hard->start.col - 1;
		sp->end.row   = soft->end.row;
		split = g_list_prepend (split, sp);

		middle->start.col = hard->start.col;
		split_left = TRUE;
	} /* else shared edge */

	/* Split off right entirely */
	if (hard->end.col < soft->end.col) {
		if (copy_fn)
			sp = copy_fn (middle);
		else
			sp = g_new (Range, 1);
		sp->start.col = hard->end.col + 1;
		sp->start.row = soft->start.row;
		sp->end.col   = soft->end.col;
		sp->end.row   = soft->end.row;
		split = g_list_prepend (split, sp);

		middle->end.col = hard->end.col;
		split_right = TRUE;
	}  /* else shared edge */

	/* Top */
	if (split_left && split_right) {
		if (hard->start.row > soft->start.row) {
			/* The top middle bit */
			if (copy_fn)
				sp = copy_fn (middle);
			else
				sp = g_new (Range, 1);
			sp->start.col = hard->start.col;
			sp->start.row = soft->start.row;
			sp->end.col   = hard->end.col;
			sp->end.row   = hard->start.row - 1;
			split = g_list_prepend (split, sp);

			middle->start.row = hard->start.row;
		} /* else shared edge */
	} else if (split_left) {
		if (hard->start.row > soft->start.row) {
			/* The top middle + right bits */
			if (copy_fn)
				sp = copy_fn (middle);
			else
				sp = g_new (Range, 1);
			sp->start.col = hard->start.col;
			sp->start.row = soft->start.row;
			sp->end.col   = soft->end.col;
			sp->end.row   = hard->start.row - 1;
			split = g_list_prepend (split, sp);

			middle->start.row = hard->start.row;
		} /* else shared edge */
	} else if (split_right) {
		if (hard->start.row > soft->start.row) {
			/* The top middle + left bits */
			if (copy_fn)
				sp = copy_fn (middle);
			else
				sp = g_new (Range, 1);
			sp->start.col = soft->start.col;
			sp->start.row = soft->start.row;
			sp->end.col   = hard->end.col;
			sp->end.row   = hard->start.row - 1;
			split = g_list_prepend (split, sp);

			middle->start.row = hard->start.row;
		} /* else shared edge */
	} else {
		if (hard->start.row > soft->start.row) {
			/* Hack off the top bit */
			if (copy_fn)
				sp = copy_fn (middle);
			else
				sp = g_new (Range, 1);
			sp->start.col = soft->start.col;
			sp->start.row = soft->start.row;
			sp->end.col   = soft->end.col;
			sp->end.row   = hard->start.row - 1;
			split = g_list_prepend (split, sp);

			middle->start.row = hard->start.row;
		} /* else shared edge */
	}

	/* Bottom */
	if (split_left && split_right) {
		if (hard->end.row < soft->end.row) {
			/* The bottom middle bit */
			if (copy_fn)
				sp = copy_fn (middle);
			else
				sp = g_new (Range, 1);
			sp->start.col = hard->start.col;
			sp->start.row = hard->end.row + 1;
			sp->end.col   = hard->end.col;
			sp->end.row   = soft->end.row;
			split = g_list_prepend (split, sp);

			middle->end.row = hard->end.row;
		} /* else shared edge */
	} else if (split_left) {
		if (hard->end.row < soft->end.row) {
			/* The bottom middle + right bits */
			if (copy_fn)
				sp = copy_fn (middle);
			else
				sp = g_new (Range, 1);
			sp->start.col = hard->start.col;
			sp->start.row = hard->end.row + 1;
			sp->end.col   = soft->end.col;
			sp->end.row   = soft->end.row;
			split = g_list_prepend (split, sp);

			middle->end.row = hard->end.row;
		} /* else shared edge */
	} else if (split_right) {
		if (hard->end.row < soft->end.row) {
			/* The bottom middle + left bits */
			if (copy_fn)
				sp = copy_fn (middle);
			else
				sp = g_new (Range, 1);
			sp->start.col = soft->start.col;
			sp->start.row = hard->end.row + 1;
			sp->end.col   = hard->end.col;
			sp->end.row   = soft->end.row;
			split = g_list_prepend (split, sp);

			middle->end.row = hard->end.row;
		} /* else shared edge */
	} else {
		if (hard->end.row < soft->end.row) {
			/* Hack off the bottom bit */
			if (copy_fn)
				sp = copy_fn (middle);
			else
				sp = g_new (Range, 1);
			sp->start.col = soft->start.col;
			sp->start.row = hard->end.row + 1;
			sp->end.col   = soft->end.col;
			sp->end.row   = soft->end.row;
			split = g_list_prepend (split, sp);

			middle->end.row = hard->end.row;
		} /* else shared edge */
	}

	return g_list_prepend (split, middle);
}

/**
 * range_copy:
 * @a: Source range to copy
 *
 * Copies the @a range.
 *
 * Return value: A copy of the Range.
 */
Range *
range_copy (const Range *a)
{
	Range *r = g_new (Range, 1);
	*r = *a;
	return r;
}

/**
 * range_fragment:
 * @a: Range a
 * @b: Range b
 * @copy_fn: Optional copy fn.
 * 
 * Fragments the ranges into totaly overlapping regions,
 * optional copy_fn ( NULL ) for default, copies the ranges.
 * NB. commutative.
 * 
 * Return value: A list of fragmented ranges or at minimum
 * simply a and b.
 **/
GList *
range_fragment (const Range *a, const Range *b)
{
	GList *split, *ans = NULL;
	
	split = range_split_ranges (a, b, NULL);
	ans   = g_list_concat (ans, split);
	
	split = range_split_ranges (b, a, NULL);
	if (split) {
		g_free (split->data);
		split = g_list_remove (split, split->data);
	}
	ans = g_list_concat (ans, split);

	return ans;
}

/**
 * range_fragment_list:
 * @ra: A list of possibly overlapping ranges.
 * 
 *  Converts the ranges into non-overlapping sub-ranges.
 * 
 * Return value: new list of fully overlapping ranges.
 **/
GList *
range_fragment_list (const GList *ra)
{
	return range_fragment_list_clip (ra, NULL);
}


/**
 *  The more I read this code the more I am convinced that
 * it is badly broken by design.
 **/
GList *
range_fragment_list_clip (const GList *ra, const Range *clip)
{
	GList *ranges, *l, *a, *remove, *nexta;

	if (clip)
		ranges = g_list_prepend (NULL, range_copy (clip));
	else
		ranges = NULL;

	for (l = (GList *)ra; l; l = l->next)
		ranges = g_list_prepend (ranges, range_copy (l->data));

#ifdef RANGE_DEBUG
	ranges_dump (ranges, "On entry : ");
#endif

	remove = NULL;

	for (a = ranges; a; a = nexta) {
		GList   *b, *nextb;
		gboolean done_split = FALSE;
		
		for (b = a->next; b && !done_split; b = nextb) {
			nextb = b->next;

			if (range_equal (a->data, b->data)) {
				g_free (b->data);
				if (g_list_remove (ranges, b->data) != ranges)
					g_error ("Impossible overwrite");

			} else if (range_overlap (a->data, b->data)) {
				GList *split;
				
				split = range_fragment (a->data, b->data);

				g_assert (split);
				if (g_list_concat (ranges, split) != ranges)
					g_error ("Overwriting ourselfs");

				done_split = TRUE;
				break;
			}
		}

		nexta = a->next;
		if (done_split) {
			g_free (a->data);
			ranges = g_list_remove (ranges, a->data);
			
			if (b == nexta)
				nexta = b->next;

			g_free (b->data);
			ranges = g_list_remove (ranges, b->data);
		}
	}

	if (clip) {
#ifdef RANGE_DEBUG
		fprintf (stderr, "Clip : ");
		range_dump (clip);
#endif
		for (a = ranges; a; a = a->next) {
			if (!range_contained (a->data, clip)) {
#ifdef RANGE_DEBUG
				fprintf (stderr, "Uncontained: ");
				range_dump (a->data);
#endif
				remove = g_list_prepend (remove, a->data);
			}
		}
	}

#ifdef RANGE_DEBUG
	ranges_dump (remove, "Removing : ");
#endif
	while (remove) {
		gpointer kill;
		
		kill = remove->data;
		
		ranges = g_list_remove (ranges, kill);
		remove = g_list_remove (remove, kill);
		
		g_free (kill);
	}

#ifdef RANGE_DEBUG
	ranges_dump (ranges, "Answer : ");
#endif

	return ranges;
}

void
range_fragment_free (GList *fragments)
{
	GList *l = fragments;

	for (l = fragments; l; l = g_list_next (l))
		g_free (l->data);

	g_list_free (fragments);
}

/**
 * range_intersection:
 * @r: intersection range
 * @a: range a
 * @b: range b
 * 
 * This computes the intersection of two ranges; on a Venn
 * diagram this would be A (upside down U) B.
 * If the ranges do not intersect, false is returned an the
 * values of r are unpredictable. 
 *
 * NB. totally commutative
 *
 * Return value: True if the ranges intersect, false otherwise
 **/
gboolean
range_intersection (Range *r, Range const *a, Range const *b)
{
	g_return_val_if_fail (range_overlap (a, b), FALSE);
	
	r->start.col = MAX (a->start.col, b->start.col);
	r->start.row = MAX (a->start.row, b->start.row);
	
	r->end.col = MIN (a->end.col, b->end.col);
	r->end.row = MIN (a->end.row, b->end.row);

	return TRUE;
}

/**
 * range_normalize:
 * @a: a range
 * 
 * Ensures that start <= end for rows and cols.
 **/
void
range_normalize (Range *src)
{
	int tmp;

	tmp = src->end.col;
	if (src->start.col >= tmp) {
		src->end.col = src->start.col;
		src->start.col = tmp;
	}
	tmp = src->end.row;
	if (src->start.row >= tmp) {
		src->end.row = src->start.row;
		src->start.row = tmp;
	}
}

/**
 * range_union:
 * @a: range a
 * @b: range b
 * 
 * This computes the union; on a Venn
 * diagram this would be A U B
 * NB. totally commutative. Also, this may
 * include cells not in either range since
 * it must return a Range.
 * 
 * Return value: the union
 **/
Range
range_union (Range const *a, Range const *b)
{
	Range ans;

	if (a->start.col < b->start.col)
		ans.start.col = a->start.col;
	else
		ans.start.col = b->start.col;

	if (a->end.col > b->end.col)
		ans.end.col   = a->end.col;
	else
		ans.end.col   = b->end.col;

	if (a->start.row < b->start.row)
		ans.start.row = a->start.row;
	else
		ans.start.row = b->start.row;

	if (a->end.row > b->end.row)
		ans.end.row   = a->end.row;
	else
		ans.end.row   = b->end.row;

	return ans;
}

gboolean
range_is_singleton (Range const *r)
{
	return r->start.col == r->end.col && r->start.row == r->end.row;
}

/**
 * range_is_infinite:
 * @r: the range.
 * 
 *  This determines whether @r completely spans a sheet
 * in either dimension ( semi-infinite )
 * 
 * Return value: TRUE if it is infinite else FALSE
 **/
gboolean
range_is_infinite (Range const *r)
{
	if (r->start.col == 0 &&
	    r->end.col   >= SHEET_MAX_COLS - 1)
		return TRUE;

	if (r->start.row == 0 &&
	    r->end.row   >= SHEET_MAX_ROWS - 1)
		return TRUE;
	
	return FALSE;
}

void
range_clip_to_finite (Range *range, Sheet *sheet)
{
	Range extent;
	
	extent = sheet_get_extent (sheet);
       	if (range->end.col >= SHEET_MAX_COLS - 2)
		range->end.col = extent.end.col;
	if (range->end.row >= SHEET_MAX_ROWS - 2)
		range->end.row = extent.end.row;	
}

static void
range_style_apply_cb (Sheet *sheet, const Range *range, gpointer user_data)
{
	mstyle_ref ((MStyle *)user_data);
	sheet_style_attach (sheet, *range, (MStyle *)user_data);
}

void
ranges_set_style (Sheet *sheet, GSList *ranges, MStyle *mstyle)
{
	range_list_foreach_area (sheet, ranges,
				 range_style_apply_cb, mstyle);
	mstyle_unref (mstyle);
}

/**
 * range_translate:
 * @range: 
 * @col_offset: 
 * @row_offset: 
 * 
 * Translate the range, returns TRUE if the range was clipped
 * otherwise FALSE.
 * 
 * Return value: range clipped.
 **/
gboolean
range_translate (Range *range, int col_offset,
		 int row_offset)
{
	gboolean clipped = FALSE;

	if (range->end.col   + col_offset < 0 ||
	    range->start.col + col_offset >= SHEET_MAX_COLS)
		clipped = TRUE;

	if (range->end.row   + row_offset < 0 ||
	    range->start.row + row_offset >= SHEET_MAX_ROWS)
		clipped = TRUE;

	range->start.col = MIN (range->start.col + col_offset,
				SHEET_MAX_COLS - 1);
	range->end.col   = MIN (range->end.col   + col_offset,
				SHEET_MAX_COLS - 1);
	range->start.row = MIN (range->start.row + row_offset,
				SHEET_MAX_ROWS - 1);
	range->end.row   = MIN (range->end.row   + row_offset,
				SHEET_MAX_ROWS - 1);

	if (range->start.col < 0) {
		range->start.col = 0;
		clipped = TRUE;
	}

	if (range->start.row < 0) {
		range->start.row = 0;
		clipped = TRUE;
	}

	if (range->end.col < 0) {
		range->end.col   = 0;
		clipped = TRUE;
	}

	if (range->end.row < 0) {
		range->end.row   = 0;
		clipped = TRUE;
	}

	return clipped;
}

/**
 * range_transpose:
 * @range: The range.
 * @boundary: The box to transpose inside
 * 
 *   Effectively mirrors the ranges in 'boundary' around a
 * leading diagonal projected from offset.
 * 
 * Return value: whether we clipped the range.
 **/
gboolean
range_transpose (Range *range, const CellPos *origin)
{
	gboolean clipped = FALSE;
	Range    src;
	int      t;

	g_return_val_if_fail (range != NULL, TRUE);

	src = *range;

	/* Start col */
	t = origin->col + (src.start.row - origin->row);
	if (t > SHEET_MAX_COLS - 1) {
		clipped = TRUE;
		range->start.col = SHEET_MAX_COLS - 1;
	} else if (t < 0) {
		clipped = TRUE;
		range->start.col = 0;
	}
		range->start.col = t;

	/* Start row */
	t = origin->row + (src.start.col - origin->col);
	if (t > SHEET_MAX_COLS - 1) {
		clipped = TRUE;
		range->start.row = SHEET_MAX_ROWS - 1;
	} else if (t < 0) {
		clipped = TRUE;
		range->start.row = 0;
	}
		range->start.row = t;


	/* End col */
	t = origin->col + (src.end.row - origin->row);
	if (t > SHEET_MAX_COLS - 1) {
		clipped = TRUE;
		range->end.col = SHEET_MAX_COLS - 1;
	} else if (t < 0) {
		clipped = TRUE;
		range->end.col = 0;
	}
		range->end.col = t;

	/* End row */
	t = origin->row + (src.end.col - origin->col);
	if (t > SHEET_MAX_COLS - 1) {
		clipped = TRUE;
		range->end.row = SHEET_MAX_ROWS - 1;
	} else if (t < 0) {
		clipped = TRUE;
		range->end.row = 0;
	}
		range->end.row = t;

	g_assert (range_valid (&range));

	return clipped;
}

/**
 * range_expand:
 * @range: the range to expand
 * @d_tlx: top left column delta
 * @d_tly: top left row delta
 * @d_brx: bottom right column delta
 * @d_bry: bottom right row delta
 * 
 * Attempts to expand a ranges area
 * 
 * Return value: TRUE if we can expand, FALSE if not enough room.
 **/
gboolean
range_expand (Range *range, int d_tlx, int d_tly, int d_brx, int d_bry)
{
	int t;

	t = range->start.col + d_tlx;
	if (t < 0 ||
	    t >= SHEET_MAX_COLS)
		return FALSE;
	range->start.col = t;

	t = range->start.row + d_tly;
	if (t < 0 ||
	    t >= SHEET_MAX_ROWS)
		return FALSE;
	range->start.row = t;

	t = range->end.col + d_brx;
	if (t < 0 ||
	    t >= SHEET_MAX_COLS)
		return FALSE;
	range->end.col = t;

	t = range->end.row + d_bry;
	if (t < 0 ||
	    t >= SHEET_MAX_ROWS)
		return FALSE;
	range->end.row = t;
	
	return TRUE;
}

