/* Gnome Scan - Scan as easy as you print
 * Copyright © 2007  Étienne Bersac <bersace03@laposte.net>
 *
 * Gnome Scan is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * gnome-scan is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with gnome-scan.  If not, write to:
	 *
 *	the Free Software Foundation, Inc.
 *	51 Franklin Street, Fifth Floor
 *	Boston, MA 02110-1301, USA
 */

/**
 * SECTION: gnome-scan-dialog
 * @short_description: Scan configuration dialog
 * @include: gnome-scan.h
 *
 * The goal of the #GnomeScanDialog is to allow the user to configure
 * a #GnomeScanJob, and trigger the execution of this job by running a
 * #GnomeScanAcquisitionDialog on it.
 **/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gdk-pixbuf/gdk-pixbuf.h>
#include <glib/gi18n.h>
#include "gnome-scan-private.h"
#include "gnome-scan-init.h"
#include "gnome-scan-dialog.h"
#include "gnome-scan-acquisition-dialog.h"
#include "gnome-scan-preview-area.h"
#include "gnome-scan-param-widget.h"
#include "gnome-scan-module-manager.h"
#include "gnome-scan-module.h"
#include "gnome-scan-backend.h"
#include "gnome-scan-param-specs.h"
#include "gnome-scanner.h"
#include "gnome-scan-preview-sink.h"
#include "gnome-scan-preview-plugin.h"

#define	GET_PRIVATE(o)	(G_TYPE_INSTANCE_GET_PRIVATE ((o), GNOME_TYPE_SCAN_DIALOG, GnomeScanDialogPrivate))

#define	GTK_WIDGET_SET_VISIBLE(wid,show)	if(wid){gtk_widget_set_no_show_all(wid,!show); \
if(show){gtk_widget_show_all(wid);}else{gtk_widget_hide(wid);}}

#define GTK_BOX_SET_VISIBLE(box,show,hval)		GTK_WIDGET_SET_VISIBLE(box,show); \
gsd_add_to_box_show(g_object_get_qdata(G_OBJECT (box), GSD_PAGE_QUARK),show?1:hval); \
GTK_WIDGET_SET_VISIBLE(gtk_widget_get_parent(box),gsd_get_box_show(gtk_widget_get_parent(box)))

#define	gsd_get_box_show(box)			(guint)g_object_get_data(G_OBJECT(box),BOX_SHOWN)
#define	gsd_set_box_show(box,val)		g_object_set_data(G_OBJECT(box),BOX_SHOWN,(gpointer)(val))
#define gsd_init_box_show(box)			gsd_set_box_show(box,0)
#define	gsd_add_to_box_show(box,val)	        gsd_set_box_show(box,gsd_get_box_show(box)+(val))

#define	BOX_SHOWN	"box-show"
#define	PREVIEW_RES	75.

typedef struct _GnomeScanDialogPrivate GnomeScanDialogPrivate;

struct _GnomeScanDialogPrivate
{
	/* 	Properties  */
	GnomeScanJob*	job;
	
	/* Internals */
	gboolean		disposed;
	GSList*			backends;
	gint			probing_backends;
	gboolean		probe_done;
	gint			scanner_count;
	gulong			scanner_changed_handler;
	gboolean		run;
	
	/* 	Widgets  */
	GtkWidget*		notebook;
	
	/* General */
	GtkWidget*		general_page;
	GtkWidget*		front_scanner_box;
	GtkWidget*		front_sink_box;
	/* front_custom_box ? */
	GtkListStore*		scanners;
	GtkTreeSelection*	scanner_selection;
	
	/* preview */
	GtkWidget*		preview_page;
	GtkWidget*		preview_acquisition_box;
	GtkWidget*		preview_progress;
	GtkWidget*		preview_stage;
	GtkWidget*		preview_box;
	GtkWidget*		preview_area;
	GtkWidget*		preview_bbox;
	GnomeScanJob*	preview_job;
	GValue*		preview_res;
	GValue*		saved_res;
	GValue*		saved_origin;
	GValue*		saved_paper_size;
	
	/* advanced tab */
	GtkWidget*		advanced_page;
	GtkWidget*		advanced_container;
	GtkWidget*		advanced_box;
	
	GtkWidget*		sink_box;
	GtkWidget*		sink_page;
	GtkWidget*		custom_widget;
	
	/* processing */
	GtkWidget *processing_page;
	GtkWidget *processing_container;
	GtkWidget *processing_box;
	
};

enum
{
	PAGE_GENERAL,
	PAGE_PREVIEW,
	PAGE_ADVANCED,
	PAGE_PROCESSING,
	PAGE_SINK,
	PAGE_CUSTOM
};

enum
{
	COLUMN_ICON,
	COLUMN_NAME,
	COLUMN_STATUS,
	COLUMN_OBJECT,
	N_COLUMNS
};

enum
{
	PROP_0,
	PROP_JOB
};

static GtkDialogClass* parent_class = NULL;

static void	gsd_message_dialog		(GnomeScanDialog *dialog,
										  GtkMessageType mtype,
										  GtkButtonsType btype,
										  const gchar* primary,
										  const gchar* secondary);
static void	gsd_build_group_box		(GnomeScanDialog *dialog,
										   GtkWidget *page,
										   GtkBox *box,
										   GnomeScanPlugin *plugin,
										   GQuark group);

static void	gsd_show_hide_param_widget	(GParamSpec *param,
											  GnomeScanDialog *dialog);

/* INTERNALS */

static void	gsd_load_backends		(GnomeScanDialog *dialog);
static gboolean	gsd_select_scanner_if_ready	(GnomeScanDialog *gsd);
static void	gsd_scanner_added		(GnomeScanBackend *backend,
										 GnomeScanner *scanner,
										 GnomeScanDialog *dialog);
static void	gsd_scanner_removed		(GnomeScanBackend *backend,
										   GnomeScanner *scanner,
										   GnomeScanDialog *dialog);
static void	gsd_plugin_params_changed	(GnomeScanPlugin *plugin,
											 GParamSpec *pspec,
											 GnomeScanDialog* gsd);
static void	gsd_scanner_status_changed	(GnomeScanner *scanner,
											  GnomeScanDialog *gsd);
static void	gsd_probe_done			(GnomeScanBackend *backend,
										  GnomeScanDialog *dialog);

/* UI */

static void	gsd_build_general_ui		(GnomeScanDialog *dialog);
static void	gsd_build_preview_ui		(GnomeScanDialog *dialog);
static void	gsd_build_processing_ui	    (GnomeScanDialog *dialog);
static void	gsd_build_sink_ui		    (GnomeScanDialog *dialog);
static void	gsd_build_scanner_ui		(GnomeScanDialog *gsd);
static void	gsd_scanner_selected		(GtkTreeSelection *selection,
											GnomeScanDialog *dialog);
static void gsd_destroy_param			(GParamSpec *param,
										  GnomeScanDialog *dialog);

/* PREVIEW */
static void	gsd_preview_scanner_selected	(GnomeScanDialog *gsd);

GS_DEFINE_QUARK (gsd_page, "page");
#define	GSD_PAGE_QUARK	(gsd_page_quark())

GS_DEFINE_QUARK (gsd_group, "group");
#define	GSD_GROUP_QUARK	(gsd_group_quark())

GS_DEFINE_QUARK (gsd_table, "table");
#define	GSD_TABLE_QUARK	(gsd_table_quark())

GS_DEFINE_QUARK (gsd_widget, "widget");
#define	GSD_WIDGET_QUARK	(gsd_widget_quark())

GS_DEFINE_QUARK (gsd_plugin, "plugin");
#define	GSD_PLUGIN_QUARK	(gsd_plugin_quark())

GS_DEFINE_QUARK (gsd_label, "label");
#define	GSD_LABEL_QUARK	(gsd_label_quark())

G_DEFINE_TYPE (GnomeScanDialog, gnome_scan_dialog, GTK_TYPE_DIALOG);

static void
gnome_scan_dialog_init (GnomeScanDialog *object)
{
	GnomeScanDialogPrivate* priv = GET_PRIVATE (object);

	priv->preview_res = g_new0 (GValue, 1);
	g_value_init (priv->preview_res, G_TYPE_DOUBLE);
	g_value_set_double (priv->preview_res, PREVIEW_RES);
}

GObject*
gnome_scan_dialog_constructor (GType type, guint n, GObjectConstructParam *params)
{
	GnomeScanDialogPrivate* priv;
	GObject* object;
	GnomeScanDialog *dialog;
	GtkWidget *widget;
	GtkWidget *label;
	GSList *node;
	
	object =
		G_OBJECT_CLASS (gnome_scan_dialog_parent_class)->constructor (type, n,
																	  params);
	widget = GTK_WIDGET (object);
	dialog = GNOME_SCAN_DIALOG (widget);
	priv = GET_PRIVATE (dialog);
	
	while (gtk_events_pending ())
		gtk_main_iteration ();
	
	/* 	Dialog  */
	gtk_dialog_add_button (GTK_DIALOG (widget),
						   GTK_STOCK_CANCEL,
						   GTK_RESPONSE_CANCEL);
	
	gtk_dialog_add_button (GTK_DIALOG (widget),
						   GS_STOCK_SCAN,
						   GTK_RESPONSE_APPLY);
	
	/* 	Notebook  */
	priv->notebook = gtk_notebook_new ();
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (widget)->vbox), priv->notebook,
						TRUE, TRUE, 0);
	/* 	dialog.border_width + notebook.border_width = 6 + 6 = 12  */
	gtk_container_set_border_width (GTK_CONTAINER (priv->notebook), 6);
	
	/* 	General  */
	label = gtk_label_new_with_mnemonic (_("_General"));
	priv->general_page = gtk_vbox_new (FALSE, 6);
	gtk_container_set_border_width (GTK_CONTAINER (priv->general_page), 12);
	gtk_notebook_insert_page (GTK_NOTEBOOK (priv->notebook),
							  priv->general_page, label,
							  PAGE_GENERAL);
	
	/* 	Preview  */
	label = gtk_label_new_with_mnemonic (_("Pre_view"));
	priv->preview_page = gtk_vbox_new (FALSE, 6);
	gtk_container_set_border_width (GTK_CONTAINER (priv->preview_page), 12);
	gtk_widget_set_no_show_all (priv->preview_page, TRUE);
	gtk_notebook_insert_page (GTK_NOTEBOOK (priv->notebook),
							  priv->preview_page, label,
							  PAGE_PREVIEW);
	
	/* 	Advanced  */
	label = gtk_label_new_with_mnemonic (_("_Advanced"));
	priv->advanced_page = gtk_alignment_new (.5, .5, 1., 1.);
	gtk_notebook_insert_page (GTK_NOTEBOOK (priv->notebook),
							  priv->advanced_page, label,
							  PAGE_ADVANCED);
	gtk_widget_set_no_show_all (priv->advanced_page, TRUE);
	
	/* 	Processing  */
	label = gtk_label_new_with_mnemonic (_("P_rocessing"));
	priv->processing_box = gtk_vbox_new (FALSE, 6);
	gtk_container_set_border_width (GTK_CONTAINER (priv->processing_box), 6);
	priv->processing_page = priv->processing_box;
	gtk_notebook_insert_page (GTK_NOTEBOOK (priv->notebook),
							  priv->processing_page, label,
							  PAGE_PROCESSING);
	
	/* 	Sink  */
	label = gtk_label_new_with_mnemonic (_("_Output"));
	priv->sink_box = gtk_vbox_new (FALSE, 6);
	gtk_container_set_border_width (GTK_CONTAINER (priv->sink_box), 6);
	
	priv->sink_page = priv->sink_box;
	/*scrolled = gtk_scrolled_window_new (NULL, NULL);
	 gtk_widget_set_no_show_all (priv->sink_page, TRUE);
	 gtk_container_set_border_width (GTK_CONTAINER (scrolled), 12);
	 gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
									 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	 gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (scrolled),
											priv->sink_box);
	 */
	gtk_notebook_insert_page (GTK_NOTEBOOK (priv->notebook),
							  priv->sink_page, label,
							  PAGE_SINK);
	
	gtk_widget_show_all (priv->notebook);
	
	gsd_build_general_ui (dialog);
	gsd_build_preview_ui (dialog);
	gsd_build_processing_ui (dialog);
	gsd_build_sink_ui (dialog);
	
	/* configure preview job */
	GnomeScanSink*sink = gnome_scan_preview_sink_new();
	priv->preview_job = gnome_scan_job_new (gnome_scan_job_get_settings(priv->job),
											sink);
	g_object_unref(sink);
	for (node = gnome_scan_job_get_processors (priv->job)->next; node; node = node->next)
		gnome_scan_job_add_processor (priv->preview_job, node->data);
	
	
	return object;
}

static void
gnome_scan_dialog_dispose (GObject *object)
{
	GnomeScanDialogPrivate *priv = GET_PRIVATE (object);
	GSList *node;
	GnomeScanner *scanner;
	GnomeScanSink *sink;
	
	if (!priv->disposed) {
		/* freeing all scanner params */
		if (scanner = gnome_scan_job_get_scanner(priv->job)) {
			gnome_scan_plugin_params_foreach (GNOME_SCAN_PLUGIN(scanner),
											  (GFunc) gsd_destroy_param,
											  object);
		}
		
		/* unref all processors params */
		for (node = gnome_scan_job_get_processors (priv->job); node; node = node->next) {
			gnome_scan_plugin_params_foreach (GNOME_SCAN_PLUGIN(node->data),
											  (GFunc) gsd_destroy_param,
											  object);
		}
		
		/* unref all sink params */
		if (sink = gnome_scan_job_get_sink(priv->job)) {
			gnome_scan_plugin_params_foreach (GNOME_SCAN_PLUGIN(sink),
											  (GFunc) gsd_destroy_param,
											  object);
		}
		
		/* unref scanners and backends */
		g_object_unref (priv->scanners);
		for (node = priv->backends; node; node = node->next) {
			g_object_unref (node->data);
		}

		/* unref jobs */
		g_object_unref(priv->job);
		g_object_unref(priv->preview_job);
		
		priv->disposed = TRUE;
	}
	
	G_OBJECT_CLASS (parent_class)->dispose (object);
}

static void
gnome_scan_dialog_finalize (GObject *object)
{
	GnomeScanDialogPrivate* priv = GET_PRIVATE (object);
	
	g_value_unset (priv->preview_res);
	g_free (priv->preview_res);
	
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gnome_scan_dialog_set_property (GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec)
{
	g_return_if_fail (GNOME_IS_SCAN_DIALOG (object));
	
	switch (prop_id)
	{
		case PROP_JOB:
			GET_PRIVATE (object)->job = GNOME_SCAN_JOB (g_value_dup_object (value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
			break;
	}
}

static void
gnome_scan_dialog_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
	g_return_if_fail (GNOME_IS_SCAN_DIALOG (object));
	
	switch (prop_id)
	{
		case PROP_JOB:
			g_value_set_object (value, GET_PRIVATE (object)->job);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
			break;
	}
}

static void
gnome_scan_dialog_class_init (GnomeScanDialogClass *klass)
{
	GObjectClass* object_class = G_OBJECT_CLASS (klass);
	parent_class = GTK_DIALOG_CLASS (g_type_class_peek_parent (klass));
	
	g_type_class_add_private (klass, sizeof (GnomeScanDialogPrivate));
	
	object_class->constructor	= gnome_scan_dialog_constructor;
	object_class->finalize		= gnome_scan_dialog_finalize;
	object_class->dispose		= gnome_scan_dialog_dispose;
	object_class->set_property	= gnome_scan_dialog_set_property;
	object_class->get_property	= gnome_scan_dialog_get_property;
	
	
	/**
	 * GnomeScanDialog:job:
	 *
	 * The job the dialog is configuring.
	 **/
	g_object_class_install_property (object_class,
									 PROP_JOB,
									 g_param_spec_object ("job",
														  "Job",
														  "The job the dialog will configure",
														  GNOME_TYPE_SCAN_JOB,
														  G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
}




/**
 * gnome_scan_dialog_new:
 * @parent: The parent #GtkWindow
 * @job: the job to configure
 * 
 * Instanciate the new dialog and populate it considering job scanner,
 * processor and sink.
 * 
 * Returns: a new #GnomeScanDialog
 */
GtkWidget*
gnome_scan_dialog_new (GtkWindow *parent, GnomeScanJob *job)
{
	GObject* object = g_object_new (GNOME_TYPE_SCAN_DIALOG,
									"border-width", 6,
									"window-position", GTK_WIN_POS_CENTER,
									"default-width", 320,
									"default-height", 420,
									"title", _("Scan"),
									"has-separator", FALSE,
									"modal", TRUE,
									"icon-name", "scanner",
									"transient-for", parent,
									"job", job,
									NULL);
	
	return GTK_WIDGET (object);
}

/**
 * gnome_scan_dialog_run:
	 * @dialog: a #GnomeScanDialog
 * 
 * Run the dialog. If no backends are loaded, a popup is shown, and
 * the function returns. If no scanners has been detected, the dialog
 * wait for scanner plug through #GnomeScanBackend::scanner-added
 * signal, and present a warning popup to user asking him to ensure
 * device is plugged, etc.
 *
 * Once the dialog has been applied, a #GnomeScanAquisitionDialog is
 * run on the #GnomeScanDialog:job.
 *
 * See: #GnomeScanJob
 **/
void
gnome_scan_dialog_run (GnomeScanDialog *dialog)
{
	GnomeScanDialogPrivate* priv = GET_PRIVATE (dialog);
	GtkWidget* acquisition_dialog;
	GtkWindow* parent = NULL;
	gint response;
	
	gsd_load_backends (dialog);
	
	if (g_slist_length (priv->backends) == 0) {
		gsd_message_dialog (dialog, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
							_("Unable to detect scanners !"),
							_("No drivers has been found."));
		return;
	}
	
	if (priv->probe_done && !priv->scanner_count) {
		gsd_message_dialog (dialog, GTK_MESSAGE_WARNING, GTK_BUTTONS_OK,
							_("No device found !"),
							_("Ensure your device is plugged, powered, supported and configured. "
							  "The dialog will run until you plug a scanner or cancel."));
	}
	
	priv->run = TRUE;
	response = gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_hide (GTK_WIDGET (dialog));
	
	if (response == GTK_RESPONSE_APPLY) {
		g_object_get (dialog, "transient-for", &parent, NULL);
		acquisition_dialog = gnome_scan_acquisition_dialog_new (parent, priv->job);
		gnome_scan_acquisition_dialog_run (GNOME_SCAN_ACQUISITION_DIALOG (acquisition_dialog));
		gtk_widget_destroy(acquisition_dialog);
	}
	
	return;
}



/* INTERNAL */	
static void
gsd_message_dialog (GnomeScanDialog *dialog, 
					GtkMessageType mtype,
					GtkButtonsType btype,
					const gchar* primary,
					const gchar* secondary)
{
	GtkWidget* message_dialog;
	GtkWindow* parent = NULL;
	
	g_object_get (dialog, "transient-for", &parent, NULL);
	if (!(parent && GTK_WIDGET_VISIBLE (GTK_WIDGET (parent))) && GTK_WIDGET_VISIBLE (GTK_WIDGET (dialog))) {
		parent = GTK_WINDOW (dialog);
	}
	else {
		parent = NULL;
	}
	
	message_dialog = gtk_message_dialog_new (parent,
											 GTK_DIALOG_MODAL
											 | GTK_DIALOG_DESTROY_WITH_PARENT,
											 mtype, btype, g_strdup (primary));
	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (message_dialog),
											  g_strdup (secondary));
	
	gtk_dialog_run (GTK_DIALOG (message_dialog));
	gtk_widget_destroy (message_dialog);
}





static void
gsd_build_group_box (GnomeScanDialog *dialog, GtkWidget *page, GtkBox *box, GnomeScanPlugin *plugin, GQuark group)
{
	GnomeScanDialogPrivate *priv = GET_PRIVATE(dialog);
	GtkWidget *label, *group_box, *alignment, *table, *widget, *eventbox;
	GSList *node = NULL;
	const gchar *domain;
	gboolean expands, shows_label;
	gboolean box_expands = FALSE;
	gint count = 0;
	
	node = gnome_scan_plugin_get_param_group (plugin, group);
	if (!node)
		return;
	
	group_box = gtk_vbox_new (FALSE, 6);
	gtk_widget_set_no_show_all (group_box, FALSE);
	gsd_init_box_show (group_box);
	g_object_set_qdata (G_OBJECT (group_box), GSD_PAGE_QUARK, page);
	gtk_box_pack_start (box, GTK_WIDGET (group_box), FALSE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (group_box), 6);
	
	/* group label */
	label = gtk_label_new (NULL);
	domain = node ? gs_param_spec_get_domain (node->data) : GETTEXT_PACKAGE;
	domain = domain ? domain : GETTEXT_PACKAGE;
	gtk_label_set_markup (GTK_LABEL (label),
						  g_strdup_printf ("<b>%s</b>",
										   dgettext (domain,
													 g_quark_to_string (group))));
	gtk_misc_set_alignment (GTK_MISC (label), 0, .5);
	gtk_box_pack_start (GTK_BOX (group_box), label, FALSE, TRUE, 0);
	
	/* group's children are padded. */
	alignment = gtk_alignment_new (1., 0., 1., 1.);
	gtk_alignment_set_padding (GTK_ALIGNMENT (alignment), 0, 0, 24, 0);
	gtk_box_pack_start (GTK_BOX (group_box), alignment, TRUE, TRUE, 0);
	
	table = gtk_table_new (0, 2, FALSE);
	gtk_table_set_col_spacings (GTK_TABLE (table), 6);
	gtk_table_set_row_spacings (GTK_TABLE (table), 4);
	gtk_container_add (GTK_CONTAINER (alignment), table);
	g_object_set_qdata (G_OBJECT (group_box), GSD_TABLE_QUARK, table);
	
	for (; node ; node = node->next)
	{
		widget = gnome_scan_param_widget_new (gnome_scan_job_get_settings (priv->job),
											  plugin, node->data);
		
		if (!widget)
			continue;
		
		expands = gnome_scan_param_widget_expands (GNOME_SCAN_PARAM_WIDGET (widget));
		shows_label = gnome_scan_param_widget_shows_label (GNOME_SCAN_PARAM_WIDGET (widget));
		box_expands = box_expands || expands;
		
		gtk_widget_set_tooltip_text (widget,
									 dgettext (gs_param_spec_get_domain (node->data),
											   g_param_spec_get_blurb (node->data)));
		
		gtk_widget_set_no_show_all (widget, FALSE);
		gsd_add_to_box_show (group_box, 1);
		g_object_set_qdata (G_OBJECT (widget), GSD_GROUP_QUARK, group_box);
		g_param_spec_set_qdata (node->data, GSD_WIDGET_QUARK, widget);
		
		
		/* Colspan if widgets shows label itself */
		if (shows_label) {
			gtk_table_attach (GTK_TABLE (table), widget,
							  0, 2,
							  count, count+1,
							  GTK_EXPAND | GTK_FILL,
							  (expands ? GTK_EXPAND | GTK_FILL : GTK_FILL),
							  0, 0);
			label = NULL;
		}
		else {
			label = gtk_label_new (g_strdup_printf (_("%s:"),
													dgettext (gs_param_spec_get_domain (node->data),
															  g_param_spec_get_nick (node->data))));
			gtk_misc_set_alignment (GTK_MISC (label), 0., .5);
			
			gtk_widget_set_tooltip_text (label,
										 dgettext (gs_param_spec_get_domain (node->data),
												   g_param_spec_get_blurb (node->data)));
			gtk_table_attach (GTK_TABLE (table), label,
							  0, 1,
							  count, count+1,
							  GTK_FILL,
							  GTK_FILL,
							  0, 0);
			gtk_table_attach (GTK_TABLE (table), widget,
							  1, 2,
							  count, count+1,
							  GTK_EXPAND | GTK_FILL,
							  (expands ? GTK_EXPAND : GTK_FILL) | GTK_FILL,
							  0, 0);
		}
		g_object_set_qdata (G_OBJECT (widget), GSD_LABEL_QUARK, label);
		count++;
	}
	
	/* special handling of front page */
	box_expands = box_expands || group == GS_PARAM_GROUP_SCANNER_FRONT;
	
	/* update packing */
	if (GTK_IS_BOX (page)) {
		gtk_box_set_child_packing (GTK_BOX (page), GTK_WIDGET (box),
								   box_expands, TRUE, 0, GTK_PACK_START);
	}
	gtk_box_set_child_packing (GTK_BOX (box), GTK_WIDGET (group_box),
							   box_expands, TRUE, 0, GTK_PACK_START);
	gtk_box_set_child_packing (GTK_BOX (group_box), GTK_WIDGET (alignment),
							   box_expands, TRUE, 0, GTK_PACK_START);
	
	gsd_add_to_box_show (page, 1);
	
	gtk_widget_set_no_show_all (page, FALSE);
	gtk_widget_show_all (page);
	gtk_widget_realize (page);
}

static void
gsd_show_hide_param_widget (GParamSpec *param, GnomeScanDialog *dialog)
{
	gboolean show = param->flags & G_PARAM_WRITABLE;
	GtkWidget *widget = g_param_spec_get_qdata (param, GSD_WIDGET_QUARK);
	
	if (!widget)
		return;
	
	gboolean shown = !gtk_widget_get_no_show_all (widget);
	
	if ((show && shown) || show ==  shown)
		return;
	
	gint inc = show ? 1 : -1;
	
	GtkWidget *label = g_object_get_qdata (G_OBJECT (widget), GSD_LABEL_QUARK);
	GtkWidget *group = g_object_get_qdata (G_OBJECT (widget), GSD_GROUP_QUARK);
	GtkWidget *page = g_object_get_qdata (G_OBJECT (group), GSD_PAGE_QUARK);
	
	GTK_WIDGET_SET_VISIBLE (widget, show);
	GTK_WIDGET_SET_VISIBLE (label, show);
	
	gsd_add_to_box_show (group, inc);
	guint count = gsd_get_box_show (group);
	GTK_WIDGET_SET_VISIBLE (group, count);
	
	if (count == 0) {
		inc = -1;
	}
	else if (count == 1) {
		inc = 1;
	}
	else {
		inc = 0;
	}
	gsd_add_to_box_show (page, inc);
	count = gsd_get_box_show (page);
	GTK_WIDGET_SET_VISIBLE (page, count);
}

static void
gsd_destroy_param (GParamSpec *param, GnomeScanDialog *dialog)
{	
	GtkWidget *widget = g_param_spec_get_qdata (param, GSD_WIDGET_QUARK);
	if (!widget) {
		GnomeScanPreviewPlugin *plugin = g_param_spec_get_qdata (param, GSD_PLUGIN_QUARK);
		if (plugin) {
			gnome_scan_preview_plugin_destroy (plugin);
		}
		return;
	}
	
	gboolean shown = !gtk_widget_get_no_show_all (widget);
	GtkWidget *label	= g_object_get_qdata (G_OBJECT (widget),	GSD_LABEL_QUARK);
	GtkWidget *group	= g_object_get_qdata (G_OBJECT (widget),	GSD_GROUP_QUARK);
	GtkWidget *table	= g_object_get_qdata (G_OBJECT (group),		GSD_TABLE_QUARK);
	GtkWidget *page		= g_object_get_qdata (G_OBJECT (group),		GSD_PAGE_QUARK);
	
	/* first destroy param specific widget : label and param widget */
	if (label)
		gtk_widget_destroy (label);
	
	gtk_widget_destroy (widget);
	g_param_spec_set_qdata (param, GSD_WIDGET_QUARK, NULL);
	
	/* decrement group box_show count */
	if (shown) {
		gsd_add_to_box_show (group, -1);
	}
	
	/* decrement page box_show count if now more widget are shown */
	guint count = gsd_get_box_show (group);
	if (count == 0 && shown) {
		gsd_add_to_box_show (page, -1);
	}
	
	/* hide page if no more groups to show */
	count = gsd_get_box_show (page);
	GTK_WIDGET_SET_VISIBLE (page, (count > 0));
	
	/* if group has no more children, destroy it */
	count = g_list_length (GTK_TABLE (table)->children);
	if (count == 0) {
		gtk_widget_destroy (group);
	}
}






/* INTERNALS */

static void
gsd_load_backends (GnomeScanDialog *dialog)
{
	GnomeScanDialogPrivate *priv = GET_PRIVATE (dialog);
	GnomeScanBackend *backend;
	GThread *thread;
	GError *error = NULL;
	GType *backend_types;
	GdkCursor *watch_cursor = gdk_cursor_new(GDK_WATCH);
	gint i;
	guint n;
	backend_types = g_type_children (GNOME_TYPE_SCAN_BACKEND, &n);
	priv->probing_backends = 0;
	priv->probe_done = FALSE;
	
	for (i = 0 ; i < n ; i++) {
		backend = gnome_scan_backend_new (backend_types[i]);
		priv->backends = g_slist_append (priv->backends, backend);
		
		if (priv->backends) {
			g_signal_connect (backend, "scanner-added",
							  (GCallback) gsd_scanner_added, dialog);
			g_signal_connect (backend, "scanner-removed",
							  (GCallback) gsd_scanner_removed, dialog);
			g_signal_connect (backend, "probe-done",
							  (GCallback) gsd_probe_done, dialog);
			
			priv->probing_backends++;
			
			gdk_window_set_cursor(GTK_WIDGET(dialog)->window,
								  watch_cursor);
			thread = g_thread_create ((GThreadFunc) gnome_scan_backend_probe_scanners,
									  backend, FALSE, &error);
		}
	}
	gdk_cursor_unref (watch_cursor);
}

static void
gsd_scanner_added (GnomeScanBackend *backend, GnomeScanner *scanner, GnomeScanDialog *dialog)
{
	GnomeScanDialogPrivate *priv = GET_PRIVATE (dialog);
	GtkTreeIter* iter = g_new0 (GtkTreeIter, 1);
	
	gtk_list_store_insert_with_values (priv->scanners, iter, G_MAXINT,
									   COLUMN_ICON, gnome_scanner_get_icon_name (scanner),
									   COLUMN_NAME, gnome_scan_plugin_get_name (GNOME_SCAN_PLUGIN (scanner)),
									   COLUMN_STATUS, gnome_scanner_get_status_string (GNOME_SCANNER (scanner)),
									   COLUMN_OBJECT, scanner,
									   -1);
	g_object_set_data (G_OBJECT (scanner), "iter", iter);
	g_signal_connect (scanner, "status-changed",
					  (GCallback) gsd_scanner_status_changed,
					  dialog);
	
	priv->scanner_count++;
}

static void
gsd_scanner_removed (GnomeScanBackend *backend, GnomeScanner *scanner, GnomeScanDialog *dialog)
{
	GnomeScanDialogPrivate *priv = GET_PRIVATE (dialog);
	GtkTreeIter *iter;
	iter = g_object_steal_data (G_OBJECT (scanner), "iter");
	gtk_list_store_remove (priv->scanners, iter);
	g_free (iter);
	priv->scanner_count--;
}


static void
gsd_scanner_status_changed (GnomeScanner *scanner, GnomeScanDialog *gsd)
{
	GtkTreeIter *iter;
	iter = g_object_get_data (G_OBJECT (scanner), "iter");
	gtk_list_store_set (GET_PRIVATE (gsd)->scanners, iter,
						COLUMN_STATUS, gnome_scanner_get_status_string (scanner),
						-1);
}

static void
gsd_plugin_params_changed (GnomeScanPlugin *plugin, GParamSpec *pspec, GnomeScanDialog* gsd)
{
	if (pspec) {
		gsd_show_hide_param_widget (pspec, gsd);
	}
}

static void
gsd_probe_done (GnomeScanBackend *backend, GnomeScanDialog *dialog)
{
	GnomeScanDialogPrivate *priv = GET_PRIVATE (dialog);
	GdkCursor *arrow_cursor;
	
	priv->probe_done = !--priv->probing_backends;
	if (!priv->scanner_count && GTK_WIDGET_VISIBLE (GTK_WIDGET (dialog))) {
		gsd_message_dialog (dialog, GTK_MESSAGE_WARNING, GTK_BUTTONS_OK,
							_("No device found !"),
							_("Ensure your device is plugged, powered, supported and configured."));
	}

	if (priv->probe_done) {
		arrow_cursor = gdk_cursor_new(GDK_ARROW);
		gdk_window_set_cursor (GTK_WIDGET(dialog)->window,
							   arrow_cursor);
		gdk_cursor_unref (arrow_cursor);
	}
}

/* GENERAL */
static void
gsd_build_general_ui (GnomeScanDialog *dialog)
{
	GnomeScanDialogPrivate *priv = GET_PRIVATE (dialog);
	GtkWidget *tree_view, *scrolled;
	GtkTreeViewColumn *column;
	GtkCellRenderer *renderer;
	
	priv->scanners = gtk_list_store_new (N_COLUMNS,
										 G_TYPE_STRING,
										 G_TYPE_STRING,
										 G_TYPE_STRING,
										 GNOME_TYPE_SCANNER);
	
	/* Tree View */
	tree_view = gtk_tree_view_new_with_model (GTK_TREE_MODEL (priv->scanners));
	g_object_set (tree_view, "search-column", COLUMN_NAME, NULL);
	
	/* Icon */
	renderer = gtk_cell_renderer_pixbuf_new ();
	g_object_set (renderer, "stock-size", GTK_ICON_SIZE_SMALL_TOOLBAR, NULL);
	column = gtk_tree_view_column_new_with_attributes (NULL, renderer,
													   "icon-name", COLUMN_ICON,
													   NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view), column);
	
	/* Name */
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Scanner"), renderer,
													   "text", COLUMN_NAME,
													   NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view), column);
	
	/* Status */
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Status"), renderer,
													   "text", COLUMN_STATUS,
													   NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view), column);
	
	
	priv->scanner_selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree_view));
	g_signal_connect (priv->scanner_selection, "changed",
					  (GCallback) gsd_scanner_selected, dialog);
	
	/* Scrolled Window */
	scrolled = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
									GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (scrolled),
										   tree_view);
	gtk_box_pack_start (GTK_BOX (priv->general_page), scrolled, TRUE, TRUE, 0);
	gtk_widget_set_size_request (scrolled, -1, 128);
	
	/* Scanner front box */
	priv->front_scanner_box = gtk_hbox_new (FALSE, 6);
	gtk_box_pack_start (GTK_BOX (priv->general_page), priv->front_scanner_box,
						FALSE, TRUE, 0);
	gsd_init_box_show (priv->front_scanner_box);
	
	/* Sink front box */
	priv->front_sink_box = gtk_vbox_new (FALSE, 6);
	gtk_box_pack_start (GTK_BOX (priv->general_page), priv->front_sink_box,
						FALSE, TRUE, 0);
	gsd_init_box_show (priv->front_sink_box);
	
	gtk_widget_show_all (priv->general_page);
	
}

static void
gsd_scanner_selected (GtkTreeSelection *selection, GnomeScanDialog *dialog)
{
	GnomeScanDialogPrivate *priv = GET_PRIVATE (dialog);
	GnomeScanner *scanner;
	GtkTreeIter iter;
	GtkTreeModel *model = GTK_TREE_MODEL (priv->scanners);
	
	scanner = gnome_scan_job_get_scanner (priv->job);
	
	if (priv->scanner_changed_handler) {
		g_signal_handler_disconnect (scanner, priv->scanner_changed_handler);
		
		gnome_scan_plugin_params_foreach (GNOME_SCAN_PLUGIN (scanner),
										  (GFunc) gsd_destroy_param,
										  dialog);
	}
	
	gtk_tree_selection_get_selected (selection,
									 &model,
									 &iter);
	gtk_tree_model_get (model, &iter,
						COLUMN_OBJECT, &scanner,
						-1);
	
	gnome_scan_job_set_scanner (priv->job, scanner);
	gnome_scan_job_set_scanner (priv->preview_job, scanner);
	
	g_timeout_add (10, (GSourceFunc) gsd_select_scanner_if_ready, dialog);
}



static gboolean
gsd_select_scanner_if_ready (GnomeScanDialog *gsd)
{
	GnomeScanDialogPrivate *priv = GET_PRIVATE (gsd);
	GnomeScanner *scanner = gnome_scan_job_get_scanner (priv->job);
	if (gnome_scanner_get_status (scanner) != GNOME_SCANNER_READY) {
		return TRUE;
	}
	else {
		priv->scanner_changed_handler =
			g_signal_connect (scanner, "params-changed",
							  (GCallback) gsd_plugin_params_changed,
							  gsd);
		
		
		gsd_build_scanner_ui (gsd);
		return FALSE;
	}
}

static void
gsd_build_sink_ui (GnomeScanDialog *gsd)
{
	GnomeScanDialogPrivate *priv = GET_PRIVATE (gsd);
	GnomeScanSink *sink = gnome_scan_job_get_sink (priv->job);
	GSList *node, *groups = NULL;
	
	gsd_build_group_box (gsd, priv->general_page,
						 GTK_BOX (priv->front_sink_box),
						 GNOME_SCAN_PLUGIN (sink),
						 GS_PARAM_GROUP_SINK_FRONT);
	
	groups = gnome_scan_plugin_params_get_other_groups (GNOME_SCAN_PLUGIN (sink),
														GS_PARAM_GROUP_SINK_FRONT,
														GS_PARAM_GROUP_HIDDEN,
														0);
	for (node = groups; node ; node = node->next) {
		gsd_build_group_box (gsd, priv->sink_page,
							 GTK_BOX (priv->sink_box),
							 GNOME_SCAN_PLUGIN (sink),
							 (GQuark) node->data);
	}
	
	gnome_scan_plugin_params_foreach (GNOME_SCAN_PLUGIN (sink),
									  (GFunc) gsd_show_hide_param_widget,
									  gsd);
	g_signal_connect (sink, "params-changed",
					  (GCallback) gsd_plugin_params_changed,
					  gsd);
	
}


static void
gsd_build_scanner_ui	(GnomeScanDialog *gsd)
{
	GnomeScanDialogPrivate *priv = GET_PRIVATE (gsd);
	GtkWidget *widget;
	GSList *node, *groups = NULL;
	GList *params;
	GnomeScanner *scanner = gnome_scan_job_get_scanner (priv->job);
	
	/* front box */
	gsd_build_group_box (gsd, priv->front_scanner_box,
						 GTK_BOX (priv->front_scanner_box),
						 GNOME_SCAN_PLUGIN (scanner),
						 GS_PARAM_GROUP_FORMAT);
	
	gsd_build_group_box (gsd, priv->front_scanner_box,
						 GTK_BOX (priv->front_scanner_box),
						 GNOME_SCAN_PLUGIN (scanner),
						 GS_PARAM_GROUP_SCANNER_FRONT);
	/* preview */
	if (node = gnome_scan_plugin_get_param_group (GNOME_SCAN_PLUGIN (scanner), GS_PARAM_GROUP_PREVIEW)) {
		gsd_preview_scanner_selected (gsd);
		GTK_WIDGET_SET_VISIBLE (priv->preview_page, TRUE);
	}
	else {
		GTK_WIDGET_SET_VISIBLE (priv->preview_page, FALSE);
	}
	
	/* advanced */
	
	/* That's ugly to use a scrolled window while the dialog has enough place for
	 e.g. 3 options. So we use a scrolled window only beginning with an arbitrary
		 amount of options. Otherwise, we pack directly the box in the notebook.
		 TODO: count only advanced options. */
	
	params = gnome_scan_plugin_get_params (GNOME_SCAN_PLUGIN (scanner));
	
	/* clean advanced tab */
	widget = gtk_bin_get_child (GTK_BIN (priv->advanced_page));
	if (widget)
		gtk_widget_destroy (widget);
	
	/* create the containing box */
	priv->advanced_box = gtk_vbox_new (FALSE, 6);
	gtk_container_set_border_width (GTK_CONTAINER (priv->advanced_box), 6);
	
	/* determine which widget is the root of the page. 12 is completely arbitrary,
	 waiting for "dialog is too big" bug report */
	if (g_list_length (params) > 12) {
		priv->advanced_container = gtk_scrolled_window_new (NULL, NULL);
		gtk_container_set_border_width (GTK_CONTAINER (priv->advanced_container), 12);
		gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (priv->advanced_container),
										GTK_POLICY_AUTOMATIC,
										GTK_POLICY_AUTOMATIC);
		gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (priv->advanced_container),
											   priv->advanced_box);
	}
	else {
		priv->advanced_container = priv->advanced_box;
	}
	
	gtk_container_add (GTK_CONTAINER (priv->advanced_page),
					   priv->advanced_container);
	
	gtk_widget_realize (priv->advanced_page);
	
	/* get unknown groups, trim manually handled groups */
	groups = gnome_scan_plugin_params_get_other_groups (GNOME_SCAN_PLUGIN (scanner),
														GS_PARAM_GROUP_SCANNER_FRONT,
														GS_PARAM_GROUP_PREVIEW,
														GS_PARAM_GROUP_FORMAT,
														GS_PARAM_GROUP_HIDDEN,
														0);
	
	/* build each group in the advanced page */
	for (node = groups; node ; node = node->next) {
		gsd_build_group_box (gsd, priv->advanced_page,
							 GTK_BOX (priv->advanced_box),
							 GNOME_SCAN_PLUGIN (scanner),
							 (GQuark) node->data);
	}
	
	gnome_scan_plugin_params_foreach (GNOME_SCAN_PLUGIN (scanner),
									  (GFunc) gsd_show_hide_param_widget,
									  gsd);
}


static void
gsd_update_scanner_ui	(GnomeScanDialog *gsd)
{
	GnomeScanDialogPrivate *priv = GET_PRIVATE (gsd);
	GnomeScanner *scanner = gnome_scan_job_get_scanner (priv->job);
	
	gnome_scan_plugin_params_foreach (GNOME_SCAN_PLUGIN (scanner),
									  (GFunc) gsd_show_hide_param_widget,
									  gsd);
}

/* PROCESSING */

static void
gsd_build_processing_ui(GnomeScanDialog *gsd)
{
	GnomeScanDialogPrivate *priv = GET_PRIVATE (gsd);
	GSList *proc;
	proc = gnome_scan_job_get_processors (priv->job);
	GSList *node, *groups = NULL;
	
	/* loop each processor and add options */
	for (; proc; proc = proc->next) {
		groups = gnome_scan_plugin_params_get_other_groups (GNOME_SCAN_PLUGIN (proc->data),
															GS_PARAM_GROUP_PREVIEW,
															GS_PARAM_GROUP_HIDDEN,
															0);
		for (node = groups; node ; node = node->next) {
			gsd_build_group_box (gsd, priv->processing_page,
								 GTK_BOX (priv->processing_box),
								 GNOME_SCAN_PLUGIN (proc->data),
								 (GQuark) node->data);
		}
		gnome_scan_plugin_params_foreach (GNOME_SCAN_PLUGIN (proc->data),
										  (GFunc) gsd_show_hide_param_widget,
										  gsd);
		g_signal_connect (GNOME_SCAN_PLUGIN(proc->data), "params-changed",
						  (GCallback) gsd_plugin_params_changed,
						  gsd);
	}
}

/* PREVIEW */

static void
gsd_preview_scanner_selected (GnomeScanDialog *gsd)
{
	GnomeScanDialogPrivate *priv = GET_PRIVATE (gsd);
	GtkWidget*widget = GTK_WIDGET (gsd);
	GParamSpec *pspec;
	GSParamSpecPaperSize *psps;
	GtkPaperSize *ps;
	GnomeScanner *scanner;
	GdkPixbuf *pixbuf;
	GnomeScanPreviewPlugin *plugin;
	GSList *node = NULL, *node0 = NULL;
	GType type;
	gboolean first =  TRUE;
	
	/* add buttons */
	scanner = gnome_scan_job_get_scanner (priv->job);
	node0 = gnome_scan_plugin_get_param_group (GNOME_SCAN_PLUGIN (scanner),
											   GS_PARAM_GROUP_PREVIEW);
	
	for (node = gnome_scan_job_get_processors (priv->job); node; node = node->next) {
		node0 = g_slist_concat(node0,
							   gnome_scan_plugin_get_param_group (GNOME_SCAN_PLUGIN(node->data),
																  GS_PARAM_GROUP_PREVIEW));
	}
	
	for (node = node0; node ; node = node->next) {
		pspec = node->data;
		type = gs_param_spec_get_widget_type (pspec);
		g_debug("new preview plugin : %s", g_type_name(type));
		plugin = gnome_scan_preview_plugin (type,
											GNOME_SCAN_PLUGIN (scanner),
											pspec,
											priv->preview_area,
											gnome_scan_job_get_settings (priv->job),
											GTK_BOX (priv->preview_bbox));
		g_param_spec_set_qdata (pspec, GSD_PLUGIN_QUARK, plugin);
		
		if (first)
			gnome_scan_preview_area_select_plugin (GNOME_SCAN_PREVIEW_AREA (priv->preview_area),
												   plugin);
		first = FALSE;
	}
	
	/* create empty preview */
	pspec = gnome_scan_plugin_params_lookup (GNOME_SCAN_PLUGIN (scanner),
											 "paper-size");
	psps = GS_PARAM_SPEC_PAPER_SIZE (pspec);
	ps = psps->enumeration->next->data;
	gint width = (gint) gs_convert (gtk_paper_size_get_width (ps, GTK_UNIT_MM), GS_UNIT_MM, GS_UNIT_PIXEL, PREVIEW_RES);
	gint height = (gint) gs_convert (gtk_paper_size_get_height (ps, GTK_UNIT_MM), GS_UNIT_MM, GS_UNIT_PIXEL, PREVIEW_RES);
	
	pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB,
							 FALSE, 8,
							 width,
							 height);
	/* transform a 24bit RGB pixel to 32bit RGBApixel */
	guint32 pixel = (widget->style->bg[GTK_WIDGET_STATE (widget)].pixel << 8) | 0x000000FF;
	gdk_pixbuf_fill (pixbuf, pixel);
	/* TODO: put a nice icon, back in gnome-scan 0.4 ;) */
	gnome_scan_preview_area_set_pixbuf (GNOME_SCAN_PREVIEW_AREA (priv->preview_area),
										pixbuf,
										PREVIEW_RES);
	g_object_unref (G_OBJECT (pixbuf));
}

static gboolean
gsd_preview_end_refresh (GnomeScanDialog *gsd)
{
	GnomeScanDialogPrivate *priv = GET_PRIVATE (gsd);
	GnomeScanPreviewSink *sink = GNOME_SCAN_PREVIEW_SINK (gnome_scan_job_get_sink (priv->preview_job));
	GnomeScanSettings *settings = gnome_scan_job_get_settings (priv->preview_job);
	
	/* first, restore settings */
	gnome_scan_settings_set_boolean (settings, "preview", FALSE);
	gnome_scan_settings_set (settings, "resolution", priv->saved_res);
	
	/* restore UI */
	GTK_WIDGET_SET_VISIBLE (priv->preview_acquisition_box, FALSE);
	GTK_WIDGET_SET_VISIBLE (priv->preview_box, TRUE);
	
	/* then, update preview. */
	GdkPixbuf *pixbuf = gnome_scan_preview_sink_get_pixbuf (sink);
	if (!GDK_IS_PIXBUF (pixbuf)) {
		g_warning (G_STRLOC ": preview failed");
		return FALSE;
	}

	gnome_scan_preview_area_set_pixbuf (GNOME_SCAN_PREVIEW_AREA (priv->preview_area),
										pixbuf,
										PREVIEW_RES);
	return FALSE;
}

static gboolean
gsd_preview_monitor (GnomeScanDialog *gsd)
{
	GnomeScanDialogPrivate *priv = GET_PRIVATE (gsd);
	gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (priv->preview_progress),
								   priv->preview_job->progress);
	gtk_label_set_markup (GTK_LABEL (priv->preview_stage),
						  g_strdup_printf ("<i>%s</i>", priv->preview_job->stage));
	
	if (priv->preview_job->progress == 1.)
		g_idle_add ((GSourceFunc) gsd_preview_end_refresh, gsd);
	
	return (priv->preview_job->progress < 1.);
}

static void
gsd_preview_refresh (GtkButton *button, GnomeScanDialog *gsd)
{
	static GdkPoint origin = {0, 0};
	GnomeScanDialogPrivate *priv = GET_PRIVATE (gsd);
	GError *error = NULL;
	GValue *value;
	GParamSpec *pspec;
	GnomeScanSettings *settings = gnome_scan_job_get_settings (priv->preview_job);
	GnomeScanner *scanner = gnome_scan_job_get_scanner (priv->preview_job);
	
	GTK_WIDGET_SET_VISIBLE(priv->preview_acquisition_box, TRUE);
	GTK_WIDGET_SET_VISIBLE(priv->preview_box, FALSE);
	
	/* CONFIGURE */

	/* USE hard coded RESOLUTION */
	priv->saved_res = gnome_scan_settings_get (settings, "resolution");
	/* transform to int or double */
	pspec = gnome_scan_plugin_params_lookup (GNOME_SCAN_PLUGIN (scanner), "resolution");
	value = g_new0 (GValue, 1);
	g_value_init (value, G_PARAM_SPEC_VALUE_TYPE (pspec));
	g_param_value_set_default (pspec, value);
	g_value_transform (priv->preview_res, value);
	gnome_scan_settings_set (settings, "resolution", value);
	g_value_unset (value);
	g_free (value);
	/* asking preview mode */
	gnome_scan_settings_set_boolean (settings, "preview", TRUE);
	
	gnome_scan_job_configure (priv->preview_job);
	
	/* TRIGGER */
	g_timeout_add (42, (GSourceFunc) gsd_preview_monitor, gsd);
	g_thread_create ((GThreadFunc) gnome_scan_job_run_once,
					 priv->preview_job, FALSE, &error);
}



static void
gsd_preview_cancel_refresh (GtkButton *button, GnomeScanDialog *gsd)
{
	GnomeScanDialogPrivate *priv = GET_PRIVATE (gsd);
	
	gnome_scan_job_cancel (priv->preview_job);
	GTK_WIDGET_SET_VISIBLE (priv->preview_acquisition_box, FALSE);
	GTK_WIDGET_SET_VISIBLE (priv->preview_box, TRUE);
}


static void
gsd_build_preview_ui (GnomeScanDialog *gsd)
{
	GnomeScanDialogPrivate *priv = GET_PRIVATE (gsd);
	GtkWidget *box, *hbox, *vbox, *alignment, *child, *scrolled;
	
	/* acquisition "dialog" */
	priv->preview_acquisition_box = alignment = gtk_alignment_new (.5, .5, 0., 0.);
	gtk_box_pack_start (GTK_BOX (priv->preview_page),  alignment, TRUE, FALSE, 0);
	
	box = gtk_vbox_new (FALSE, 6);
	gtk_container_add (GTK_CONTAINER (alignment), box);
	
	hbox = gtk_hbox_new (FALSE, 6);
	gtk_box_pack_start (GTK_BOX (box), hbox, FALSE, FALSE, 0);
	
	child = gtk_image_new_from_icon_name ("scan-preview", GTK_ICON_SIZE_DIALOG);
	gtk_box_pack_start (GTK_BOX (hbox), child, FALSE, FALSE, 0);
	
	vbox = gtk_vbox_new (FALSE, 6);
	gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, FALSE, 0);
	
	child = gtk_label_new (NULL);
	gtk_misc_set_alignment (GTK_MISC (child), 0., .5);
	gtk_label_set_markup (GTK_LABEL (child),
						  g_strconcat ("<big><b>",
									   _("Acquiring Preview"),
									   "</b></big>",
									   NULL));
	gtk_box_pack_start (GTK_BOX (vbox), child, FALSE, FALSE, 0);
	
	child = gtk_label_new (_("The software preview acquisition and processing."));
	gtk_misc_set_alignment (GTK_MISC (child), 0., .5);
	gtk_box_pack_start (GTK_BOX (vbox), child, FALSE, FALSE, 0);
	
	priv->preview_progress = gtk_progress_bar_new ();
	gtk_box_pack_start (GTK_BOX (box), priv->preview_progress, FALSE, FALSE, 0);
	
	priv->preview_stage = gtk_label_new (NULL);
	gtk_misc_set_alignment (GTK_MISC (priv->preview_stage), 0., .5);
	gtk_label_set_markup (GTK_LABEL (priv->preview_stage),
						  g_strconcat("<i>", _("Inactive"), "</i>", NULL));
	gtk_box_pack_start (GTK_BOX (box), priv->preview_stage, FALSE, FALSE, 0);
	
	hbox = gtk_hbutton_box_new ();
	gtk_box_pack_start (GTK_BOX (box), hbox, FALSE, TRUE, 0);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (hbox),
							   GTK_BUTTONBOX_END);
	
	child = gtk_button_new_from_stock (GTK_STOCK_CANCEL);
	g_signal_connect (child, "clicked",
					  (GCallback) gsd_preview_cancel_refresh,
					  gsd);
	gtk_container_add (GTK_CONTAINER (hbox), child);
	
	
	/* preview dialog */
	priv->preview_box = box = gtk_hbox_new (FALSE, 6);
	gtk_box_pack_start (GTK_BOX (priv->preview_page), box, TRUE, TRUE, 0);
	
	scrolled = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
									GTK_POLICY_AUTOMATIC, 
									GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start (GTK_BOX (box), scrolled, TRUE, TRUE, 0);
	
	priv->preview_area = gnome_scan_preview_area_new ();
	gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (scrolled),
										   priv->preview_area);
	
	priv->preview_bbox = vbox = gtk_vbutton_box_new ();
	gtk_box_set_spacing (GTK_BOX (vbox), 4);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (vbox),
							   GTK_BUTTONBOX_START);
	gtk_box_pack_start (GTK_BOX (box), vbox, FALSE, TRUE, 0);
	
	child = gtk_button_new_from_stock (GTK_STOCK_REFRESH);
	gtk_container_add (GTK_CONTAINER (vbox), child);
	g_signal_connect (child, "clicked",
					  (GCallback) gsd_preview_refresh,
					  gsd);
	
	GTK_WIDGET_SET_VISIBLE (priv->preview_acquisition_box, FALSE);
	GTK_WIDGET_SET_VISIBLE (priv->preview_box, TRUE);
}
		
